""" Modified on top of examples/text_to_image/train_text_to_image_lora.py from diffusers """

#!/usr/bin/env python
# coding=utf-8
# Copyright 2024 The HuggingFace Inc. team. All rights reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#


#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""Fine-tuning script for Stable Diffusion for text2image with support for LoRA."""

import argparse
import logging
import math
import os
import random
import shutil
from contextlib import nullcontext
from pathlib import Path

import datasets
import numpy as np
import torch
import torch.nn.functional as F
import torch.utils.checkpoint
import transformers
from accelerate import Accelerator
from accelerate.logging import get_logger
from accelerate.utils import ProjectConfiguration, set_seed
from datasets import load_dataset
from huggingface_hub import create_repo, upload_folder
from packaging import version
from peft import LoraConfig
from peft.utils import get_peft_model_state_dict
from torchvision import transforms
from tqdm.auto import tqdm
from transformers import CLIPTextModel, CLIPTokenizer

import diffusers
from diffusers import AutoencoderKL, DDPMScheduler, DiffusionPipeline, StableDiffusionPipeline, UNet2DConditionModel, EulerDiscreteScheduler, DDIMScheduler
from diffusers.optimization import get_scheduler
from diffusers.training_utils import cast_training_params, compute_snr
from diffusers.utils import check_min_version, convert_state_dict_to_diffusers, is_wandb_available
from diffusers.utils.hub_utils import load_or_create_model_card, populate_model_card
from diffusers.utils.import_utils import is_xformers_available
from diffusers.utils.torch_utils import is_compiled_module
from datetime import datetime


# Will error if the minimal version of diffusers is not installed. Remove at your own risks.
check_min_version("0.28.0.dev0")

logger = get_logger(__name__, log_level="INFO")



# the CLIP-score metric between image and text
from torchmetrics.functional.multimodal.clip_score import _get_clip_model_and_processor, _clip_score_update

clip_model, clip_processor = _get_clip_model_and_processor("openai/clip-vit-base-patch32")


def _clip_score_metric(images, text):
    device = images.device if isinstance(images, torch.Tensor) else images[0].device
    scores, _ = _clip_score_update(images, text, clip_model.to(device), clip_processor)
    return torch.maximum(scores, torch.zeros_like(scores))


@torch.no_grad()
def clip_score_metric(imgs, texts, cpu=True):
    # imgs: [B, 3, H, W] in [0, 1]
    # texts: List[str]
    if cpu:
        imgs = imgs.cpu()
    scores = _clip_score_metric(imgs, texts)
    return scores


def parse_args():
    parser = argparse.ArgumentParser(description="Simple example of a training script.")
    parser.add_argument(
        "--pretrained_model_name_or_path",
        type=str,
        default="stabilityai/stable-diffusion-2-1-base",
        required=False,
        help="Path to pretrained model or model identifier from huggingface.co/models.",
    )
    parser.add_argument(
        "--revision",
        type=str,
        default=None,
        required=False,
        help="Revision of pretrained model identifier from huggingface.co/models.",
    )
    parser.add_argument(
        "--variant",
        type=str,
        default=None,
        help="Variant of the model files of the pretrained model identifier from huggingface.co/models, 'e.g.' fp16",
    )
    parser.add_argument(
        "--dataset_name",
        type=str,
        default=None,
        help=(
            "The name of the Dataset (from the HuggingFace hub) to train on (could be your own, possibly private,"
            " dataset). It can also be a path pointing to a local copy of a dataset in your filesystem,"
            " or to a folder containing files that 🤗 Datasets can understand."
        ),
    )
    parser.add_argument(
        "--dataset_config_name",
        type=str,
        default=None,
        help="The config of the Dataset, leave as None if there's only one config.",
    )
    parser.add_argument(
        "--train_data_dir",
        type=str,
        default=None,
        help=(
            "A folder containing the training data. Folder contents must follow the structure described in"
            " https://huggingface.co/docs/datasets/image_dataset#imagefolder. In particular, a `metadata.jsonl` file"
            " must exist to provide the captions for the images. Ignored if `dataset_name` is specified."
        ),
    )
    parser.add_argument(
        "--image_column", type=str, default="image", help="The column of the dataset containing an image."
    )
    parser.add_argument(
        "--caption_column",
        type=str,
        default="text",
        help="The column of the dataset containing a caption or a list of captions.",
    )
    parser.add_argument(
        "--validation_prompt", type=str, default=None, help="A prompt that is sampled during training for inference."
    )
    parser.add_argument(
        "--num_validation_images",
        type=int,
        default=4,
        help="Number of images that should be generated during validation with `validation_prompt`.",
    )
    parser.add_argument(
        "--validation_iters",
        type=int,
        default=500,
        help="Run validation every X iterations",
    )
    parser.add_argument(
        "--max_train_samples",
        type=int,
        default=None,
        help=(
            "For debugging purposes or quicker training, truncate the number of training examples to this "
            "value if set."
        ),
    )
    parser.add_argument(
        "--output_dir",
        type=str,
        default="sd-model-finetuned-lora",
        help="The output directory where the model predictions and checkpoints will be written.",
    )
    parser.add_argument(
        "--cache_dir",
        type=str,
        default=None,
        help="The directory where the downloaded models and datasets will be stored.",
    )
    parser.add_argument("--seed", type=int, default=None, help="A seed for reproducible training.")
    parser.add_argument(
        "--resolution",
        type=int,
        default=512,
        help=(
            "The resolution for input images, all the images in the train/validation dataset will be resized to this"
            " resolution"
        ),
    )
    parser.add_argument(
        "--center_crop",
        default=False,
        action="store_true",
        help=(
            "Whether to center crop the input images to the resolution. If not set, the images will be randomly"
            " cropped. The images will be resized to the resolution first before cropping."
        ),
    )
    parser.add_argument(
        "--random_flip",
        action="store_true",
        help="whether to randomly flip images horizontally",
    )
    parser.add_argument(
        "--train_batch_size", type=int, default=16, help="Batch size (per device) for the training dataloader."
    )
    parser.add_argument("--num_train_epochs", type=int, default=100)
    parser.add_argument(
        "--max_train_steps",
        type=int,
        default=None,
        help="Total number of training steps to perform.  If provided, overrides num_train_epochs.",
    )
    parser.add_argument(
        "--gradient_accumulation_steps",
        type=int,
        default=1,
        help="Number of updates steps to accumulate before performing a backward/update pass.",
    )
    parser.add_argument(
        "--gradient_checkpointing",
        action="store_true",
        help="Whether or not to use gradient checkpointing to save memory at the expense of slower backward pass.",
    )
    parser.add_argument(
        "--learning_rate",
        type=float,
        default=1e-4,
        help="Initial learning rate (after the potential warmup period) to use.",
    )
    parser.add_argument(
        "--scale_lr",
        action="store_true",
        default=False,
        help="Scale the learning rate by the number of GPUs, gradient accumulation steps, and batch size.",
    )
    parser.add_argument(
        "--lr_scheduler",
        type=str,
        default="constant",
        help=(
            'The scheduler type to use. Choose between ["linear", "cosine", "cosine_with_restarts", "polynomial",'
            ' "constant", "constant_with_warmup"]'
        ),
    )
    parser.add_argument(
        "--lr_warmup_steps", type=int, default=500, help="Number of steps for the warmup in the lr scheduler."
    )
    parser.add_argument(
        "--snr_gamma",
        type=float,
        default=None,
        help="SNR weighting gamma to be used if rebalancing the loss. Recommended value is 5.0. "
        "More details here: https://arxiv.org/abs/2303.09556.",
    )
    parser.add_argument(
        "--use_8bit_adam", action="store_true", help="Whether or not to use 8-bit Adam from bitsandbytes."
    )
    parser.add_argument(
        "--allow_tf32",
        action="store_true",
        help=(
            "Whether or not to allow TF32 on Ampere GPUs. Can be used to speed up training. For more information, see"
            " https://pytorch.org/docs/stable/notes/cuda.html#tensorfloat-32-tf32-on-ampere-devices"
        ),
    )
    parser.add_argument(
        "--dataloader_num_workers",
        type=int,
        default=0,
        help=(
            "Number of subprocesses to use for data loading. 0 means that the data will be loaded in the main process."
        ),
    )
    parser.add_argument("--adam_beta1", type=float, default=0.9, help="The beta1 parameter for the Adam optimizer.")
    parser.add_argument("--adam_beta2", type=float, default=0.999, help="The beta2 parameter for the Adam optimizer.")
    parser.add_argument("--adam_weight_decay", type=float, default=1e-2, help="Weight decay to use.")
    parser.add_argument("--adam_epsilon", type=float, default=1e-08, help="Epsilon value for the Adam optimizer")
    parser.add_argument("--max_grad_norm", default=1.0, type=float, help="Max gradient norm.")
    parser.add_argument("--push_to_hub", action="store_true", help="Whether or not to push the model to the Hub.")
    parser.add_argument("--hub_token", type=str, default=None, help="The token to use to push to the Model Hub.")
    parser.add_argument(
        "--prediction_type",
        type=str,
        default=None,
        help="The prediction_type that shall be used for training. Choose between 'epsilon' or 'v_prediction' or leave `None`. If left to `None` the default prediction type of the scheduler: `noise_scheduler.config.prediction_type` is chosen.",
    )
    parser.add_argument(
        "--hub_model_id",
        type=str,
        default=None,
        help="The name of the repository to keep in sync with the local `output_dir`.",
    )
    parser.add_argument(
        "--mixed_precision",
        type=str,
        default=None,
        choices=["no", "fp16", "bf16"],
        help=(
            "Whether to use mixed precision. Choose between fp16 and bf16 (bfloat16). Bf16 requires PyTorch >="
            " 1.10.and an Nvidia Ampere GPU.  Default to the value of accelerate config of the current system or the"
            " flag passed with the `accelerate.launch` command. Use this argument to override the accelerate config."
        ),
    )
    parser.add_argument(
        "--report_to",
        type=str,
        default="tensorboard",
        help=(
            'The integration to report the results and logs to. Supported platforms are `"tensorboard"`'
            ' (default), `"wandb"` and `"comet_ml"`. Use `"all"` to report to all integrations.'
        ),
    )
    parser.add_argument("--local_rank", type=int, default=-1, help="For distributed training: local_rank")
    parser.add_argument(
        "--checkpointing_steps",
        type=int,
        default=500,
        help=(
            "Save a checkpoint of the training state every X updates. These checkpoints are only suitable for resuming"
            " training using `--resume_from_checkpoint`."
        ),
    )
    parser.add_argument(
        "--checkpoints_total_limit",
        type=int,
        default=None,
        help=("Max number of checkpoints to store."),
    )
    parser.add_argument(
        "--resume_from_checkpoint",
        type=str,
        default=None,
        help=(
            "Whether training should be resumed from a previous checkpoint. Use a path saved by"
            ' `--checkpointing_steps`, or `"latest"` to automatically select the last available checkpoint.'
        ),
    )
    parser.add_argument(
        "--enable_xformers_memory_efficient_attention", action="store_true", help="Whether or not to use xformers."
    )
    parser.add_argument("--noise_offset", type=float, default=0, help="The scale of noise offset.")
    parser.add_argument(
        "--rank",
        type=int,
        default=4,
        help=("The dimension of the LoRA update matrices."),
    )

    # for inference
    parser.add_argument(
        "--ckpt",
        type=str,
        default=None,
        help="The checkpoint of the LoRA to load",
    )

    args = parser.parse_args()
    env_local_rank = int(os.environ.get("LOCAL_RANK", -1))
    if env_local_rank != -1 and env_local_rank != args.local_rank:
        args.local_rank = env_local_rank

    return args


def main():
    args = parse_args()

    # load custom dataset
    from dataset import load_domainnet
    captions_threshold = None
    split = "test"

    file_name = f"_clip_scores_val_{split}.npy" if captions_threshold is not None else f"_clip_scores_val_{split}_full.npy"
    save_path = args.ckpt.replace(".safetensors", file_name)

    if os.path.exists(save_path):
        eval_dict = np.load(save_path, allow_pickle=True).item()
        if "indomain_fid" in eval_dict:
            print(f"Skipping inference for {args.ckpt} as it already exists.")
            return

    test_captions, indomain_test_captions, outdomain_test_captions = [], [], []
    dataset = load_domainnet(full=False)
    for domain in dataset:
        for category in dataset[domain]:
            captions = list(set(dataset[domain][category][split]['text']))[:captions_threshold]
            indomain_test_captions.extend(captions)

    dataset = load_domainnet(full=True)
    for domain in dataset:
        for category in dataset[domain]:
            captions = list(set(dataset[domain][category][split]['text']))[:captions_threshold]
            test_captions.extend(captions)
    outdomain_test_captions = list(set(test_captions) - set(indomain_test_captions))
    print(f"split {split}, indomain: {len(indomain_test_captions)}, outdomain: {len(outdomain_test_captions)}, total: {len(test_captions)}")

    print(f"FEW CAPTIONS: {test_captions[:10]}")    

    # Final inference on test set
    pipeline = DiffusionPipeline.from_pretrained(
        args.pretrained_model_name_or_path,
        revision=args.revision,
        variant=args.variant,
        torch_dtype=torch.float32,
    )
    pipeline.safety_checker = None
    # pipeline.scheduler = DDIMScheduler.from_pretrained(args.pretrained_model_name_or_path, subfolder="scheduler", rescale_betas_zero_snr=True)
    pipeline = pipeline.to("cuda")
    
    # disable progress bar
    pipeline.set_progress_bar_config(disable=True)

    breakpoint()

    if "pretrained" in args.ckpt:
        print(f"Evaluating pretrained SD model")
    elif "multitask" in args.ckpt:
        print(f"Evaluating multitask model")
        peft_unet = pipeline.load_lora_weights("/h/calvinyu/evalmerge/output/all_all_May-17-2024-03-01-33")
        peft_unet.replace_with_AB_adapter(progressbar=True)

    else:
        # changed to return peft module after loading adapters
        peft_unet = pipeline.load_lora_weights("/h/calvinyu/evalmerge/output/clipart_cloth_May-12-2024-15-41-51/checkpoint-1/")
        assert hasattr(peft_unet, "merge_and_unload")

        # # num of total and trainable params
        # total_params = sum(p.numel() for p in peft_unet.parameters())
        # trainable_params = sum(p.numel() for p in peft_unet.parameters() if p.requires_grad)
        # print(f"[Before] Total parameters: {total_params // 1e6}, Trainable parameters: {trainable_params // 1e6}")
        
        # merge the A and B loras to full AB
        peft_unet.replace_with_AB_adapter(progressbar=True)

        # total_params = sum(p.numel() for p in peft_unet.parameters())
        # trainable_params = sum(p.numel() for p in peft_unet.parameters() if p.requires_grad)
        # print(f"[After] Total parameters: {total_params // 1e6}, Trainable parameters: {trainable_params // 1e6}")

        state_dict = {}
        from safetensors import safe_open
        checkpoint = args.ckpt
        with safe_open(checkpoint, framework="pt", device=0) as f:
            for k in f.keys():
                state_dict[k.replace("unet.", "model.")] = f.get_tensor(k)
        missing, unexpected = peft_unet.load_state_dict(state_dict, strict=False)
        assert len(unexpected) == 0

    # run inference
    generator = torch.Generator(device="cuda")
    if args.seed is not None:
        generator = generator.manual_seed(args.seed)
    if torch.backends.mps.is_available():
        autocast_ctx = nullcontext()
    else:
        autocast_ctx = torch.autocast("cuda")

    # build dataloader for batching
    class CaptionDataset(torch.utils.data.Dataset):
        def __init__(self, captions):
            self.captions = captions

        def __len__(self):
            return len(self.captions)

        def __getitem__(self, idx):
            return self.captions[idx]
    
    # repeat indomain captions 3x times (to make FID relevant)
    indomain_test_captions = indomain_test_captions * 3

    # # debug
    # indomain_test_captions = indomain_test_captions[:10]
    # outdomain_test_captions = outdomain_test_captions[:10]

    dataset = CaptionDataset(indomain_test_captions + outdomain_test_captions)
    dataloader = torch.utils.data.DataLoader(dataset, batch_size=8, shuffle=False, drop_last=False)

    # batching: https://huggingface.co/docs/transformers/en/main_classes/pipelines
    clip_scores = []; fid_images = []
    with autocast_ctx:
        for captions in tqdm(dataloader, total=len(dataloader), desc="inference"):
            images = pipeline(captions, num_inference_steps=30, generator=generator).images
            fid_images.extend(images)
            
            # calculate clip score
            all_images = [np.array(image).astype(np.float32) for image in images]
            all_images = torch.from_numpy(np.array(all_images))
            batch_clip_scores = clip_score_metric((all_images).int(), captions, cpu=False)
            clip_scores.extend(batch_clip_scores.tolist())
            

    indomain_clip_scores = clip_scores[:len(indomain_test_captions)]
    outdomain_clip_scores = clip_scores[len(indomain_test_captions):]
    
    scores = {
        "indomain": indomain_clip_scores,
        "outdomain": outdomain_clip_scores,
        "indomain_mean": np.mean(indomain_clip_scores),
        "outdomain_mean": np.mean(outdomain_clip_scores),
    }

    np.save(save_path, scores)
    print(f"saved clip scores to {save_path} | indomain: {len(indomain_clip_scores)} | outdomain: {len(outdomain_clip_scores)}")
    
    # save images for FID computation
    ts = datetime.now().strftime("%Y-%m-%d-%H-%M-%S")
    in_fid_folder = os.path.join(f"inference", ts, f"indomain")
    os.makedirs(in_fid_folder, exist_ok=True)
    for img, caption, idx in tqdm(zip(fid_images[:len(indomain_test_captions)], indomain_test_captions, range(len(indomain_test_captions))), total=len(indomain_test_captions), desc="saving indomain images"):
        img.save(os.path.join(in_fid_folder, f"{idx}_{caption}.jpg"))

    out_fid_folder = os.path.join(f"inference", ts, f"outdomain")
    os.makedirs(out_fid_folder, exist_ok=True)
    for img, caption, idx in tqdm(zip(fid_images[len(indomain_test_captions):], outdomain_test_captions, range(len(outdomain_test_captions))), total=len(outdomain_test_captions), desc="saving outdomain images"):
        img.save(os.path.join(out_fid_folder, f"{idx}_{caption}.jpg"))

    # compute FID
    indomain_dir = "/h/calvinyu/evalmerge/inference/fid/indomain"
    outdomain_dir = "/h/calvinyu/evalmerge/inference/fid/outdomain"

    # add to FID: --dims 768
    import subprocess
    fid_cmd = f"python -m pytorch_fid {indomain_dir} {in_fid_folder} --dims 192"
    indomain_result = subprocess.run(fid_cmd.split(" "), capture_output=True, text=True)
    indomain_result = repr(indomain_result.stdout)
    try:
        indomain_result = float(indomain_result.replace("\\n'", "").replace("'FID:", "").strip())
    except:
        pass 
    print(f"FID: {indomain_result}")

    fid_cmd = f"python -m pytorch_fid {outdomain_dir} {out_fid_folder} --dims 192"
    outdomain_result = subprocess.run(fid_cmd.split(" "), capture_output=True, text=True)
    outdomain_result = repr(outdomain_result.stdout)
    try:
        outdomain_result = float(outdomain_result.replace("\\n'", "").replace("'FID:", "").strip())
    except:
        pass 
    print(f"FID: {outdomain_result}")
    scores.update({"indomain_fid": indomain_result, "outdomain_fid": outdomain_result})
    np.save(save_path, scores)

    # delete saved images
    os.system(f"rm -rf {os.path.dirname(in_fid_folder)}")


if __name__ == "__main__":
    main()

"""
python scripts/inference.py --ckpt /scratch/ssd004/scratch/calvinyu/git_merge/stats/stats_sketch_water_transportation_AB.safetensors

python -m pytorch_fid "/h/calvinyu/evalmerge/inference/fid/outdomain" "/h/calvinyu/evalmerge/inference/2024-05-03-12-54-00/outdomain"

python -m pytorch_fid "/h/calvinyu/evalmerge/inference/fid/indomain" "/h/calvinyu/evalmerge/inference/2024-05-03-12-54-00/indomain" --dims 768

python -m pytorch_fid "/h/calvinyu/evalmerge/inference/fid/indomain" "/h/calvinyu/evalmerge/inference/2024-05-03-12-54-00/indomain" --dims 192

python -m pytorch_fid "/h/calvinyu/evalmerge/inference/fid/indomain" "/h/calvinyu/evalmerge/inference/2024-05-03-12-54-00/indomain" --dims 64


I need 200 hours to get FID and CLIP scores for all sequential models. If we only do CLIP, I can manage in 100 hours. This is A40 hours. 

On Vector, I generally use 6-8 A40s in parallel which expire in couple hours. 



"""
