import { getDefaultValueBySchema, ISchema } from '@shellagent/form-engine';
import { FormRef } from '@shellagent/ui';
import { useRequest } from 'ahooks';
import { useInjection } from 'inversify-react';
import { merge } from 'lodash-es';
import React, { useRef, useEffect, useState, useMemo } from 'react';

import { CommonWidgetConfigProps } from '@/components/app/config-form/widget-config';
import NodeForm from '@/components/app/node-form';
import { SettingsModel } from '@/components/settings/settings.model';

import { COMFYUI_API, DEFAULT_COMFYUI_API } from './constant';
import { useEventEmitter, EventType } from './emitter';
import { getComfyuiSchema, defaultSchema } from './schema';
import { getFile } from './services';
import { generateHash } from './utils';
import { ComfyUIEditor } from './widgets/comfyui-editor';

const settingsDisabled = process.env.NEXT_PUBLIC_DISABLE_SETTING === 'yes';

const ComfyUIPlugin: React.FC<CommonWidgetConfigProps> = ({
  values,
  onChange,
  modeMap,
  onModeChange,
  parent,
}) => {
  const formRef = useRef<FormRef>(null);
  const [schema, setSchema] = useState<ISchema>(defaultSchema);

  const model = useInjection(SettingsModel);

  useEffect(() => {
    const initializeForm = async () => {
      const values = formRef.current?.getValues();
      if (!values?.comfy_workflow_id) {
        formRef.current?.setValue('comfy_workflow_id', generateHash());
      }

      if (!settingsDisabled) {
        // 本地版
        const settings = await model.loadSettingsEnv();
        const api = settings?.envs?.find(env => env.key === COMFYUI_API)?.value;
        formRef.current?.setValue('api', api);
      } else {
        // 线上版
        formRef.current?.setValue('api', DEFAULT_COMFYUI_API);
      }
    };

    initializeForm();
  }, [model]);

  const defaultValues = useMemo(
    () => getDefaultValueBySchema(schema, false),
    [schema],
  );

  useEffect(() => {
    onChange(merge({}, defaultValues, values));
  }, [defaultValues]);

  const { run: getComfySchema, loading: isLoading } = useRequest(getFile, {
    manual: true,
    onSuccess: result => {
      if (result.success) {
        const {
          data: { schemas },
        } = result;
        setSchema(
          getComfyuiSchema({
            inputs: schemas.inputs || {},
            outputs: schemas.outputs || {},
          }),
        );
      }
    },
  });

  useEffect(() => {
    if (values?.comfy_workflow_id) {
      getComfySchema({
        comfy_workflow_id: values?.comfy_workflow_id,
        filename: 'workflow.shellagent.json',
      });
    }
  }, [values?.comfy_workflow_id]);

  useEventEmitter(EventType.UPDATE_FORM, eventData => {
    if (eventData.id === values?.comfy_workflow_id) {
      setSchema(
        getComfyuiSchema({
          inputs: eventData?.data?.inputs || {},
          outputs: eventData?.data?.outputs || {},
        }),
      );
    }
  });

  if (!values) {
    return null;
  }

  return (
    <div className="comfyui-widget flex flex-col">
      <NodeForm
        parent={parent}
        ref={formRef}
        key={JSON.stringify(schema)}
        schema={schema}
        values={values}
        onChange={onChange}
        onModeChange={onModeChange}
        loading={isLoading}
        modeMap={modeMap}
        components={{
          ComfyUIEditor,
        }}
      />
    </div>
  );
};

export default ComfyUIPlugin;
