import json
import os 
import tempfile
import requests

import gradio as gr
from huggingface_hub import HfApi, create_repo

inputs_description = """This is a description of the inputs that the prompt expects.

{{input_var}}: {{Description}}
...
"""
usage_description = """Below is a code snippet for how to use the prompt.

```python
{{Code snippet}}
```
"""
input_variables_description = "Comma-separated list of input variables. E.g. question,name"
template_description = "Imagine you're a teacher called {name}. A student asks the following question: {question}. What do you answer?"

api = HfApi()

def submit(name, description, inputs_description, usage_description, input_variables, template, token):
  # Join the organization
  headers = {"Authorization" : f"Bearer: {token}", "Content-Type": "application/json"}
  response = requests.post("https://huggingface.co/organizations/LangChainHub-Prompts/share/VNemVvLTwKsAPQpMKekDrIgzCyoXmKakzI", headers=headers)
    
  variables = input_variables.split(",")

  card = f"""
---
tags:
- langchain
- prompt
---

# Description of {name}

{description}

## Inputs

{inputs_description}

## Usage

{usage_description}
"""
  
  with tempfile.TemporaryDirectory() as tmpdir:
      with open(os.path.join(tmpdir, "prompt.json"), "w") as f:
        data = {
          'input_variables': variables, 
          'output_parser': None,
          "template": template,
          "template_format": "f-string"
        }
        json.dump(data, f, indent=4)

      with open(os.path.join(tmpdir, "README.md"), "w") as f:
        f.write(card)
      
      name = name.replace(" ", "_")
      model_id = f"LangChainHub/{name}"
      repo_url = create_repo(model_id, token=token, repo_type="dataset")
      res = api.upload_folder(
          repo_id=model_id,
          folder_path=tmpdir,
          token=token,
          repo_type="dataset"
      )
  return f'Success! Check out the result <a href=\'{repo_url}\' target="_blank" style="text-decoration:underline">here</a>'

with gr.Blocks() as form:
    gr.Markdown("# LangChain Hub Form")
    gr.Markdown("## Submit a prompt")
    name = gr.Textbox(lines=1, placeholder="Name for the prompt", label="Name")
    high_level_description = gr.Textbox(lines=1, placeholder="High level text description of the prompt, including use cases.", interactive=True, label="Description")
    inputs_description = gr.Textbox(lines=2, value=inputs_description, interactive=True, label="Inputs Description")
    usage_description = gr.Textbox(lines=3, value=usage_description, interactive=True, label="Usage Description")

    input_variables = gr.Textbox(value=input_variables_description, interactive=True, label="Input Variables")
    template = gr.Textbox(lines=3, value=template_description, interactive=True, label="Template (use the input variables with {})")
    token = gr.Textbox(label="Write Token (from https://huggingface.co/settings/tokens)", type="password")

    btn = gr.Button(value="Share Prompt")
    inputs = [name, high_level_description, inputs_description, usage_description, input_variables, template, token]
    output = gr.Markdown(label="output")
    btn.click(submit, inputs=inputs, outputs=[output])

form.launch(debug=True)