import gradio as gr
from huggingface_hub import hf_hub_download
import json
import torch
import os
from env import AttrDict
from meldataset import mel_spectrogram, MAX_WAV_VALUE
from models import BigVGAN as Generator
import librosa
import numpy as np
from utils import plot_spectrogram, plot_spectrogram_clipped
import PIL

torch.backends.cudnn.benchmark = False


def load_checkpoint(filepath, device):
    assert os.path.isfile(filepath)
    print("Loading '{}'".format(filepath))
    checkpoint_dict = torch.load(filepath, map_location=device)
    print("Complete.")
    return checkpoint_dict


def inference_gradio(input, model_choice):  # input is audio waveform in [T, channel]
    sr, audio = input  # unpack input to sampling rate and audio itself
    audio = np.transpose(audio)  # transpose to [channel, T] for librosa
    audio = audio / MAX_WAV_VALUE  # convert int16 to float range used by BigVGAN

    h = list_config[model_choice]
    model = list_model[model_choice]

    if sr != h.sampling_rate:  # convert audio to model's sampling rate
        audio = librosa.resample(audio, sr, h.sampling_rate)
    if len(audio.shape) == 2:  # stereo
        audio = librosa.to_mono(audio)  # convert to mono if stereo
    audio = librosa.util.normalize(audio) * 0.95
    output, spec_gen = inference_model(audio, h, model)  # output is generated audio in ndarray

    spec_plot_gen = plot_spectrogram(spec_gen.numpy())

    output_video = gr.make_waveform((h.sampling_rate, output))
    output_image_gen = PIL.Image.frombytes('RGB',
                                           spec_plot_gen.canvas.get_width_height(),
                                           spec_plot_gen.canvas.tostring_rgb())

    return output_video, output_image_gen


def inference_model(audio_input, h, model):
    def get_mel(x):
        return mel_spectrogram(x, h.n_fft, h.num_mels, h.sampling_rate, h.hop_size, h.win_size, h.fmin, h.fmax)

    with torch.no_grad():
        wav = torch.FloatTensor(audio_input).to(device)
        # compute mel spectrogram from the ground truth audio
        spec_gt = get_mel(wav.unsqueeze(0))

        y_g_hat = model(spec_gt)

        audio_gen = y_g_hat.squeeze()
        spec_gen = get_mel(audio_gen.unsqueeze(0))
        audio_gen = audio_gen * MAX_WAV_VALUE
        audio_gen = audio_gen.cpu().numpy().astype('int16')

        return audio_gen, spec_gen[0].cpu()


css = """
        a {
            color: inherit;
            text-decoration: underline;
        }
        .gradio-container {
            font-family: 'IBM Plex Sans', sans-serif;
        }
        .gr-button {
            color: white;
            border-color: #000000;
            background: #000000;
        }
        input[type='range'] {
            accent-color: #000000;
        }
        .dark input[type='range'] {
            accent-color: #dfdfdf;
        }
        .container {
            max-width: 730px;
            margin: auto;
            padding-top: 1.5rem;
        }
        #gallery {
            min-height: 22rem;
            margin-bottom: 15px;
            margin-left: auto;
            margin-right: auto;
            border-bottom-right-radius: .5rem !important;
            border-bottom-left-radius: .5rem !important;
        }
        #gallery>div>.h-full {
            min-height: 20rem;
        }
        .details:hover {
            text-decoration: underline;
        }
        .gr-button {
            white-space: nowrap;
        }
        .gr-button:focus {
            border-color: rgb(147 197 253 / var(--tw-border-opacity));
            outline: none;
            box-shadow: var(--tw-ring-offset-shadow), var(--tw-ring-shadow), var(--tw-shadow, 0 0 #0000);
            --tw-border-opacity: 1;
            --tw-ring-offset-shadow: var(--tw-ring-inset) 0 0 0 var(--tw-ring-offset-width) var(--tw-ring-offset-color);
            --tw-ring-shadow: var(--tw-ring-inset) 0 0 0 calc(3px var(--tw-ring-offset-width)) var(--tw-ring-color);
            --tw-ring-color: rgb(191 219 254 / var(--tw-ring-opacity));
            --tw-ring-opacity: .5;
        }
        #advanced-btn {
            font-size: .7rem !important;
            line-height: 19px;
            margin-top: 12px;
            margin-bottom: 12px;
            padding: 2px 8px;
            border-radius: 14px !important;
        }
        #advanced-options {
            margin-bottom: 20px;
        }
        .footer {
            margin-bottom: 45px;
            margin-top: 35px;
            text-align: center;
            border-bottom: 1px solid #e5e5e5;
        }
        .footer>p {
            font-size: .8rem;
            display: inline-block;
            padding: 0 10px;
            transform: translateY(10px);
            background: white;
        }
        .dark .footer {
            border-color: #303030;
        }
        .dark .footer>p {
            background: #0b0f19;
        }
        .acknowledgments h4{
            margin: 1.25em 0 .25em 0;
            font-weight: bold;
            font-size: 115%;
        }
        #container-advanced-btns{
            display: flex;
            flex-wrap: wrap;
            justify-content: space-between;
            align-items: center;
        }
        .animate-spin {
            animation: spin 1s linear infinite;
        }
        @keyframes spin {
            from {
                transform: rotate(0deg);
            }
            to {
                transform: rotate(360deg);
            }
        }
        #share-btn-container {
            display: flex; padding-left: 0.5rem !important; padding-right: 0.5rem !important; background-color: #000000; justify-content: center; align-items: center; border-radius: 9999px !important; width: 13rem;
            margin-top: 10px;
            margin-left: auto;
        }
        #share-btn {
            all: initial; color: #ffffff;font-weight: 600; cursor:pointer; font-family: 'IBM Plex Sans', sans-serif; margin-left: 0.5rem !important; padding-top: 0.25rem !important; padding-bottom: 0.25rem !important;right:0;
        }
        #share-btn * {
            all: unset;
        }
        #share-btn-container div:nth-child(-n+2){
            width: auto !important;
            min-height: 0px !important;
        }
        #share-btn-container .wrap {
            display: none !important;
        }
        .gr-form{
            flex: 1 1 50%; border-top-right-radius: 0; border-bottom-right-radius: 0;
        }
        #prompt-container{
            gap: 0;
        }
        #generated_id{
            min-height: 700px
        }
        #setting_id{
          margin-bottom: 12px;
          text-align: center;
          font-weight: 900;
        }
"""

######################## script for loading the models ########################

model_path = "L0SG/BigVGAN"
list_model_name = ["bigvgan_24khz_100band",
                   "bigvgan_base_24khz_100band",
                   "bigvgan_22khz_80band",
                   "bigvgan_base_22khz_80band"]

list_model = []
list_config = []

for model_name in list_model_name:
    model_file = hf_hub_download(model_path, "{}/g_05000000".format(model_name),
                                 use_auth_token="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")
    config_file = hf_hub_download(model_path, "{}/config.json".format(model_name),
                                  use_auth_token="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")
    with open(config_file) as f:
        data = f.read()

    json_config = json.loads(data)
    h = AttrDict(json_config)

    torch.manual_seed(h.seed)
    if torch.cuda.is_available():
        torch.cuda.manual_seed(h.seed)
        device = torch.device('cuda')
    else:
        device = torch.device('cpu')

    generator = Generator(h).to(device)
    state_dict_g = load_checkpoint(model_file, device)
    generator.load_state_dict(state_dict_g['generator'])
    generator.eval()
    generator.remove_weight_norm()

    list_model.append(generator)
    list_config.append(h)

######################## script for gradio UI ########################

iface = gr.Blocks(css=css)

with iface:
    gr.HTML(
        """
            <div style="text-align: center; max-width: 700px; margin: 0 auto;">
              <div
                style="
                  display: inline-flex;
                  align-items: center;
                  gap: 0.8rem;
                  font-size: 1.75rem;
                "
              >
                <h1 style="font-weight: 900; margin-bottom: 7px; line-height: normal;">
                  BigVGAN: A Universal Neural Vocoder with Large-Scale Training
                </h1>
              </div>
              <p style="margin-bottom: 10px; font-size: 94%">
                <a href="https://arxiv.org/abs/2206.04658">[Paper]</a>  <a href="https://github.com/NVIDIA/BigVGAN">[Code]</a>  <a href="https://bigvgan-demo.github.io/">[Demo]</a>  <a href="https://research.nvidia.com/labs/adlr/projects/bigvgan/">[Project page]</a>
              </p>
            </div>
        """
    )

    gr.HTML(
        """
            </div>
            <p>Select the model and submit the audio waveform. BigVGAN generates audio waveform using the mel spectrogram of the input.</p>
            <ul>
            <li>bigvgan_24khz_100band: 112M / 24kHz / 100-band mel spectrogram / fmax=12000</li>
            <li>bigvgan_base_24khz_100band: 14M / 24kHz / 100-band mel spectrogram / fmax=12000</li>
            <li>bigvgan_22khz_80band: 112M / 22.05kHz / 80-band mel spectrogram / fmax=8000</li>
            <li>bigvgan_base_22khz_80band: 14M / 22.05kHz / 80-band mel spectrogram / fmax=8000</li>
            </ul>
            <p><b>NOTE: All models are trained using speech audio datasets ONLY! (24kHz models: LibriTTS, 22kHz models: LibriTTS + VCTK + LJSpeech).</b></p>
            </div>
        """)

    with gr.Group():
        with gr.Box():
            model_choice = gr.Radio(label="Select the model. Default: bigvgan_24khz_100band",
                                    value="bigvgan_24khz_100band",
                                    choices=[m for m in list_model_name],
                                    type="index",
                                    interactive=True)
            audio_input = gr.Audio(label="Input Audio",
                                   elem_id="input-audio",
                                   interactive=True)
            button = gr.Button("Submit").style(full_width=True)
            output_video = gr.Video(label="Output Audio",
                                    elem_id="output-video")
            output_image_gen = gr.Image(label="Output Mel Spectrogram",
                                        elem_id="output-image-gen")
            button.click(inference_gradio,
                         inputs=[audio_input, model_choice],
                         outputs=[output_video, output_image_gen])

            gr.Examples(
                [
                    [os.path.join(os.path.dirname(__file__), "examples/jensen.wav"), "bigvgan_24khz_100band"],
                    [os.path.join(os.path.dirname(__file__), "examples/libritts.wav"), "bigvgan_24khz_100band"],
                    [os.path.join(os.path.dirname(__file__), "examples/queen.wav"), "bigvgan_24khz_100band"],
                    [os.path.join(os.path.dirname(__file__), "examples/dance.wav"), "bigvgan_24khz_100band"],
                    [os.path.join(os.path.dirname(__file__), "examples/megalovania.wav"), "bigvgan_24khz_100band"],
                ],
                fn=inference_gradio,
                inputs=[audio_input, model_choice],
                outputs=[output_video, output_image_gen],
                cache_examples=True
            )

iface.queue(concurrency_count=3)
iface.launch()
