import re
import unidecode
import numpy as np
import pandas as pd
import datetime_cleaner
from datetime import datetime


def run():
    
    # The path to the local git repo for Indic NLP library
    INDIC_NLP_LIB_HOME=r"./indic_nlp_library"

    # The path to the local git repo for Indic NLP Resources
    INDIC_NLP_RESOURCES=r"./indic_nlp_resources"
    import sys
    sys.path.append(r'{}'.format(INDIC_NLP_LIB_HOME))

    from indicnlp import common
    common.set_resources_path(INDIC_NLP_RESOURCES)

    from indicnlp import loader
    loader.load()

run()

from indicnlp.normalize.indic_normalize import IndicNormalizerFactory
from indicnlp.tokenize import sentence_tokenize
import normalizer_utils 

lang='sa'
factory=IndicNormalizerFactory()
normalizer=factory.get_normalizer(lang)


DEFAULT_TEXT = "अयं द्वितीयशब्दः २ अस्ति। प्रथमः शब्दः १ अस्ति।  २३ २ ३ अन्ये शब्दाः सर्वे द्वितीयं शब्दं प्रयोजयन्ति। इत्थं सप्ततिः शब्दाः लिखिताः सन्ति। अस्मिन लेखने सर्वे अक्षराः संस्कृते लिखिताः सन्ति। अन्ये लिखन्ति ३, ४, ५ इत्यादि। तथापि, अहं एकं अक्षरं एव उपयोगामि।"
# DEFAULT_TEXT = "आन्द्रिया २ महोदयः, जोसेफ् ३२ 6 7 महोदयः च कालः श्रीमती जेनेट् इत्यनेन सह स्थलं गतवन्तौ।"

df = pd.read_csv('non_standard_words.txt')
columns = ['non_standard_words']
df.columns = columns
nsw = df['non_standard_words'].tolist()
dict_num = normalizer_utils.dict_num
punctuation_marks = normalizer_utils.punctuation_marks


def tokenize_sentence(text):
    '''Tokenize a paragraph into sentences'''
    sentences = sentence_tokenize.sentence_split(text, lang='sa')
    sentences = "  #  ".join(sentences)
    return sentences


def segment_sentence(text):
    '''Segment a sentence into individual words'''
    segmented_sentence = text.split(' ')
    return segmented_sentence


def parse_datetime(text):
    '''Parses date and time and returns them as normalized texts'''
    text_list = []
    for word in text:
        if datetime_cleaner.is_date_or_year(word):
            #If the word is a datetime string, normalize the date.
            normalized_date = datetime_cleaner.normalize_date(word)
            text_list.append(normalized_date)
        else:
            text_list.append(word)
    text_str = " ".join(text_list)
    return text_str

def handle_abbreviations(text):
    '''Handles abbreviations'''
    abbrev_dict = normalizer_utils.abbreviation_dict
    text_list = []
    output_string = " "
    for char in text:     
        if char in abbrev_dict:
            output_string = abbrev_dict[char]
            char = char.replace(char, output_string)

        text_list.append(char)
    text_str = " ".join(text_list)
    return text_str


def remove_nsw(text):
    """return non standard words"""
    text_list = []
    for string in text:
        if string in nsw:
            string.replace(string, "")
        text_list.append(string)
    text_str = " ".join(text_list)

    return text_str


def normalize_text(text):
    text_list = []
    output_string = " "
    #Map sanskrit numbers to their normalized form.
    for char in text.split(" "):     
        if char in dict_num:
            output_string = dict_num[char]
            char = char.replace(char, output_string)

        text_list.append(char)
    text_str = " ".join(text_list)
    return text_str


def syllabify_text(text):
    '''Syllabifies text'''
    text_list = []
    #Syllabify text
    for char in text:
        if char in normalizer_utils.DEPENDENT_VOWELS:
            char = "(" + char + ")"
            text_list.append(char)
        else:
            text_list.append(char)

    full_text = " + ".join(text_list).replace("'", "")
    return full_text



def clean_text(text):
    processed_text = re.sub(r'\+  +', '', text)
    processed_text = re.sub(': +', '\n \n', processed_text)
    processed_text = re.sub(r'\+ ।', '\n \n', processed_text)
    processed_text = re.sub(r'\+$', '', processed_text)
    processed_text = re.sub(r'\+ , +', '', processed_text)
    processed_text = re.sub(r'\+ #', '\n', processed_text)
    return processed_text

def remove_punctuation(text):
    text_list = []
    for char in text:
        if char in punctuation_marks:
            char = char.replace(char, "")
            text_list.append(char)
        else:
            text_list.append(char)
    text_str = "".join(text_list)

    return text_str

def preprocess_text(text):
    cleaned_text = clean_text(text)

    #Remove unnecessary characters from a string.
    text_cleaned = []
    for index, text in enumerate(cleaned_text.split('\n')):
        if text.startswith('+'):
            text = text[2:]

        elif text.startswith(' +'):
            text = text[3:]

        elif text.endswith('+') or text.endswith(' +'):
            text = text[:-2]

        text_cleaned.append(text)
    
    text_cleaned_str = "\n ".join(text_cleaned)

    return text_cleaned_str
    
def pipeline(sentence):
    '''The whole pipeline for cleaning text from text normalization to removing special characters.'''
    tokenized_sentence = tokenize_sentence(sentence)
    segmented_sentence_list = segment_sentence(tokenized_sentence)
    formatted_datetime_list = list(map(datetime_cleaner.handle_time, segmented_sentence_list))
    formatted_datetime = ''.join(formatted_datetime_list)
    parsed_datetime_sentence = parse_datetime(formatted_datetime)
    formatted_abbreviations = list(map(handle_abbreviations, parsed_datetime_sentence))
    nsw_cleaned = remove_nsw(formatted_abbreviations)
    normalized_text = normalize_text(nsw_cleaned)
    syllabified_text = syllabify_text(normalized_text)
    text_wo_punctuation = remove_punctuation(syllabified_text)

    cleaned_text = clean_text(text_wo_punctuation)
    preprocessed_text = preprocess_text(cleaned_text)
    return preprocessed_text

g2p_vocab = ['(', ')', '+']
def g2p_pipeline(text):
    text_list = []
    for char in text:
        if char in g2p_vocab:
            char = char.replace(char, '')
            text_list.append(char)
        else:
            text_list.append(char)
    text = ''.join(text_list)
    return text



# vocab = ['ा', 'ि', 'ी', 'ु', 'ू', 'े', 'ै', 'ो', 'ौ', 'ं', 'ः', 'ृ', 'ॄ', '(', ')', '+', ' ']
vocab = ['(', ')', '+', ' ']
def voice_smoothening(text):
    
    '''This function removes syllables from text after performing text cleaning. 
        Used for g2p tab.'''
    text_list = []
    #Loop through characters in text and remove special characters such as + and syllables.
    for char in text:
        # Search for brackets in the sentence
        if char in vocab:
            char = char.replace(char, '')
            text_list.append(char)
        else:
            text_list.append(char)

    smoothened_text = "".join(text_list)
    cleaned_smoothened_text = clean_text(smoothened_text)
    return cleaned_smoothened_text

g2p_dict = {'ॠ': 'ḹ', 'द': 'jha', 'र': 'o', 'व': 'ṅa', 'ओ': 'ṃ', 'अः': 'i',
            'क': 'va', 'श': 'ḍa', 'झ': 'la', 'स': 'ā', 'ऋ': 'na', 'अ': 'au', 
            'आ': 'tha', 'उ': 'ṝ', 'ष': 'kha', 'ऌ': 'u', 'थ': 'ḷ', 'च': 'ṭa', 
            'ठ': 'gha', 'घ': 'da', 'ऊ': 'ma', 'ढ': 'śa', 'प': 'ca', 'ड': 'ya', 
            'इ': 'ṭha', 'य': 'ṇa', 'त': 'ha', 'अं': 'ja', 'फ': 'ṣa', 'ग': 'ṛ', 
            'ऐ': 'ta', 'ट': 'pa', 'ल': 'ba', 'ई': 'sa', 'ए': 'ai', 'ब': 'dha', 
            'न': 'pha', 'ण': 'a', 'ॡ': 'ḥ', 'ह': 'e', 'ख': 'ga', 'छ': 'ī', 
            'ञ': 'cha', 'म': 'bha', 'औ': 'ra', 'ङ': 'ña', 'ध': 'ka', 'भ': 'ḍha', 'ज': 'ū'}


def grapheme_to_phoneme(text):
    '''Takes cleaned text (grapheme) as input and returns its phoneme equivalent. Done after voice smoothening part.'''
    smoothened_text = voice_smoothening(text)
    text_list = []
    for char in smoothened_text:
        #Search for graphemes.
        if char in g2p_dict.keys():
            char = char.replace(char, g2p_dict[char])
            print(f"Replaced char: {char}")
            text_list.append(char)
        else:
            print(f"Unreplaced char: {char}")
            text_list.append(char)
    g2p_text = ' '.join(text_list)
    return g2p_text


sample_text = 'स + (ं) + स + ् + क + (ृ) + त + म + ् + ज + ग + त + (ः) + ए + क + त + म + (ा) + अ + त + (ि) + प + ् + र + (ा) + च + (ी) + न + (ा) + स + म + (ृ) + द + ् + ध + (ा) + श + (ा) + स + ् + त + ् + र + (ी) + य + (ा) + च + भ + (ा) + ष + (ा) + स + (ु) + व + र + ् + त + त + (े)'
