import torch
import commons
import utils
import re

from models import SynthesizerTrn
from text import text_to_sequence
import numpy as np
from mel_processing import spectrogram_torch
import gradio as gr
import soundfile as sf
from indic_transliteration import sanscript
from cleaner_utils import pipeline, voice_smoothening, g2p_pipeline

SCRIPT_DICT={
    'Devanagari':sanscript.DEVANAGARI
}
DEFAULT_TEXT='संस्कृतम् जगतः एकतमा अतिप्राचीना समृद्धा शास्त्रीया च भाषासु वर्तते'

DEPENDENT_VOWELS = ["ा", "ि", "ी", "ु", "ू", "े", "ै", "ो", "ौ", "ं", "ः", "ृ", "ॄ"]


def get_text(text, hps, cleaned=False):
    if cleaned:
        text_norm = text_to_sequence(text, hps.symbols, [])
    else:
        text_norm = text_to_sequence(text, hps.symbols, hps.data.text_cleaners)
    if hps.data.add_blank:
        text_norm = commons.intersperse(text_norm, 0)
    text_norm = torch.LongTensor(text_norm)
    return text_norm

def default_text(script):
    if script=='Devanagari':
        return DEFAULT_TEXT
    else:
        return sanscript.transliterate(DEFAULT_TEXT,sanscript.DEVANAGARI,SCRIPT_DICT[script])

def speech_synthesize(text,script, speaker_id, length_scale):
    text=text.replace('\n','')
    text = g2p_pipeline(text)
    print(f"G2P pipeline text is: {text}")
    if script!='Devanagari':
        text=sanscript.transliterate(text,SCRIPT_DICT[script],sanscript.DEVANAGARI)
    print(text, type(text))
    stn_tst = get_text(text, hps_ms)
    with torch.no_grad():
        x_tst = stn_tst.unsqueeze(0)
        x_tst_lengths = torch.LongTensor([stn_tst.size(0)])
        sid = torch.LongTensor([speaker_id])
        audio = net_g_ms.infer(x_tst, x_tst_lengths, sid=sid, noise_scale=0.667, noise_scale_w=0.8, length_scale=length_scale)[0][0,0].data.cpu().float().numpy()

    sf.write(f'audio_outputs/{text[:4]}.mp3', audio, hps_ms.data.sampling_rate)
    return (hps_ms.data.sampling_rate, audio)

def speech_synthesize_smoothened(text,script, speaker_id, length_scale):
    '''Synthesizes smoothened voice.'''
    text = text.replace('\n', ' ')
    text = voice_smoothening(text)
    print(f"Smoothened text is: {text}")
    if script!='Devanagari':
        text=sanscript.transliterate(text,SCRIPT_DICT[script],sanscript.DEVANAGARI)
    print(text, type(text))
    stn_tst = get_text(text, hps_ms)
    with torch.no_grad():
        x_tst = stn_tst.unsqueeze(0)
        x_tst_lengths = torch.LongTensor([stn_tst.size(0)])
        sid = torch.LongTensor([speaker_id])
        audio = net_g_ms.infer(x_tst, x_tst_lengths, sid=sid, noise_scale=0.667, noise_scale_w=0.8, length_scale=length_scale)[0][0,0].data.cpu().float().numpy()

    sf.write(f'audio_outputs/{text[:5]}.mp3', audio, hps_ms.data.sampling_rate)
    return (hps_ms.data.sampling_rate, audio)



if __name__=='__main__':
    hps_ms = utils.get_hparams_from_file('model/config.json')
    n_speakers = hps_ms.data.n_speakers
    n_symbols = len(hps_ms.symbols)
    speakers = hps_ms.speakers

    net_g_ms = SynthesizerTrn(
        n_symbols,
        hps_ms.data.filter_length // 2 + 1,
        hps_ms.train.segment_size // hps_ms.data.hop_length,
        n_speakers=n_speakers,
        **hps_ms.model)
    
    _ = net_g_ms.eval()
    utils.load_checkpoint('model/model.pth', net_g_ms)

    with gr.Blocks() as app:
        gr.Markdown('# Sanskrit Text to Speech\n')
        
        with gr.Tabs():
            # Tab for cleaning and printing input text.
            with gr.TabItem('Text processing'):
                with gr.Row():
                    text_script=gr.Radio(['Devanagari'],label='Script', value='Devanagari')
                    text_input = gr.TextArea(label='Text', value = DEFAULT_TEXT )
                    text_output = gr.Textbox(label="Output", interactive = True)

                text_button = gr.Button('Clean text')
                text_button.click(pipeline, inputs = text_input, outputs = text_output)

            with gr.TabItem('Grapheme to phoneme'):
                    #Utilities for speech synthesis.
                text_script=gr.Radio(['Devanagari'],label='Script',interactive=True,value='Devanagari')
                text_input = gr.TextArea(label='Text', placeholder='संस्कृत')
                speaker_id=gr.Dropdown(speakers,label='Speaker',type='index',interactive=True,value=speakers[0])
                length_scale=gr.Slider(minimum = 1, maximum = 2, step=0.2, label='Speaking Speed',interactive=False)
                
                tts_button = gr.Button('Synthesize')
                audio_output = gr.Audio(label='Speech Synthesized')
            text_script.change(default_text,[text_script],[text_input])
            tts_button.click(speech_synthesize,[text_input,text_script,speaker_id, length_scale],[audio_output])


            with gr.TabItem('Text to speech'):
                #Utilities for g2p sanskrit
                    #Utilities for speech synthesis.
                text_script=gr.Radio(['Devanagari'],label='Script',interactive=True,value='Devanagari')
                text_input = gr.TextArea(label='Text', placeholder='संस्कृत')
                speaker_id=gr.Dropdown(speakers,label='Speaker',type='index',interactive=True,value=speakers[0])
                length_scale=gr.Slider(minimum = 1, maximum = 2, step=0.2, label='Speaking Speed',interactive=False)
                tts_button = gr.Button('Synthesize')
                audio_output = gr.Audio(label='Speech Synthesized')
            text_script.change(voice_smoothening,[text_script],[text_input])
            tts_button.click(speech_synthesize_smoothened,[text_input,text_script,speaker_id, length_scale],[audio_output])

            

    app.launch(share = False)
