import '@babylonjs/loaders/glTF'
import { Animation, CreateBox, CubicEase, SceneLoader, Vector3 } from '@babylonjs/core'
import type { Animatable, ISceneLoaderAsyncResult, Mesh, Scene } from '@babylonjs/core'
import { MODEL_ASSETS_ROOT_URL } from './constants'
import { assert } from './utils'
import { LaserCutter } from './LaserCutter'

enum Pose {
  TakeOff,
  Land,
  SpinLeft,
  SpinRight,
  Hover,
  Forward,
  Backward,
}
export class Robot {
  static Pose = Pose
  assets?: ISceneLoaderAsyncResult
  scene: Scene
  mesh: Mesh
  pose: Pose = Pose.Land
  laserCutter: LaserCutter

  private _movingAnimatable = new Set<Animatable>()

  constructor(scene: Scene) {
    this.mesh = CreateBox('Root', { size: 2 }, scene)
    this.mesh.isVisible = false
    this.scene = scene
    this.laserCutter = new LaserCutter(scene)
    this.mesh.addChild(this.laserCutter.pivot)
    this.loadAssets(scene)
  }

  private async loadAssets(scene: Scene) {
    const result = await SceneLoader.ImportMeshAsync(null, `${MODEL_ASSETS_ROOT_URL}/buster_drone/`, 'buster_drone.gltf', scene)
    const root = result.meshes[0]
    const bbox = root.getHierarchyBoundingVectors()

    this.assets = result
    this.assets.animationGroups.forEach(anim => anim.pause())
    this.mesh.addChild(root)
    this.laserCutter.pivot.translate(Vector3.Up(), 1)
    root.translate(Vector3.Up(), -bbox.min.y)
  }

  async takeOff() {
    await this._playAnimation(Pose.TakeOff, false)
    this._playAnimation(Pose.Hover, true)
    if (this._movingAnimatable.size === 0)
      await this.moveTo(new Vector3(0.0, 12.0, 0.0))
  }

  async land() {
    await this._playAnimation(Pose.Land, false)
  }

  async moveStop() {
    await this._playAnimation(Pose.Backward, false)
    await this._playAnimation(Pose.Forward, false)
    this._playAnimation(Pose.Hover, true)
  }

  async moveTo(destination: Vector3) {
    const tn = this.mesh
    assert(!!tn, 'Root must be existed.')

    this._movingAnimatable.forEach((anim) => {
      anim.stop()
      this._movingAnimatable.delete(anim)
    })

    const SPEED = 6.0
    const position = tn.position.clone()
    const frameRate = 1 / (destination.subtract(position).length() / SPEED)
    const anim = new Animation('Move', 'position', frameRate, Animation.ANIMATIONTYPE_VECTOR3, Animation.ANIMATIONLOOPMODE_CONSTANT)
    anim.setKeys([
      { frame: 0, value: position },
      { frame: 1, value: destination },
    ])
    anim.setEasingFunction(new CubicEase())
    tn.animations.push(anim)
    const animatable = this.scene.beginAnimation(tn, 0, 1)
    this._movingAnimatable.add(animatable)
    await animatable.waitAsync()
    tn.animations.splice(tn.animations.findIndex(a => a === anim), 1)
    this._movingAnimatable.delete(animatable)

    await this.moveStop()
  }

  private async _playAnimation(pose: Pose, loop: boolean, percentage: number = 1) {
    this.pose = pose
    const anims: Record<Pose, [number, number, number] | [number, number]> = {
      [Pose.TakeOff]: [0, 200, 2],
      [Pose.Land]: [200, 0, 2],
      [Pose.Forward]: [230, 201, 1.5],
      [Pose.Backward]: [201, 230, 1.5],
      [Pose.SpinLeft]: [231, 293],
      [Pose.SpinRight]: [293, 231],
      [Pose.Hover]: [400, 600],
    }
    if (anims[pose]) {
      let [startFrame, endFrame, speedRatio = 1] = anims[pose]
      if (startFrame > endFrame)
        startFrame = (startFrame - endFrame) * percentage + endFrame
      else
        endFrame = startFrame + (endFrame - startFrame) * percentage

      await this._playFrames(startFrame, endFrame, loop, speedRatio)
    }
  }

  private _playFrames(from: number, to: number, loop: boolean, speedRatio: number) {
    // Frames inspected in Blender is 600, but in here it's 1500, scale to align with Blender.
    const SCALE = 1500 / 600
    const { scene } = this
    const anims = this.assets?.animationGroups.flatMap((g) => {
      return g.targetedAnimations.flatMap((target) => {
        target.animation.enableBlending = true
        return scene.beginAnimation(target.target, from * SCALE, to * SCALE, loop, speedRatio)
      })
    })
    return Promise.any((anims ?? []).flatMap(anim => anim.waitAsync()))
  }

  static create(scene: Scene) {
    return new Robot(scene)
  }
}
