import { Observable } from '@babylonjs/core'
import { Image as GUIImage } from '@babylonjs/gui'
import type { ImageClassificationResult, ImageDetectionItem } from './interfaces'
import { API_BASE_URL } from './constants'

interface IBaseImage {
  isLoaded(): this is ILoadedImage
  isDetected(): this is IDetectedImage
  isClassified(): this is IClassifiedImage
}
interface ILoadedImage extends IBaseImage {
  file: File
  image: GUIImage
}
interface IDetectedImage extends IBaseImage {
  file: File
  image: GUIImage
  dectection: ImageDetectionItem[]
}
interface IClassifiedImage extends IBaseImage {
  file: File
  image: GUIImage
  detections: ImageDetectionItem[]
  classification: ImageClassificationResult
}
export interface IImageEvent {
  type: 'loaded' | 'detected' | 'classified'
  target: Image
}
export class Image implements IBaseImage {
  file?: File
  image?: GUIImage
  detections?: ImageDetectionItem[]
  classification?: ImageClassificationResult
  observable = new Observable<IImageEvent>()

  isLoaded(): this is ILoadedImage {
    return !!this.file && !!this.image
  }

  isDetected(): this is IDetectedImage {
    return this.isLoaded() && !!this.detections
  }

  isClassified(): this is IClassifiedImage {
    return this.isDetected() && !!this.classification
  }

  clear(): void {
    this.file = undefined
    this.image = undefined
    this.detections = undefined
    this.classification = undefined
  }

  async load(): Promise<GUIImage> {
    await this._prompt()
    const image = await this._parse()
    this.observable.notifyObservers({ type: 'loaded', target: this })
    return image
  }

  private async _prompt(): Promise<File> {
    const input = document.createElement('input')
    input.type = 'file'
    input.accept = '.png,.jpg,.jpeg'
    return new Promise((resolve, reject) => {
      input.addEventListener('change', () => {
        const file = input.files?.[0]
        if (!file) {
          reject(new Error('No file is selected.'))
        }
        else {
          this.file = file
          resolve(file)
        }
      })
      input.click()
    })
  }

  private async _parse(): Promise<GUIImage> {
    const { file } = this
    if (!file)
      throw new Error('Image must be prompted before calling parsed.')
    return new Promise((resolve) => {
      const url = URL.createObjectURL(file)
      const image = new GUIImage('Picture', url)
      this.image = image
      image.onImageLoadedObservable.addOnce(() => resolve(image))
    })
  }

  async detect(): Promise<ImageDetectionItem[]> {
    if (!this.isLoaded())
      throw new Error('Image must be loaded before calling detect.')

    const { file } = this
    const form = new FormData()
    form.append('image', file)
    const response = await fetch(
      `${API_BASE_URL}detect-image`,
      {
        method: 'POST',
        body: form,
      },
    )
    const result = await response.json()
    this.detections = result
    this.observable.notifyObservers({ type: 'detected', target: this })
    return result
  }

  async classify(candidateLable: string): Promise<ImageClassificationResult> {
    if (!this.isDetected())
      throw new Error('Image must be loaded and detected before calling classify.')

    const { file, detections } = this
    const form = new FormData()
    form.append('image', file)
    form.append('candidate_label', candidateLable)
    form.append('detections', JSON.stringify(detections))
    const response = await fetch(
      `${API_BASE_URL}classify-image`,
      {
        method: 'POST',
        body: form,
      },
    )
    const result = await response.json()
    this.classification = result
    this.observable.notifyObservers({ type: 'classified', target: this })
    return result
  }

  async fromText(prompt?: string) {
    const headers = new Headers()
    headers.set('Content-Type', 'application/json')
    const response = await fetch(
      `${API_BASE_URL}text-to-image`,
      {
        headers,
        method: 'POST',
        body: JSON.stringify({ prompt: prompt || 'A lot of people walking through a street, there are lot of cars in the background.' }),
      },
    )
    const file = await response.arrayBuffer()
    this.file = new File([file], 'image', { type: 'image/png' })
    await this._parse()
    this.observable.notifyObservers({ type: 'loaded', target: this })
  }
}
