import { Matrix, MeshBuilder } from '@babylonjs/core'
import type { GroundMesh, Mesh, Scene } from '@babylonjs/core'
import { AdvancedDynamicTexture, Container, Control, Rectangle } from '@babylonjs/gui'
import type { Image } from './Image'
import { warn } from './utils'

export class Ground {
  mesh: GroundMesh
  image: Image
  private _pictureTexture: AdvancedDynamicTexture
  constructor(scene: Scene, image: Image) {
    this.mesh = this._createMesh(scene)
    this._pictureTexture = AdvancedDynamicTexture.CreateForMesh(this.mesh)
    this.image = image
    image.observable.add((event) => {
      if (event.type === 'loaded')
        this._displayImage()
      else if (event.type === 'detected')
        this._displayDetections()
      else if (event.type === 'classified')
        this._displayClassification()
    })
  }

  private _createMesh(scene: Scene) {
    const ground = MeshBuilder.CreateGround('Ground', { width: 25, height: 25 }, scene)
    return ground
  }

  private async _displayImage() {
    const { image, mesh, _pictureTexture } = this
    if (!image.isLoaded())
      return
    this._clearImage()
    this._clearDetections()
    this._clearClassification()
    const _image = image.image
    const bbox = mesh.getBoundingInfo().boundingBox
    const scaleX = (_image.imageWidth / _image.imageHeight) / (bbox.extendSize.x / bbox.extendSize.z)
    const scaling = Matrix.Scaling(scaleX, 1, 1)
    mesh.bakeTransformIntoVertices(scaling)
    mesh.getChildMeshes().forEach(child => (child as Mesh).bakeTransformIntoVertices(scaling))
    const container = new Container('ImageContainer')
    container.addControl(_image)
    _pictureTexture.addControl(container)
  }

  private _clearImage() {
    const container = this._pictureTexture.getControlByName('ImageContainer')
    if (container)
      this._pictureTexture.removeControl(container)
  }

  private async _displayDetections() {
    if (!this.image.isDetected()) {
      warn('Image must be loaded before setting detections.')
      return
    }
    this._clearDetections()
    const { _pictureTexture } = this
    const { image: _image, detections } = this.image
    const container = new Container('DetectionsContainer')
    detections!.forEach((detection) => {
      const rect = new Rectangle(detection.label)
      rect.width = (detection.box.xmax - detection.box.xmin) / _image.imageWidth
      rect.height = (detection.box.ymax - detection.box.ymin) / _image.imageHeight
      rect.left = `${detection.box.xmin / _image.imageWidth * 100}%`
      rect.top = `${detection.box.ymin / _image.imageHeight * 100}%`
      rect.color = '#ffff00'
      rect.thickness = 4
      rect.background = '#ffff0026'
      rect.verticalAlignment = Control.VERTICAL_ALIGNMENT_TOP
      rect.horizontalAlignment = Control.HORIZONTAL_ALIGNMENT_LEFT
      container.addControl(rect)
    })
    _pictureTexture.addControl(container)
  }

  private _clearDetections() {
    const container = this._pictureTexture.getControlByName('DetectionsContainer')
    if (container)
      this._pictureTexture.removeControl(container)
  }

  private _displayClassification() {
    if (!this.image.isClassified()) {
      warn('Image must be classified before displayed.')
      return
    }
    this._clearClassification()
    const { _pictureTexture } = this
    const { classification: { detection }, image: _image } = this.image
    const rect = new Rectangle(detection.label)
    rect.width = (detection.box.xmax - detection.box.xmin) / _image.imageWidth
    rect.height = (detection.box.ymax - detection.box.ymin) / _image.imageHeight
    rect.left = `${detection.box.xmin / _image.imageWidth * 100}%`
    rect.top = `${detection.box.ymin / _image.imageHeight * 100}%`
    rect.color = '#00ff00'
    rect.thickness = 4
    rect.background = '#00ff0026'
    rect.verticalAlignment = Control.VERTICAL_ALIGNMENT_TOP
    rect.horizontalAlignment = Control.HORIZONTAL_ALIGNMENT_LEFT
    const container = new Container('ClassificationContainer')
    container.addControl(rect)
    _pictureTexture.addControl(container)
  }

  private _clearClassification() {
    const container = this._pictureTexture.getControlByName('ClassificationContainer')
    if (container)
      this._pictureTexture.removeControl(container)
  }

  static create(scene: Scene, image: Image) {
    return new Ground(scene, image)
  }
}
