import { AdvancedDynamicTexture, Button, Container, Control, InputText, Rectangle, TextBlock } from '@babylonjs/gui'
import type { Scene } from '@babylonjs/core'
import { Observable } from '@babylonjs/core'
import { EAppUIState } from './enums'
import { Theme } from './constants'
import { warn } from './utils'
import { Loading } from './Loading'

interface IAppUIState {
  activate(appUI: AppUI): void
}
interface IAppUIBaseEvent<T, U> {
  type: T
  value: any
  target: U
}

interface IUploadImageButtonClickEvent extends IAppUIBaseEvent<'UploadImageButtonClick', Button> {}
interface IInputTextChangeEvent extends IAppUIBaseEvent<'InputTextChange', InputText> {}
interface ICreateImageButtonClickEvent extends IAppUIBaseEvent<'CreateImageButtonClick', ILoadingButton> {}

export type IAppUIEvent = IUploadImageButtonClickEvent | IInputTextChangeEvent | ICreateImageButtonClickEvent

export class AppUI {
  scene: Scene
  ui: AdvancedDynamicTexture
  observalbe: Observable<IAppUIEvent>
  controls: Control[] = []
  private _state: IAppUIState
  constructor(scene: Scene) {
    this.scene = scene
    this.ui = AdvancedDynamicTexture.CreateFullscreenUI('GUI', true, scene)
    this._state = new AppUIInitialState(this)
    this.observalbe = new Observable()
  }

  setState(state: EAppUIState) {
    this.ui.getChildren().forEach(child => child.dispose())
    if (state === EAppUIState.Initial)
      this._state = new AppUIInitialState(this)
    else if (state === EAppUIState.Input)
      this._state = new AppUIInputState(this)
    else
      warn(`Unaviable AppUIState: ${state}`)
  }
}

export type ILoadingButton = InstanceType<typeof LoadingButton>
const LoadingButton = Loading(class extends Button {
  scene: Scene

  private _innerTextBlock: TextBlock

  constructor(scene: Scene, name: string, text: string) {
    super(name)
    this.scene = scene
    const textBlock = new TextBlock(`${name}_button`, text)
    textBlock.textWrapping = true
    textBlock.textHorizontalAlignment = Control.HORIZONTAL_ALIGNMENT_CENTER
    this.addControl(textBlock)
    this._innerTextBlock = textBlock
  }
})

export type ILoadingRectangle = InstanceType<typeof LoadingRectangle>
const LoadingRectangle = Loading(class extends Rectangle {
  scene: Scene
  constructor(scene: Scene, ...args: any[]) {
    super(...args)
    this.scene = scene
  }
})

abstract class BaseAppUIState implements IAppUIState {
  appUI: AppUI
  constructor(appUI: AppUI) {
    appUI.controls = []
    this.appUI = appUI
    this.activate(appUI)
  }
  abstract activate(appUI: AppUI): void
}

class AppUIInitialState extends BaseAppUIState {
  activate(appUI: AppUI): void {
    const container = new Container('GreetingModal')
    appUI.ui.addControl(container)
    container.width = '860px'
    container.height = '420px'

    const background = new Rectangle('Background')
    container.addControl(background)
    background.cornerRadius = 12
    background.background = Theme.PRIMARY_COLOR
    background.color = '#FFFFFF'
    background.thickness = 4

    const title = new TextBlock('Title')
    container.addControl(title)
    title.text = 'Cut It'
    title.color = '#FFFFFF'
    title.fontSize = 48
    title.fontWeight = '600'
    title.height = '60px'
    title.verticalAlignment = Control.VERTICAL_ALIGNMENT_TOP
    title.top = '32px'

    const paragraph = new TextBlock('Paragraph')
    container.addControl(paragraph)
    paragraph.text = 'Welcome, Cut It is a application serving for cutting anything you need from an image. It combines two models to detect and classify object in an image, while offering a feature to creat image from text (may cost a few minutes). \n\nUpload your image or create one by the model to begin, try it now!'
    paragraph.textWrapping = true
    paragraph.color = '#FFFFFF'
    paragraph.fontSize = 24
    paragraph.fontWeight = '400'
    paragraph.height = '100%'
    paragraph.width = '750px'
    paragraph.textVerticalAlignment = Control.VERTICAL_ALIGNMENT_TOP
    paragraph.textHorizontalAlignment = Control.HORIZONTAL_ALIGNMENT_LEFT
    paragraph.verticalAlignment = Control.VERTICAL_ALIGNMENT_TOP
    paragraph.top = '118px'

    const buttonGroup = createButtonGroup(this.appUI)
    container.addControl(buttonGroup)
    buttonGroup.verticalAlignment = Control.VERTICAL_ALIGNMENT_BOTTOM
    buttonGroup.top = '-32px'
  }
}

class AppUIInputState extends BaseAppUIState {
  activate(appUI: AppUI): void {
    const container = new Container('InputUIContainer')
    appUI.ui.addControl(container)
    container.width = '1080px'
    container.height = '70px'
    container.verticalAlignment = Control.VERTICAL_ALIGNMENT_BOTTOM
    container.top = '-54px'

    const inputContainter = new Container('InputContainer')
    container.addControl(inputContainter)
    inputContainter.width = '666px'
    inputContainter.height = '70px'
    inputContainter.horizontalAlignment = Control.HORIZONTAL_ALIGNMENT_LEFT

    const inputBackground = new LoadingRectangle(appUI.scene, 'InputBackground')
    inputContainter.addControl(inputBackground)
    inputBackground.cornerRadius = 8
    inputBackground.background = '#F3F3F3'
    inputBackground.color = Theme.PRIMARY_COLOR
    inputBackground.thickness = 2

    const input = new InputText('Input')
    inputContainter.addControl(input)
    input.placeholderText = 'Describe object and hit "Enter" or text to image by clicking "Create Image"'
    input.placeholderColor = '#AFAFAF'
    input.width = '616px'
    input.color = '#3E3E3E'
    input.fontSize = '18px'
    input.fontWeight = '400'
    input.background = '#00000000'
    input.focusedBackground = '#00000000'
    input.thickness = 0
    input.horizontalAlignment = Control.HORIZONTAL_ALIGNMENT_CENTER
    input.onKeyboardEventProcessedObservable.add((event) => {
      if (!inputBackground.isLoading && event.key === 'Enter')
        appUI.observalbe.notifyObservers({ type: 'InputTextChange', target: input, value: input.text })
    })
    appUI.controls.push(input)

    const buttonGroup = createButtonGroup(this.appUI)
    container.addControl(buttonGroup)
    buttonGroup.horizontalAlignment = Control.HORIZONTAL_ALIGNMENT_RIGHT
  }
}

function createButtonGroup(appUI: AppUI) {
  const uploadImageButton = createUploadImageButton(appUI)
  const createImageButton = createCreateImageButton(appUI)
  const container = new Container('ButtonGroupContainer')
  container.addControl(uploadImageButton)
  container.addControl(createImageButton)
  container.adaptHeightToChildren = true
  container.widthInPixels = uploadImageButton.widthInPixels + createImageButton.widthInPixels + 12
  uploadImageButton.horizontalAlignment = Control.HORIZONTAL_ALIGNMENT_LEFT
  createImageButton.horizontalAlignment = Control.HORIZONTAL_ALIGNMENT_RIGHT
  return container
}

function createUploadImageButton(appUI: AppUI) {
  const button = Button.CreateSimpleButton('Button', 'Upload Image')
  button.width = '190px'
  button.height = '70px'
  button.color = '#FFFFFF'
  button.background = Theme.PRIMARY_COLOR
  button.fontSize = '24px'
  button.fontWeight = '500'
  button.cornerRadius = 8
  button.thickness = 2

  button.onPointerClickObservable.add(() => appUI.observalbe.notifyObservers({ type: 'UploadImageButtonClick', target: button, value: undefined }))
  appUI.controls.push(button)
  return button
}

function createCreateImageButton(appUI: AppUI) {
  const button = new LoadingButton(appUI.scene, 'Button', 'Create Image')
  button.width = '190px'
  button.height = '70px'
  button.color = '#EEEEEE'
  button.background = Theme.SECONDARY_COLOR
  button.fontSize = '24px'
  button.fontWeight = '500'
  button.cornerRadius = 8
  button.thickness = 2
  button.onPointerClickObservable.add(() => {
    if (button.isLoading)
      return
    const input = appUI.controls.find(c => c.name === 'Input') as InputText | undefined
    appUI.observalbe.notifyObservers({ type: 'CreateImageButtonClick', target: button, value: input?.text })
  })
  appUI.controls.push(button)
  return button
}
