import type { Nullable } from '@babylonjs/core'
import { Engine, FollowCamera, HemisphericLight, Observable, Scene, Vector3 } from '@babylonjs/core'
import { Inspector } from '@babylonjs/inspector'
import { EAppState, EAppUIState } from './enums'
import { error } from './utils'
import type { Robot } from './Robot'
import type { Image } from './Image'
import type { ILoadingRectangle } from './AppUI'

const ImageModule = import('./Image')
const GroundModule = import('./Ground')
const AppUIModule = import('./AppUI')
const RobotModule = import('./Robot')
export class App {
  engine: Engine
  scene?: Scene

  // App State
  stateObservable = new Observable<EAppState>()
  private _state: EAppState = EAppState.Initializing
  get state() {
    return this._state
  }

  set state(value) {
    this._state = value
    this.stateObservable.notifyObservers(value)
  }

  constructor(readonly canvas: HTMLCanvasElement) {
    this.state = EAppState.Initializing
    this.engine = new Engine(canvas)
    window.addEventListener('resize', () => {
      this.engine.resize()
    })
    this.engine.displayLoadingUI()
    this._createScene(this.engine, this.canvas).then(async (scene) => {
      this.scene = scene
      await this.scene.whenReadyAsync()
      this.state = EAppState.Initialized
      this.engine.hideLoadingUI()
    })
  }

  run() {
    if (import.meta.env.DEV) {
      // for development: make inspector visible/invisible
      window.addEventListener('keydown', (ev) => {
        // Shift+Ctrl+Alt+I
        if (ev.shiftKey && ev.ctrlKey && ev.altKey && ev.keyCode === 73) {
          if (Inspector.IsVisible)
            Inspector.Hide()
          else if (this.scene)
            Inspector.Show(this.scene, {})
        }
      })
    }
    this.engine.runRenderLoop(() => {
      this.scene?.render()
    })
  }

  private async _createScene(engine: Engine, canvas: HTMLCanvasElement) {
    const scene = new Scene(engine)
    const [{ Image }, { AppUI }, { Ground }, { Robot }] = await Promise.all([ImageModule, AppUIModule, GroundModule, RobotModule])
    const image = new Image()
    const ui = new AppUI(scene)
    const ground = Ground.create(scene, image)
    const robot = Robot.create(scene)
    const cammandRobotMoveToImage = async (robot: Robot, image: Image) => {
      if (!image.isClassified())
        return

      const bbox = image.classification.detection.box
      const groundBbox = ground.mesh.getBoundingInfo().boundingBox
      const groundSize = ground.mesh.getBoundingInfo().boundingBox.extendSize.scale(2)
      const imageSize = { x: image.image.imageWidth, y: 0, z: image.image.imageHeight }
      const scales = { x: groundSize.x / imageSize.x, y: 0, z: groundSize.z / imageSize.z }
      const bboxOrigin = new Vector3(groundBbox.minimum.x, 0, groundBbox.maximum.z)
      const bboxLeftTop = new Vector3(bbox.xmax * scales.x, 0, -bbox.ymax * scales.z)
      const bboxRightTop = new Vector3(bbox.xmin * scales.x, 0, -bbox.ymax * scales.z)
      const bboxRightBottom = new Vector3(bbox.xmin * scales.x, 0, -bbox.ymin * scales.z)
      const bboxLeftBottom = new Vector3(bbox.xmax * scales.x, 0, -bbox.ymin * scales.z)
      const destination = bboxOrigin.add(bboxLeftTop)
      await robot.moveTo(destination)
      await robot.laserCutter.cut([bboxLeftTop, bboxRightTop, bboxRightBottom, bboxLeftBottom, bboxLeftTop].map(v => bboxOrigin.add(v)))
      await robot.land()
    }

    ui.observalbe.add(async (event) => {
      try {
        if (event.type === 'UploadImageButtonClick') {
          image.clear()
          this.state = EAppState.ImageUploading
          await image.load()
          this.state = EAppState.ImageUploaded
          ui.setState(EAppUIState.Input)
          await robot.takeOff()
        }
        else if (event.type === 'CreateImageButtonClick') {
          image.clear()
          event.target.isLoading = true
          this.state = EAppState.ImageUploading
          await image.fromText(event.value)
            .finally(() => event.target.isLoading = false)
          this.state = EAppState.ImageUploaded
          ui.setState(EAppUIState.Input)
          await robot.takeOff()
        }
        else if (event.type === 'InputTextChange') {
          const takeoff = async () => {
            if (robot.pose === Robot.Pose.Land)
              await robot.takeOff()
          }
          const detect = async () => {
            this.state = EAppState.ImageDetecting
            await image.detect()
            this.state = EAppState.ImageDetected
          }
          const classify = async () => {
            this.state = EAppState.ImageClassifying
            await image.classify(event.target.text)
            this.state = EAppState.ImageClassified
          }
          takeoff()
          const background = event.target.parent?.getChildByName('InputBackground') as Nullable<ILoadingRectangle>
          background && (background.isLoading = true)
          try {
            await detect()
            await classify()
          }
          finally {
            background && (background.isLoading = false)
          }
          await cammandRobotMoveToImage(robot, image)
        }
      }
      catch (reason) {
        this.state = EAppState.Error
        error(reason)
      }
    })

    const camera = new FollowCamera('RobotCamera', new Vector3(0, 1, 0), scene, robot.mesh)
    camera.lowerHeightOffsetLimit = 0
    camera.maxCameraSpeed = 6
    camera.radius = 6
    camera.attachControl()

    const light = new HemisphericLight('light', new Vector3(0, 1, 0), scene)
    light.intensity = 0.7

    return scene
  }
}
