from typing import Annotated
from fastapi import FastAPI, UploadFile, File, Form, Response
from fastapi.staticfiles import StaticFiles
from fastapi.responses import FileResponse
from fastapi.middleware.cors import CORSMiddleware
from pydantic import BaseModel
import io
import json

# FastAPI tutorial
# https://fastapi.tiangolo.com/tutorial/
app = FastAPI()

# Allow CORS for localhost
origins = [
    "http://localhost:3000",
]

app.add_middleware(
    CORSMiddleware,
    allow_origins=origins,
    allow_credentials=True,
    allow_methods=["*"],
    allow_headers=["*"],
)

# Use a pipeline as a high-level helper
# https://huggingface.co/docs/transformers/main_classes/pipelines#transformers.ObjectDetectionPipeline
from transformers import pipeline, CLIPModel, CLIPProcessor
from PIL import Image
import numpy as np

detector = pipeline(model="facebook/detr-resnet-50")


@app.post("/detect-image")
async def detect_image(image: Annotated[UploadFile, File()]):
    return detector(Image.open(image.file))


# Text-to-image StableDiffusionPipeline
# https://huggingface.co/docs/diffusers/api/pipelines/stable_diffusion/text2img
from diffusers import DiffusionPipeline

text_to_image_creator = DiffusionPipeline.from_pretrained(
    "runwayml/stable-diffusion-v1-5"
)
text_to_image_creator.enable_attention_slicing()


class TextImageItem(BaseModel):
    prompt: str


@app.post("/text-to-image")
async def text_to_image(item: TextImageItem):
    image = text_to_image_creator(prompt=item.prompt, num_inference_steps=8).images[0]
    img_byte_arr = io.BytesIO()
    image.save(img_byte_arr, format="JPEG")
    return Response(img_byte_arr.getvalue(), media_type="image/jpeg")


# Load model directly
# https://huggingface.co/docs/transformers/main/en/model_doc/clip#transformers.CLIPProcessor
# https://huggingface.co/docs/transformers/main/en/model_doc/clip#transformers.CLIPModel
classifier_processor = CLIPProcessor.from_pretrained("openai/clip-vit-base-patch32")
classifier_model = CLIPModel.from_pretrained("openai/clip-vit-base-patch32")


@app.post("/classify-image")
async def classify_image(
    image: Annotated[UploadFile, File()],
    candidate_label: Annotated[str, Form()],
    detections: Annotated[str, Form()],
):
    _dectections = json.loads(detections)
    _image = Image.open(image.file)

    # Copy from space
    # https://huggingface.co/spaces/vishnun/CLIPnCROP/blob/main/app.py
    images_list = []
    for detection in _dectections:
        box = detection["box"]
        im_arr = np.array(_image)
        roi = im_arr[box["ymin"] : box["ymax"], box["xmin"] : box["xmax"]]
        roi_im = Image.fromarray(roi)
        images_list.append(roi_im)

    _input = classifier_processor(
        text=[candidate_label], images=images_list, return_tensors="pt", padding=True
    )
    _output = classifier_model(**_input)
    logits_per_image = _output.logits_per_text
    probs = logits_per_image.softmax(-1)
    l_idx = np.argsort(probs[-1].detach().numpy())[::-1][0:1]

    final_ims = []
    for i, j in enumerate(_dectections):
        json_dict = {}
        if i in l_idx:
            json_dict["detection"] = _dectections[i]
            json_dict["score"] = probs[-1].detach().numpy()[i].astype(float)
            final_ims.append(json_dict)

    fi = sorted(final_ims, key=lambda item: item.get("score"), reverse=True)
    return fi[0]


# Serve static resources
app.mount("/", StaticFiles(directory="dist", html=True), name="dist")


@app.get("/")
def index() -> FileResponse:
    return FileResponse(path="/app/dist/index.html", media_type="text/html")
