from shiny import reactive, render, ui, module
from shinywidgets import output_widget, render_widget

import plotly.graph_objects as go
import plotly.express as px
from plotly.subplots import make_subplots

from pathlib import Path
from PIL import Image

import numpy as np
import pandas as pd
import math as math
import re
import matplotlib.pyplot as plt

# Importar los algoritmos y modelss desde scikit learn:
from sklearn.model_selection import train_test_split
from sklearn.tree import DecisionTreeClassifier, plot_tree
from sklearn.ensemble import RandomForestClassifier
from sklearn.linear_model import LogisticRegression
from sklearn import metrics

# Importar todos los paquetes de información, ui y warnings generados
from GeneralInfo.presentation import presentation_ui, presentation_server
from GeneralInfo.initialData import initial_data_1_ui, initial_data_2_ui
from GeneralInfo.observation import observation_1_ui
from GeneralInfo.cleaning import cleaning_1_ui, cleaning_2_ui, cleaning_extra_ui, cleaning_server
from GeneralInfo.correlation import correlation_1_ui, correlation_2_ui, correlation_server
from GeneralInfo.subsetSplit import subsetSplit_1_ui, subsetSplit_2_ui, subsetSplit_server

from AlgorithmsInfo.decTreeInfo import decTree_def_ui, decTree_howTo_ui, decTree_performance_ui, decTree_server
from AlgorithmsInfo.sharedInfo import shared_varImp_ui, shared_conf_mat_metrics_ui, shared_fitting_ui, shared_tree_rep_info_ui, shared_algs_server
from AlgorithmsInfo.ranForestInfo import ranForest_def_ui, ranForest_howTo_ui, ranForest_performance_ui, ranForest_server
from AlgorithmsInfo.logRegInfo import logReg_def_ui, logReg_howTo_ui, logReg_performance_ui, logReg_server

from UIMessages.warningsGeneral import correlation_warning_ui, warnings_general_server
from UIMessages.warningsAlgorithms import diagnosis_warning_ui, test_split_warning_ui, test_split_low_warning_ui, test_split_high_warning_ui, features_warning_ui, feat_imp_warning_ui, conf_matrix_warning_ui, warnings_algorithms_server



################################## DATAFRAMES Y DICCIONARIOS #################################

# Datos del dataframe de Breast Cancer Data Winsconsin
infile = Path(__file__).parent / "data.csv"
original_df = pd.read_csv( infile )
clean_df = original_df.copy()

empty_column_dict = {}
all_column_dict = {}
for col in clean_df.columns:
    if col != "diagnosis":
        all_column_dict[col] = col
outcome_var='diagnosis'

# Datos para Decision tree
dec_tree_feat_imp_df = pd.DataFrame()

# Datos para Random Forest
ran_forest_feat_imp_df = pd.DataFrame()

# Datos para Logistic regression
log_reg_feat_imp_df = pd.DataFrame()

# Variable para visualizar subplots
subplot_cols_number = 4

# Paths para guardado de imágenes
decTree_image_folder = Path(Path(__file__).parent / 'DecTrees')
ranForest_image_folder = Path(Path(__file__).parent / 'RanForests')


##############################################################################################
##############################################################################################
####################################### MÓDULO DE UI #########################################
##############################################################################################
##############################################################################################

@module.ui
def learningTool_ui():
    return ui.div(
        ui.panel_main(
############################# TÍTULO - PRESENTACIÓN ###########################
            ui.panel_main(
                {"id": "tool-intro"},
                ui.tags.h1("PRESENTACIÓN A LA HERRAMIENTA"),
                presentation_ui("Presentation_Main"),
                width=12
            ),
            ui.panel_main(
                ui.tags.p(style="padding-bottom:20px;"),
                ui.tags.hr(),
                width=12
            ),

#################################### DATOS INICIALES ###########################
            ui.panel_main(
                ui.tags.h3("NUESTROS DATOS"),
                initial_data_1_ui("initial_data_1"),
                width=12
            ),
            ui.panel_main(
                {"id": "original-table"},
                ui.div(
                    ui.input_switch("view_original_table", "¡Ver los datos originales!", width="50%"),
                    style="font-weight: bold;"
                ),
                width=12
            ),
            ui.panel_main(
                {"id": "original-table-types"},
                initial_data_2_ui("initial_data_2"),
                ui.div(
                    ui.input_switch("view_original_table_types", "¡Ver los tipos de los datos originales!", width="50%"),
                    style="font-weight: bold;"
                ),
                width=12, style="padding-top:20px;"
            ),
            ui.panel_main(
                ui.tags.hr(),
                width=12, style="padding-top:10px;"
            ),

#################################### OBSERVACIÓN DE DATOS ###########################
            ui.panel_main(
                ui.tags.h3("OBSERVACIÓN DE DATOS"),
                observation_1_ui("observation_1"),
                width=12
            ),
            ui.panel_main(
                {"id": "data-observation-diagnosis"},
                ui.div(
                    ui.input_switch("view_diganosis_hist", "¡Ver la distribución de los datos según la variable diagnosis!", width="50%"),
                    style="font-weight: bold;"
                ),
                width=12
            ),
            ui.panel_main(
                ui.tags.p(style="padding-bottom:10px;"),
                width=12
            ),
            ui.panel_main(
                {"id": "data-observation-general"},
                ui.input_select("dropKeyWordSelector", "Selecciona un grupo de características según su tipo de medida:", {"mean": "Grupo de medida de media (mean)", "worst": "Grupo de medida de peor/mayor (worst)", "se": "Grupo de medida de error estándar (se)"}, width="50%"),
                ui.div(
                    ui.input_switch("view_general_hist", "¡Ver el grupo de características seleccionado en función de la variable de diagnosis!", width="70%"),
                    style="font-weight: bold;"
                ),
                width=12,
            ),
            ui.panel_main(
                ui.tags.hr(),
                width=12, style="padding-top:10px;"
            ),

#################################### LIMPIEZA DE DATOS ###########################
            ui.panel_main(
                ui.tags.h3("LIMPIEZA DE DATOS"),
                cleaning_1_ui("cleaning_1"),
                width=12
            ),
            ui.panel_main(
                {"id": "clean-convert-diagnosis"},
                ui.input_action_button("convert_diagnosis", "Transformar datos de la variable diagnosis"),
                cleaning_2_ui("cleaning_2"),
                ui.input_select("dropIdSelector", "Selecciona la columna a eliminar:", all_column_dict, width="40%"),
                width=12
            ),
            ui.panel_main(
                {"id": "clean-drop-index"},
                ui.input_action_button("drop_selected_column_clean", "Eliminar columna seleccionada"),
                width=12
            ),
            ui.panel_main(
                {"id": "clean-table"},
                ui.tags.p("Por último, podemos observar como han quedado nuestros datos tras realizar la limpieza."
                            , style="padding-right:50px; padding-top:20px; text-align: justify; text-justify: inter-word;"),
                ui.div(
                    ui.input_switch("view_clean_table", "¡Ver los datos limpios!", width="50%"),
                    style="font-weight: bold;"
                ),
                width=12, style="padding-top:30px;"
            ),
            ui.panel_main(
                cleaning_extra_ui("cleaning_extra"),
                width=12, style="padding-top:10px;"
            ),
            ui.panel_main(
                ui.tags.p(style="padding-bottom:20px;"),
                ui.tags.hr(),
                width=12
            ),

#################################### CORRELACIÓN ###########################
            ui.panel_main(
                correlation_1_ui("correlation_1"),
                width=12
            ),
            ui.panel_main(
                correlation_2_ui("correlation_2"),
                width=12
            ),
            ui.panel_main(
                {"id": "correlation"},
                ui.input_slider("maximum_correlation", "Máxima correlación:", min=0, max=1, value=0.7, step=0.01),
                ui.input_action_button("drop_correlation", "Eliminar columnas con correlación superior a la seleccionada"),
                ui.tags.p(style="padding-top:20px;"),
                ui.div(
                    ui.input_switch("view_correlation", "¡Ver la correlación entre datos!", width="50%"),
                    style="font-weight: bold;"
                ),
                width=12
            ),
            ui.panel_main(
                ui.tags.hr(),
                width=12, style="padding-top:10px;"
            ),


#################################### TABS DE ALGORITMOS ###########################
            ui.panel_main(
                ui.tags.h3("ALGORITMOS DE PREDICCIÓN"),
                width=12, style="padding-bottom:10px;"
            ),
            ui.panel_main(
                ui.tags.h5("División de los datos en conjuntos de entrenamiento y prueba"),
                subsetSplit_1_ui("subset_1"),
                width=12,
            ),
            ui.panel_main(
                {"id": "test_split"},
                subsetSplit_2_ui("subset_2"),
                ui.input_slider("test_split_value", "Tamaño del subconjunto de prueba:", min=0, max=1, value=0.2, step=0.01),
                ui.input_action_button("make_test_split", "Divide los datos en subconjunto de entrenamiento y testeo"),
                ui.tags.p("", style="padding-bottom:30px;"),
                ui.tags.hr(),
                width=12, style="padding-bottom:10px;"
            ),
            ui.navset_tab(
                ##############################################################
                ################ ÁRBOL DE DECISIÓN ###########################
                ##############################################################
                ui.nav(
                    "Árbol de decisión",
                    ui.tags.h3("Árbol de decisión", style="padding-top:20px;"),
                    ui.panel_main(
                        ui.tags.h5("¿Qué es un arbol de decisión?", style="padding-top:20px; padding-bottom:10px;"),
                        decTree_def_ui("decTree_1"),
                        width=12
                    ),
                    ui.panel_main(
                        decTree_howTo_ui("decTree_1"),
                        width=12
                    ),
                    ui.panel_main(
                        ui.tags.p(style="padding-bottom:20px;"),
                        width=12
                    ),
                ######### AD: AJUSTES, CARACTERÍSTICAS Y CREACIÓN #########
                    ui.row(
                        ui.column(
                            3,
                            ui.panel_well(
                                ui.tags.h5("Ajustes:"),
                                ui.tags.hr(),
                                ui.input_select("dec_tree_criterion","Criterion", {"gini": "Gini (default)", "entropy": "Entropy", "log_loss": "Log_loss"}),
                                ui.input_select("dec_tree_splitter","Splitter", {"best": "Best (default)", "random": "Random"}),
                                ui.input_slider("dec_tree_max_depth", "Max Depth (0 = None / default)", 0, 32, 0, step=1),
                                ui.input_slider("dec_tree_min_samples_split", "Min samples split (default = 2)", 1, 6, 2, step=1),
                                ui.input_slider("dec_tree_min_samples_leaf", "Min samples leaf (default = 1)", 1, 5, 1, step=1),
                                ui.input_select("dec_tree_max_features","Max features", {"None": "None (default)", "sqrt": "Sqrt", "log2": "Log2"}),
                            ),
                        ),
                        ui.column(
                            3,
                            ui.panel_well(
                                ui.tags.h5("Características:"),
                                ui.tags.hr(),
                                ui.input_checkbox_group("dec_tree_features_sel", "", choices=all_column_dict, selected=list(all_column_dict)),
                            ),
                        ),
                        ui.column(
                            6,
                            ui.panel_main(
                                {"id": "dec_tree_generator"},
                                ui.tags.h5("¡Crea el modelo de predicción!", style="padding-bottom:10px;"),
                                ui.input_action_button("generate_decission_tree", "Generar el modelo de árbol de decisión"),
                                width=12
                            ),
                            ui.panel_main(
                                {"id": "var_imp_dec_tree"},
                                ui.tags.hr(),
                                ui.tags.h5("Importancia de las características para el modelo:"),
                                shared_varImp_ui("decTree_shared_1"),
                                ui.div(
                                    ui.input_switch("view_variable_importance_dec_tree", "¡Ver la importancia de las características!", width="80%"),
                                    style="font-weight: bold;"
                                ),
                                width=12
                            ),
                            ui.panel_main(
                                {"id": "var_imp_slider_dec_tree"},
                                ui.input_slider("minimum_importance_dec_tree", "Mínima importancia:", min=0, max=100, value=5.0, step=0.1),
                                ui.input_action_button("deselect_not_imp_vars_dec_tree", "Deseleccionar características poco importantes automáticamente"),
                                width=12
                            ),
                        ),
                    ),
                    ui.panel_main(
                        ui.tags.p(style="padding-bottom:20px;"),
                        ui.tags.hr(),
                        width=12
                    ),
                ######### AD: MATRIZ DE CONFUSIÓN #########
                    ui.panel_main(
                        ui.tags.h5("Resultados del modelo: matriz de confusión y métricas básicas"),
                        shared_conf_mat_metrics_ui("decTree_shared_1", label="decTree"),
                        width=12
                    ),
                    ui.panel_main(
                        {"id": "dec_tree_conf_matrix"},
                        ui.tags.p("Puedes ver las matrices de confusión del modelo generado y sus métricas aquí:"
                                    , style="padding-top:20px; padding-right:50px; text-align: justify; text-justify: inter-word;"),
                        ui.div(
                            ui.input_switch("conf_mat_dec_tree_switch", "¡Ver la matriz de confusión del árbol de decisión generado!", width="60%"),
                            style="font-weight: bold;"
                        ),
                        width=12
                    ),
                    ui.row(
                        ui.column(6,
                            ui.panel_main(
                                {"id": "dec_tree_conf_matrix_train"},
                                width=12
                            ),
                        ),
                        ui.column(6,
                            ui.panel_main(
                                {"id": "dec_tree_conf_matrix_test"},
                                width=12
                            ),
                        ),
                    ),
                ######### AD: RESULTADOS CON ENTRENAMIENTO Y TEST #########
                    ui.row(
                        ui.column(6,
                            ui.tags.p("Resultados con los datos de entrenamiento:", style="font-weight: bold;"),
                            ui.panel_main(
                                ui.output_text_verbatim("decision_tree_precision"),
                                ui.output_text_verbatim("decision_tree_recall"),
                                ui.output_text_verbatim("decision_tree_f1"),
                                ui.output_text_verbatim("decision_tree_accuracy"),
                                width=7
                            ),
                        ),
                        ui.column(6,
                            ui.tags.p("Resultados con los datos de prueba:", style="font-weight: bold;"),
                            ui.panel_main(
                                ui.output_text_verbatim("decision_tree_precision_test"),
                                ui.output_text_verbatim("decision_tree_recall_test"),
                                ui.output_text_verbatim("decision_tree_f1_test"),
                                ui.output_text_verbatim("decision_tree_accuracy_test"),
                                width=7
                            ),
                        ),
                        style="padding-top:30px;"
                    ),
                    decTree_performance_ui("decTree_3"),
                    ui.panel_main(
                        shared_fitting_ui("decTree_shared_1", label="decTree"),
                        width=12
                    ),                        
                    ui.panel_main(
                        ui.tags.p(style="padding-bottom:20px;"),
                        ui.tags.hr(),
                        width=12
                    ),
                ########## AD: REPRESENTACIÓN ÁRBOL ##########
                    ui.panel_main(
                        {"id": "dec_tree_view"},
                        ui.tags.h5("Representación del árbol de decisión"),
                        ui.tags.p("Finalmente podemos ver la representación del árbol de decisión. Gracias a ella podemos ver como nuestro modelo elige si una instancia pertenece a una clase o a otra."
                                    , style="padding-top:10px; padding-bottom:10px; padding-right:50px; text-align: justify; text-justify: inter-word;"),
                        ui.div(
                            ui.input_switch("view_tree_dec_tree_switch", "¡Ver la representación del árbol de decisión generado!", width="60%"),
                            style="font-weight: bold;"
                        ),
                        width=12,  style="padding-bottom:10px;"
                    ),
                    ui.panel_main(
                        shared_tree_rep_info_ui("decTree_shared_1", label="decTree"),
                        width=12, style="padding-bottom:40px;"
                    ),
                ),
                ##############################################################
                ################ RANDOM FOREST ###############################
                ##############################################################                    
                ui.nav("Bosque aleatorio",
                    ui.tags.h3("Bosque aleatorio", style="padding-top:20px;"),
                    ui.panel_main(
                        ui.tags.h5("¿Qué es un bosque aleatorio?", style="padding-top:20px; padding-bottom:10px;"),
                        ranForest_def_ui("ranForest_1"),
                        width=12
                    ),
                    ui.panel_main(
                        ranForest_howTo_ui("ranForest_1"),
                        width=12
                    ),
                    ui.panel_main(
                        ui.tags.p(style="padding-bottom:20px;"),
                        width=12
                    ),
                ######### RF: AJUSTES, CARACTERÍSTICAS Y CREACIÓN #########
                    ui.row(
                        ui.column(
                            3,
                            ui.panel_well(
                                ui.tags.h5("Ajustes:"),
                                ui.tags.hr(),
                                ui.input_slider("ran_forest_n_estimators", "Num Estimators (default = 100)", 1, 100, 10, step=1),
                                ui.input_select("ran_forest_criterion","Criterion", {"gini": "Gini (default)", "entropy": "Entropy", "log_loss": "Log_loss"}),
                                ui.input_slider("ran_forest_max_depth", "Max Depth (0 = None / default)", 0, 32, 0, step=1),
                                ui.input_slider("ran_forest_min_samples_split", "Min samples split (default = 2)", 1, 6, 2, step=1),
                                ui.input_slider("ran_forest_min_samples_leaf", "Min samples leaf (default = 1)", 1, 5, 1, step=1),
                                ui.input_select("ran_forest_max_features","Max features", {"None": "None (default)", "sqrt": "Sqrt", "log2": "Log2"}),
                            ),
                        ),
                        ui.column(
                            3,
                            ui.panel_well(
                                ui.tags.h5("Características:"),
                                ui.tags.hr(),
                                ui.input_checkbox_group("ran_forest_features_sel", "", choices=all_column_dict, selected=list(all_column_dict)),
                            ),
                        ),
                        ui.column(
                            6,
                            ui.panel_main(
                                {"id": "ran_forest_generator"},
                                ui.tags.h5("¡Crea el modelo de predicción!"),
                                ui.input_action_button("generate_random_forest", "Generar el modelo de bosque aleatorio"),
                                width=12
                            ),
                            ui.panel_main(
                                {"id": "var_imp_ran_forest"},
                                ui.tags.hr(),
                                ui.tags.h5("Importancia de las características para el modelo:"),
                                shared_varImp_ui("ranForest_shared_1"),
                                ui.div(
                                    ui.input_switch("view_variable_importance_ran_forest", "¡Ver la importancia de las características!", width="80%"),
                                    style="font-weight: bold;"
                                ),
                                width=12
                            ),
                            ui.panel_main(
                                {"id": "var_imp_slider_ran_forest"},
                                ui.input_slider("minimum_importance_ran_forest", "Mínima importancia:", min=0, max=100, value=5.0, step=0.1),
                                ui.input_action_button("deselect_not_imp_vars_ran_forest", "Deseleccionar características poco importantes automaticamente!"),
                                width=12
                            ),
                        ),
                    ),
                    ui.panel_main(
                        ui.tags.p(style="padding-bottom:20px;"),
                        ui.tags.hr(),
                        width=12
                    ),
                ######### RF: MATRIZ DE CONFUSIÓN #########
                    ui.panel_main(
                        ui.tags.h5("Resultados del modelo: matriz de confusión y métricas básicas"),
                        shared_conf_mat_metrics_ui("ranForest_shared_1", label="ranForest"),
                        width=12
                    ),
                    ui.panel_main(
                        {"id": "ran_forest_conf_matrix"},
                        ui.tags.p("Puedes ver las matrices de confusión del modelo generado y sus métricas aquí:"
                                    , style="padding-top:20px; padding-right:50px; text-align: justify; text-justify: inter-word;"),
                        ui.div(
                            ui.input_switch("conf_mat_ran_forest_switch", "¡Ver la matriz de confusión del bosque aleatorio generado!", width="60%"),
                            style="font-weight: bold;"
                        ),
                        width=12
                    ),
                    ui.row(
                        ui.column(6,
                            ui.panel_main(
                                {"id": "ran_forest_conf_matrix_train"},
                                width=12
                            ),
                        ),
                        ui.column(6,
                            ui.panel_main(
                                {"id": "ran_forest_conf_matrix_test"},
                                width=12
                            ),
                        ),
                    ),
                ######### RF: RESULTADOS CON ENTRENAMIENTO Y TEST #########
                    ui.row(
                        ui.column(6,
                            ui.tags.p("Resultados con los datos de entrenamiento:", style="font-weight: bold;"),
                            ui.panel_main(
                                ui.output_text_verbatim("random_forest_precision"),
                                ui.output_text_verbatim("random_forest_recall"),
                                ui.output_text_verbatim("random_forest_f1"),
                                ui.output_text_verbatim("random_forest_accuracy"),
                                width=7
                            ),
                        ),
                        ui.column(6,
                            ui.tags.p("Resultados con los datos de prueba:", style="font-weight: bold;"),
                            ui.panel_main(
                                ui.output_text_verbatim("random_forest_precision_test"),
                                ui.output_text_verbatim("random_forest_recall_test"),
                                ui.output_text_verbatim("random_forest_f1_test"),
                                ui.output_text_verbatim("random_forest_accuracy_test"),
                                width=7
                            ),
                        ),
                        style="padding-top:30px;"
                    ),
                    ranForest_performance_ui("ranForest_3"),
                    ui.panel_main(
                        shared_fitting_ui("ranForest_shared_1", label="ranForest"),
                        width=12
                    ),                        
                    ui.panel_main(
                        ui.tags.p(style="padding-bottom:20px;"),
                        ui.tags.hr(),
                        width=12
                    ),
                ########## RF: REPRESENTACIÓN ÁRBOL ##########
                    ui.panel_main(
                        {"id": "ran_forest_view"},
                        ui.tags.h5("Representación de los árboles"),
                        ui.tags.p("Un bosque aleatorio está formado por múltiples árboles de decisión. A continuación puedes ver los 5 primeros como máximo."),
                        ui.input_select("view_tree_ran_forest_number", "Selecciona el árbol que quieres mostrar", empty_column_dict),
                        ui.div(
                            ui.input_switch("view_tree_ran_forest_switch", "¡Ver la representación de los árboles de decisión generados!", width="60%"),
                            style="font-weight: bold;"
                        ),
                        width=12, style="padding-bottom:10px;"
                    ),
                    ui.panel_main(
                        shared_tree_rep_info_ui("ranForest_shared_1", label="ranForest"),
                        width=12, style="padding-bottom:40px;"
                    ),
                ),
                ##############################################################
                ################ REGRESIÓN LOGÍSTICA #########################
                ##############################################################
                ui.nav("Regresión logística",
                    ui.tags.h3("Regresión logística", style="padding-top:20px;"),
                    ui.panel_main(
                        ui.tags.h5("¿Qué es una regresión logística?", style="padding-top:20px; padding-bottom:10px;"),
                        logReg_def_ui("logReg_1"),
                        width=12
                    ),
                    ui.panel_main(
                        logReg_howTo_ui("logReg_1"),
                        width=12
                    ),
                    ui.panel_main(
                        ui.tags.p(style="padding-bottom:20px;"),
                        width=12
                    ),
                ######### RL: AJUSTES, CARACTERÍSTICAS Y CREACIÓN #########
                    ui.row(
                        ui.column(
                            3,
                            ui.panel_well(
                                ui.tags.h5("Ajustes:"),
                                ui.tags.hr(),
                                ui.input_select("log_reg_solver","Solver", {"lbfgs": "Lbfgs (default)", "liblinear": "Liblinear", "newton-cg": "Newton-cg", "newton-cholesky": "Newton-cholesky", "sag": "Sag", "saga": "Saga"}, selected="lbfgs"),
                                ui.input_select("log_reg_penalty","Penalty", {"l2": "L2 (default)", "None": "None"}, selected="l2"),
                                ui.input_slider("log_reg_tol", "Tolerance (default = 1e-4) - 1e(valor seleccionado)", -10, 0, -4, step=1),
                                ui.input_slider("log_reg_c", "C (default = 1)", 1, 3000, 1, step=1),
                                ui.input_slider("log_reg_max_iter", "Max iterations (default = 100)", 100, 5000, 100, step=10),
                            ),
                        ),
                        ui.column(
                            3,
                            ui.panel_well(
                                ui.tags.h5("Características:"),
                                ui.tags.hr(),
                                ui.input_checkbox_group("log_reg_features_sel", "", choices=all_column_dict, selected=list(all_column_dict)),
                            ),
                        ),
                        ui.column(
                            6,
                            ui.panel_main(
                                {"id": "log_reg_generator"},
                                ui.tags.h5("¡Crea el modelo de predicción!"),
                                ui.input_action_button("generate_logistic_regression", "Generar el modelo de Regresión logística"),
                                width=12
                            ),
                            ui.panel_main(
                                {"id": "var_imp_log_reg"},
                                ui.tags.hr(),
                                ui.tags.h5("Importancia de las características para el modelo:"),
                                shared_varImp_ui("logReg_shared_1"),
                                ui.div(
                                    ui.input_switch("view_variable_importance_log_reg", "¡Ver la importancia de las características!", width="80%"),
                                    style="font-weight: bold;"
                                ),
                                width=12
                            ),
                            ui.panel_main(
                                {"id": "var_imp_slider_log_reg"},
                                ui.input_slider("minimum_importance_log_reg", "Mínima importancia:", min=0, max=100, value=5.0, step=0.1),
                                ui.input_action_button("deselect_not_imp_vars_log_reg", "Deseleccionar características poco importantes automaticamente"),
                                width=12
                            ),
                        ),
                    ),
                    ui.panel_main(
                        ui.tags.p(style="padding-bottom: 20px"),
                        ui.tags.hr(),
                        width=12
                    ),
                ######### RL: MATRIZ DE CONFUSIÓN #########
                    ui.panel_main(
                        ui.tags.h5("Resultados del modelo: matriz de confusión y métricas básicas"),
                        shared_conf_mat_metrics_ui("logReg_shared_1", label="logReg"),
                        width=12
                    ),
                    ui.panel_main(
                        {"id": "log_reg_conf_matrix"},
                        ui.tags.p("Puedes ver las matrices de confusión del modelo generado y sus métricas aquí:"
                                    , style="padding-top:20px; padding-right:50px; text-align: justify; text-justify: inter-word;"),
                        ui.div(
                            ui.input_switch("conf_mat_log_reg_switch", "¡Ver la matriz de confusión de la regresión logística generada!", width="60%"),
                            style="font-weight: bold;"
                        ),
                        width=12
                    ),
                    ui.row(
                        ui.column(6,
                            ui.panel_main(
                                {"id": "log_reg_conf_matrix_train"},
                                width=12
                            ),
                        ),
                        ui.column(6,
                            ui.panel_main(
                                {"id": "log_reg_conf_matrix_test"},
                                width=12
                            ),
                        ),
                    ),
                ######### RL: RESULTADOS CON ENTRENAMIENTO Y TEST #########
                    ui.row(
                        ui.column(6,
                            ui.tags.p("Resultados con los datos de entrenamiento:", style="font-weight: bold;"),
                            ui.panel_main(
                                ui.output_text_verbatim("logistic_regression_precision"),
                                ui.output_text_verbatim("logistic_regression_recall"),
                                ui.output_text_verbatim("logistic_regression_f1"),
                                ui.output_text_verbatim("logistic_regression_accuracy"),
                                width=7
                            ),
                        ),
                        ui.column(6,
                            ui.tags.p("Resultados con los datos de prueba:", style="font-weight: bold;"),
                            ui.panel_main(
                                ui.output_text_verbatim("logistic_regression_precision_test"),
                                ui.output_text_verbatim("logistic_regression_recall_test"),
                                ui.output_text_verbatim("logistic_regression_f1_test"),
                                ui.output_text_verbatim("logistic_regression_accuracy_test"),
                                width=7
                            ),
                        ),
                        style="padding-top:30px;"
                    ),
                    logReg_performance_ui("logReg_3"),
                    ui.panel_main(
                        shared_fitting_ui("logReg_shared_1", label="logReg"),
                        width=12
                    ),                        
                    ui.panel_main(
                        ui.tags.p(style="padding-bottom:40px;"),
                        width=12
                    ),
                ),
            ),

####################### RESET DATAFRAME LIMPIA #############################
            ui.panel_main(
                ui.tags.hr(),
                ui.tags.h3("¿LA BASE DE DATOS LIMPIA YA NO SIRVE?", style="padding-top:10px;"),
                ui.tags.p("Si has eliminado demasiadas columnas, te has equivocado o simplemente quieres volver a empezar... ¡Restablece aquí la base de datos limpia!", style="padding-top:10px;"),
                ui.input_action_button("reset_clean_df", "Restablece los datos limpios"),
                width=12, style="padding-top:10px; padding-bottom:50px;"
            ),
            width=12
        ),
    )



##############################################################################################
##############################################################################################
#################################### MÓDULO DE SERVIDOR ######################################
##############################################################################################
##############################################################################################

@module.server
def learningTool_server(input, output, session):
  
################# VARIABLES REACTIVAS Y DE CONTROL DE LA HERRAMIENTA #########################
    
    #Controles generales:
    diagnosis_data_converted = reactive.Value(False)
    correlation_execution_counter = reactive.Value(0)
    test_split_done = reactive.Value(False)
    reset_dataframe_counter = reactive.Value(0)
    
    #Decision Tree:
    decision_tree_execution_counter = reactive.Value(0)

    accuracy_decTree = reactive.Value(-1)
    recall_decTree = reactive.Value(-1)
    precision_decTree = reactive.Value(-1)
    f1_decTree = reactive.Value(-1)

    accuracy_decTree_test = reactive.Value(-1)
    recall_decTree_test = reactive.Value(-1)
    precision_decTree_test = reactive.Value(-1)
    f1_decTree_test = reactive.Value(-1)

    tree_plot_x_coords = reactive.Value()
    tree_plot_y_coords = reactive.Value()
    tree_plot_texts = reactive.Value()

    tree_conf_mat_train = reactive.Value()
    tree_conf_mat_test = reactive.Value()

    #Random Forest:
    random_forest_execution_counter = reactive.Value(0)
    random_forest_last_estimators_num = reactive.Value(0)

    accuracy_ranForest = reactive.Value(-1)
    recall_ranForest = reactive.Value(-1)
    precision_ranForest = reactive.Value(-1)
    f1_ranForest = reactive.Value(-1)

    accuracy_ranForest_test = reactive.Value(-1)
    recall_ranForest_test = reactive.Value(-1)
    precision_ranForest_test = reactive.Value(-1)
    f1_ranForest_test = reactive.Value(-1)

    ranForest_tree_plot_x_coords = reactive.Value()
    ranForest_tree_plot_y_coords = reactive.Value()
    ranForest_tree_plot_texts = reactive.Value()

    ranForest_tree_conf_mat_train = reactive.Value()
    ranForest_tree_conf_mat_test = reactive.Value()

    #Logistic regression:
    logistic_regression_execution_counter = reactive.Value(0)

    accuracy_logReg = reactive.Value(-1)
    recall_logReg = reactive.Value(-1)
    precision_logReg = reactive.Value(-1)
    f1_logReg = reactive.Value(-1)

    accuracy_logReg_test = reactive.Value(-1)
    recall_logReg_test = reactive.Value(-1)
    precision_logReg_test = reactive.Value(-1)
    f1_logReg_test = reactive.Value(-1)

    logReg_conf_mat_train = reactive.Value()
    logReg_conf_mat_test = reactive.Value()


################# MODULOS DE SERVIDORES AUXILIARES DE LA HERRAMIENTA #########################

    presentation_server("Presentation_Main")
    cleaning_server("cleaning_extra")
    correlation_server("correlation_1")
    subsetSplit_server("subset_1")
    decTree_server("decTree_1")
    ranForest_server("ranForest_1")
    logReg_server("logReg_1")
    shared_algs_server("decTree_shared_1", label="decTree")
    shared_algs_server("ranForest_shared_1", label="ranForest")
    shared_algs_server("logReg_shared_1", label="logReg")
    warnings_general_server("general_warnings")
    warnings_algorithms_server("dec_tree_warnings")
    warnings_algorithms_server("ran_forest_warnings")
    warnings_algorithms_server("log_reg_warnings")



##############################################################################################
#################################### DATOS INICIALES #########################################
##############################################################################################

#################################### TABLAS ##################################################
    @output
    @render.table
    def originalTable():
        return original_df
    
    @output
    @render.table
    def originalTableTypes():
        original_table_types = original_df.dtypes.to_frame().reset_index().transpose().reset_index(drop=True)
        headers = original_table_types.iloc[0]
        original_table_types  = pd.DataFrame(original_table_types.values[1:], columns=headers)
        original_table_types = original_table_types.replace(['int64', 'float64', 'object'],['numérico', 'numérico', 'categórico'])
        return original_table_types
    
#################################### EFECTOS REACTIVOS #######################################
    
    # VER TABLA ORIGINAL:
    @reactive.Effect
    def _():
        original_table_switch = input.view_original_table()
        if original_table_switch == True:
            original_table = ui.output_table("originalTable", style = "overflow-x:scroll; height:260px; overflow-y:auto;"),
            ui.insert_ui(
                ui.div({"id": "inserted-original-table"}, original_table, style = "width:100%; overflow-x:auto;"),
                selector="#original-table",
                where="beforeEnd",
            )
        else:
            ui.remove_ui("#inserted-original-table")
    
    # VER TIPOS DE DATOS ORIGINALES
    @reactive.Effect
    def _():
        original_table_types_switch = input.view_original_table_types()
        if original_table_types_switch == True:
            original_table_types = ui.output_table("originalTableTypes", style = "overflow-x:auto;"),
            ui.insert_ui(
                ui.div({"id": "inserted-original-table-types"}, original_table_types),
                selector="#original-table-types",
                where="beforeEnd",
            )
        else:
            ui.remove_ui("#inserted-original-table-types")


##############################################################################################
#################################### OBSERVACIÓN DE DATOS ####################################
##############################################################################################

#################################### EFECTOS REACTIVOS #######################################

    @reactive.Effect
    def _():
        diagnosis_data_converted.get()
        diganosis_hist_switch = input.view_diganosis_hist()
        if diganosis_hist_switch == True:
            ui.remove_ui("#diagnosis-hist-plot")
            diganosis_hist_plot = output_widget("widget_diagnosisObservation")
            ui.insert_ui(
                ui.div({"id": "diagnosis-hist-plot"}, diganosis_hist_plot, style = "width:50%; height: 180px; overflow-x:auto;"),
                selector="#data-observation-diagnosis",
                where="beforeEnd",
            )
        else:
            ui.remove_ui("#diagnosis-hist-plot")
    
    @reactive.Effect
    def _():
        # Elementos a los que reaccionar:
        input.dropKeyWordSelector()
        input.drop_selected_column_clean()
        diagnosis_data_converted.get()
        input.drop_correlation()

        general_hist_switch = input.view_general_hist()
        if general_hist_switch == True:
            ui.remove_ui("#general-hist-plot")
            # Cálculo de la altura del plot para evitar que se mueva la posición de la página al recalcular el plot
            selected_cols = [col for col in clean_df.columns if input.dropKeyWordSelector() in col]
            print (selected_cols)
            subplot_rows_number = math.ceil(len(selected_cols) / subplot_cols_number)
            plot_height = "height: " + str(180*subplot_rows_number) + "px;"
            general_hist_plot = output_widget("widget_generalObservation")
            ui.insert_ui(
                ui.div({"id": "general-hist-plot"}, general_hist_plot, style = "width:100%; overflow-x:auto;" + plot_height),
                selector="#data-observation-general",
                where="beforeEnd",
            )
        else:
            ui.remove_ui("#general-hist-plot")

#################################### WIDGETS #################################################

    # WIDGET HISTOGRAMA DATOS DIAGNOSIS
    @output
    @render_widget
    def widget_diagnosisObservation():
        # Elementos a los que reaccionar:
        input.dropIdSelector()
        diagnosis_data_converted.get()

        fig = go.Figure()
        if diagnosis_data_converted.get() == False:
            fig.add_trace(go.Histogram(x=clean_df['diagnosis'].loc[clean_df['diagnosis'] == 'B'], name='Benigno', marker_color='#5ec962'))
            fig.add_trace(go.Histogram(x=clean_df['diagnosis'].loc[clean_df['diagnosis'] == 'M'], name='Maligno', marker_color='#440154'))            
        else:
            fig.add_trace(go.Histogram(x=clean_df['diagnosis'].loc[clean_df['diagnosis'] == 1], name='Maligno = 1', marker_color='#440154', legendgroup="Maligno"))
            fig.add_trace(go.Histogram(x=clean_df['diagnosis'].loc[clean_df['diagnosis'] == 0], name='Benigno = 0', marker_color='#5ec962', legendgroup="Benigno"))
        
        fig.update_layout(autosize=True,
                          height=180,
                          margin=dict(l=20, r=20, t=40, b=20),
                          barmode='stack')
        
        fig.update_traces(hovertemplate='%{y}')
        
        return fig
    
    # WIDGET HISTOGRAMA DATOS GENERALES
    @output
    @render_widget
    def widget_generalObservation():
        # Elementos a los que reaccionar:
        diagnosis_data_converted.get()
        input.drop_selected_column_clean()
        input.drop_correlation()

        selected_cols = [col for col in clean_df.columns if input.dropKeyWordSelector() in col]
        
        # Dividir dataframe en dos a partir de diagnosis
        dfM=pd.DataFrame()
        dfB=pd.DataFrame()
        if diagnosis_data_converted.get() == False:
            dfM=clean_df[clean_df['diagnosis'] == 'M']
            dfB=clean_df[clean_df['diagnosis'] == 'B']
        else:
            dfM=clean_df[clean_df['diagnosis'] == 1]
            dfB=clean_df[clean_df['diagnosis'] == 0]
        
        subplot_rows_number = math.ceil(len(selected_cols) / subplot_cols_number)

        fig = make_subplots(rows=subplot_rows_number, cols=subplot_cols_number, 
                            subplot_titles=selected_cols,
        )
        
        for idx,curr_col in enumerate(selected_cols):
            fig.add_trace(go.Histogram(x=dfM[curr_col], name = "Maligno", marker_color='#440154', opacity=0.7, legendgroup="Maligno", showlegend=idx==0),
                          row=math.floor(idx/subplot_cols_number)+1, col=(idx%subplot_cols_number)+1)
            fig.add_trace(go.Histogram(x=dfB[curr_col], name = "Benigno", marker_color='#5ec962', opacity=0.7, legendgroup="Benigno", showlegend=idx==0),
                          row=math.floor(idx/subplot_cols_number)+1, col=(idx%subplot_cols_number)+1)
        
        fig.update_layout(autosize=True,
                          barmode='overlay',
                          height=180 * subplot_rows_number,
                          showlegend=True,
                          margin=dict(l=20, r=20, t=40, b=20))

        fig.update_traces(hovertemplate='%{y}<br>Rango: %{x}')  
        
        return fig


##############################################################################################
#################################### LIMPIEZA DE DATOS #######################################
##############################################################################################

#################################### TABLAS ##################################################
    
    @output
    @render.table
    def cleanTable():
        #Elementos a los que reaccionar:
        input.dropIdSelector()
        diagnosis_data_converted.get()
        correlation_execution_counter.get()
        reset_dataframe_counter.get()
        
        return clean_df

#################################### EFECTOS REACTIVOS #######################################

    # CONVERTIR DIAGNOSIS
    @reactive.Effect
    @reactive.event(input.convert_diagnosis)
    def _():
        clean_df['diagnosis'] = original_df['diagnosis'].map({'M':1,'B':0})
        diagnosis_data_converted.set(True)

    # ELIMINAR COLUMNA SELECCIONADA
    @reactive.Effect
    @reactive.event(input.drop_selected_column_clean)
    def _():
        if input.dropIdSelector() in clean_df.columns:
            clean_df.drop(input.dropIdSelector(),axis=1,inplace=True)
            update_all_selectors()
    
    # MOSTRAR TABLA LIMPIA
    @reactive.Effect
    def _():
        clean_table_switch = input.view_clean_table()
        if clean_table_switch == True:
            clean_table = ui.output_table("cleanTable", style = "overflow-x:scroll; height:260px; overflow-y:auto;"),
            ui.insert_ui(
                ui.div({"id": "inserted-clean-table"}, clean_table),
                selector="#clean-table",
                where="beforeEnd",
            )
        else:
            ui.remove_ui("#inserted-clean-table")

#################################### UPDATES Y OTROS #########################################
    
    # ACTUALIZAR SELECTOR COLUMNA
    def update_dropIdSelector():
        column_dict = {}
        for col in clean_df.columns:
            if col != "diagnosis":
                column_dict[col] = col
        ui.update_select("dropIdSelector", choices=column_dict, selected=None)


##############################################################################################
#################################### CORRELACIÓN DE DATOS ####################################
##############################################################################################

#################################### EFECTOS REACTIVOS #######################################

    # ELIMINAR COLUMNAS CON CORRELACIÓN SUPERIOR
    @reactive.Effect
    @reactive.event(input.drop_correlation)
    def _():
        correlation_map = clean_df.drop(['diagnosis'], axis=1).corr().abs()
        upper_tri = correlation_map.where(np.triu(np.ones(correlation_map.shape),k=1).astype(bool))
        columns_to_drop = [column for column in upper_tri.columns if any(upper_tri[column] >= input.maximum_correlation())]
        clean_df.drop(columns_to_drop, axis=1, inplace=True)
        correlation_execution_counter.set(correlation_execution_counter.get() + 1)
        update_all_selectors()
        
        # Actualizar el widget de correlacion tras los cálculos:
        correlation_switch = input.view_correlation()
        if correlation_switch == True:
            ui.remove_ui("#correlation-plot")
            correlation_plot = output_widget("widget_correlation")
            ui.insert_ui(
                ui.div({"id": "correlation-plot"}, correlation_plot, style = "width:100%; overflow-x:auto; overflow-y:auto;"),
                selector="#correlation",
                where="beforeEnd",
            )

    # VER WIDGET CORRELACIÓN
    @reactive.Effect
    def _():
        correlation_switch = input.view_correlation()
        if correlation_switch == True:
            if diagnosis_data_converted.get() == True:
                ui.remove_ui("#correlation-plot")
                correlation_plot = output_widget("widget_correlation")
                ui.insert_ui(
                    ui.div({"id": "correlation-plot"}, correlation_plot, style = "width:100%; height:1000px; overflow-x:auto; overflow-y:auto;"),
                    selector="#correlation",
                    where="beforeEnd",
                )
            else:
                ui.insert_ui(
                    ui.div({"id": "correlation-plot"}, correlation_warning_ui("general_warnings")),
                    selector="#correlation",
                    where="beforeEnd",
                )
        else:
            ui.remove_ui("#correlation-plot")

#################################### WIDGETS #################################################    

    # WIDGET CORRELACIÓN
    @output
    @render_widget
    def widget_correlation():
        #Elementos a los que reaccionar:
        input.dropIdSelector()
        input.maximum_correlation()
        correlation_execution_counter.get()

        if diagnosis_data_converted.get() == False:
            return go.Figure()

        correlation_map = clean_df.corr().round(decimals=3)
        fig = go.Figure(data=[go.Heatmap(z=correlation_map,
                                        x = correlation_map.columns.values,
                                        y = correlation_map.columns.values,
                                        xgap = 1,
                                        ygap = 1,
                                        colorscale=px.colors.sequential.Viridis_r,
                                        name="") 
        ])

        fig.update_layout(autosize=True,
                          height=min(80*len(clean_df.columns), 1000),
                          yaxis=dict(scaleanchor = 'x'),
                          margin=dict(l=20, r=20, t=40, b=20),)
        
        fig = fig.update_traces(text=correlation_map,
                                texttemplate="%{text}",
                                hovertemplate='%{x} - %{y}<br>Correlación: %{z}')
        
        fig.update_yaxes(autorange="reversed")

        return fig


##############################################################################################
#################################### ALGORITMOS DE PREDICCIÓN ################################
##############################################################################################

    # SEPARACIÓN ENTRENAMIENTO - TEST (sin funcionalidad real, sólo sirve para crear una mejor UX, la división se realiza al crear los modelos)
    @reactive.Effect
    @reactive.event(input.make_test_split)
    def _():
        test_split_done.set(True)


##############################################################################################
#################################### ÁRBOL DE DECISIÓN #######################################
##############################################################################################

#################################### IMPORTANTES #############################################

    # COMPROBACIONES PREVIAS ÁRBOL DE DECISIÓN
    def dec_tree_previous_checks(test_size_split, df_len):
        if diagnosis_data_converted.get() == False:
            ui.insert_ui(
                ui.div({"id": "dec-tree-warning"}, diagnosis_warning_ui("dec_tree_warnings")),
                selector="#dec_tree_generator",
                where="beforeEnd",
            )
            return True

        if test_split_done.get() == False:
            ui.insert_ui(
                ui.div({"id": "dec-tree-warning"}, test_split_warning_ui("dec_tree_warnings")),
                selector="#dec_tree_generator",
                where="beforeEnd",
            )
            return True
        
        if len(list(input.dec_tree_features_sel())) == 0:
            ui.insert_ui(
                ui.div({"id": "dec-tree-warning"}, features_warning_ui("dec_tree_warnings")),
                selector="#dec_tree_generator",
                where="beforeEnd",
            )
            return True
        
        if df_len * test_size_split < 1.0:
            ui.insert_ui(
                ui.div({"id": "dec-tree-warning"}, test_split_low_warning_ui("dec_tree_warnings")),
                selector="#dec_tree_generator",
                where="beforeEnd",
            )
            return True
        
        if df_len * ( 1 - test_size_split ) < 1.0:
            ui.insert_ui(
                ui.div({"id": "dec-tree-warning"}, test_split_high_warning_ui("dec_tree_warnings")),
                selector="#dec_tree_generator",
                where="beforeEnd",
            )
            return True
        
        return False

    # FIT, PREDICCIÓN Y GUARDADO DE DATOS DEL ÁRBOL DE DECISIÓN
    def classification_model_dec_tree(model, data, size_test, predictors, outcome):
        # Crear la división de test y entrenamiento!
        data_train, data_test = train_test_split(data, test_size = size_test)
        
        # Fit del modelo:
        model.fit(data_train[predictors],data_train[outcome])

        # Hacer predicciones del set de entrenamiento:
        predictions = model.predict(data_train[predictors])

        # Setear los resultados del set de entrenamiento:
        accuracy_decTree.set((metrics.accuracy_score(predictions,data_train[outcome]) * 100).round(decimals=3))
        recall_decTree.set((metrics.recall_score(predictions,data_train[outcome]) * 100).round(decimals=3))
        precision_decTree.set((metrics.precision_score(predictions,data_train[outcome]) * 100).round(decimals=3))
        f1_decTree.set((metrics.f1_score(predictions,data_train[outcome]) * 100).round(decimals=3))

        # Hacer predicciones del set de test:
        predictions_test = model.predict(data_test[predictors])

        # Setear los resultados del set de test:
        accuracy_decTree_test.set((metrics.accuracy_score(predictions_test,data_test[outcome]) * 100).round(decimals=3))
        recall_decTree_test.set((metrics.recall_score(predictions_test,data_test[outcome]) * 100).round(decimals=3))
        precision_decTree_test.set((metrics.precision_score(predictions_test,data_test[outcome]) * 100).round(decimals=3))
        f1_decTree_test.set((metrics.f1_score(predictions_test,data_test[outcome]) * 100).round(decimals=3))

        # Creación y guardado de la matriz de confusión
        cm_train = metrics.confusion_matrix(predictions,data_train[outcome])
        cm_test = metrics.confusion_matrix(predictions_test,data_test[outcome])
        tree_conf_mat_train.set(cm_train)
        tree_conf_mat_test.set(cm_test)

        # Creación de la figura del árbol de decisión
        plt.figure(figsize=(12,12))
        m_tree = plot_tree(model, filled=True, feature_names=predictors, class_names=["Benign", "Malign"], rounded=True, fontsize=5)
        plt.savefig( str(decTree_image_folder) + "\\" + str(session.id) + '_dec_tree.jpg',format='jpg',bbox_inches = "tight", dpi=600)
        # Cerrar todas las figuras para evitar llenar la memoria de información innecesaria
        plt.close('all')
        
        # Guardado de datos de la figura del árbol de decisión
        coords = list()
        coords_x = list()
        coords_y = list()
        texts = list()

        for node in m_tree:
            coords.append(list(node.get_position()))
            texts.append(node.get_text().replace("\n", "<br>"))

        for x, y in coords:
            coords_x.append(x)
            coords_y.append(y)
        
        tree_plot_x_coords.set(coords_x)
        tree_plot_y_coords.set(coords_y)
        tree_plot_texts.set(texts)

#################################### EFECTOS REACTIVOS #######################################

    # GENERAR EL MODELO DE ÁRBOL DE DECISIÓN Y REALIZAR TODOS LOS CÁLCULOS
    @reactive.Effect
    @reactive.event(input.generate_decission_tree)
    def _():
        ui.remove_ui("#dec-tree-warning")

        # Obtener el tamaño de la separación de entrenamiento y la longitud de la base de datos para comprobaciones:
        test_size_split = input.test_split_value()
        df_len = len(clean_df)

        # Comprobaciones previas. Si algo falla, el modelo no se calcula y se reseta todo lo generado para no causar confusión:
        if dec_tree_previous_checks(test_size_split, df_len) == True:
            # Cerrar todas las visualizaciones
            ui.update_switch("view_variable_importance_dec_tree", value=False)
            ui.update_switch("conf_mat_dec_tree_switch", value=False)
            ui.update_switch("view_tree_dec_tree_switch", value=False)
            # Resetear todos los resultados
            reset_dec_tree_result_values()
            empty_dec_tree_feature_importance_df()
            decision_tree_execution_counter.set(0)
            return

        # Arreglar valores None para poder ser aceptados por el modelo:
        max_depth_val = input.dec_tree_max_depth()
        if max_depth_val == 0:
            max_depth_val = None
        
        max_features_value = input.dec_tree_max_features()
        if max_features_value == 'None':
            max_features_value = None

        # Crear el modelo de árbol de decisión
        dec_tree_model = DecisionTreeClassifier(criterion=input.dec_tree_criterion(),
                                                splitter=input.dec_tree_splitter(), 
                                                max_depth=max_depth_val,
                                                min_samples_split=input.dec_tree_min_samples_split(),
                                                min_samples_leaf=input.dec_tree_min_samples_leaf(),
                                                max_features=max_features_value)
        
        # Lista de las características que usamos:
        features_list = list(input.dec_tree_features_sel())

        # Fit y predicciónes del modelo. Guardado de todos los datos
        classification_model_dec_tree(dec_tree_model,clean_df,test_size_split,features_list,outcome_var)

        # Variables importantes y guardado de sus resultados
        empty_dec_tree_feature_importance_df()
        dec_tree_feat_imp = pd.Series(dec_tree_model.feature_importances_, index=features_list).sort_values(ascending=False)
        dec_tree_feat_imp_df.insert(0, "Característica", dec_tree_feat_imp.index)
        dec_tree_feat_imp_df.insert(1, "Valor", dec_tree_feat_imp.values.round(decimals=3) * 100)

        decision_tree_execution_counter.set(decision_tree_execution_counter.get()+1)

    # MOSTRAR EL WIDGET DE IMPORTANCIA DE VARIABLES DEL ÁRBOL DE DECISIÓN
    @reactive.Effect
    def _():
        var_imp_dec_tree_switch = input.view_variable_importance_dec_tree()
        if var_imp_dec_tree_switch == True:
            ui.remove_ui("#var-imp-dec-tree-plot")
            if decision_tree_execution_counter.get() > 0:
                var_imp_dec_tree_plot = output_widget("widget_dec_tree_var_imp")
                ui.insert_ui(
                    ui.div({"id": "var-imp-dec-tree-plot"}, var_imp_dec_tree_plot, style = "width:100%; overflow-x:auto; overflow-y:auto;"),
                    selector="#var_imp_dec_tree",
                    where="beforeEnd",
                )
            else:
                ui.insert_ui(
                    ui.div({"id": "var-imp-dec-tree-plot"}, feat_imp_warning_ui("dec_tree_warnings")),
                    selector="#var_imp_dec_tree",
                    where="beforeEnd",
                )
        else:
            ui.remove_ui("#var-imp-dec-tree-plot")
    
    # DESELECCIONAR VARIABLES POCO IMPORTANTES DEL ÁRBOL DE DECISIÓN
    @reactive.Effect
    @reactive.event(input.deselect_not_imp_vars_dec_tree)
    def _():
        minimum_importance = input.minimum_importance_dec_tree()
        important_columns_auto = [feature["Característica"] for idx, feature in dec_tree_feat_imp_df.iterrows() if (feature["Valor"] >= minimum_importance)]
        ui.update_checkbox_group("dec_tree_features_sel", selected=important_columns_auto)

    # MOSTRAR LA MATRIZ DE CONFUSIÓN DEL ÁRBOL DE DECISIÓN
    @reactive.Effect
    def _():
        conf_mat_dec_tree_switch = input.conf_mat_dec_tree_switch()
        if conf_mat_dec_tree_switch == True:
            ui.remove_ui("#dec-tree-conf-mat-train")
            ui.remove_ui("#dec-tree-conf-mat-test")
            if decision_tree_execution_counter.get() > 0:
                dec_tree_conf_mat_train = output_widget("widget_dec_tree_conf_mat_train")
                ui.insert_ui(
                    ui.div({"id": "dec-tree-conf-mat-train"}, dec_tree_conf_mat_train, style = "width:100%; height:300px; overflow-x:auto; overflow-y:auto;"),
                    selector="#dec_tree_conf_matrix_train",
                    where="beforeEnd",
                )
                dec_tree_conf_mat_test = output_widget("widget_dec_tree_conf_mat_test")
                ui.insert_ui(
                    ui.div({"id": "dec-tree-conf-mat-test"}, dec_tree_conf_mat_test, style = "width:100%; height:300px; overflow-x:auto; overflow-y:auto;"),
                    selector="#dec_tree_conf_matrix_test",
                    where="beforeEnd",
                )
            else:
                ui.insert_ui(
                    ui.div({"id": "dec-tree-conf-mat-train"}, conf_matrix_warning_ui("dec_tree_warnings")),
                    selector="#dec_tree_conf_matrix",
                    where="beforeEnd",
                )
        else:
            ui.remove_ui("#dec-tree-conf-mat-train")
            ui.remove_ui("#dec-tree-conf-mat-test")

    # MOSTRAR EL WIDGET DEL ÁRBOL DE DECISIÓN
    @reactive.Effect
    def _():
        view_tree_dec_tree_switch = input.view_tree_dec_tree_switch()
        if view_tree_dec_tree_switch == True:
            ui.remove_ui("#dec-tree-view-img")
            if decision_tree_execution_counter.get() > 0:
                dec_tree_view = output_widget("widget_dec_tree_view")
                ui.insert_ui(
                    ui.div({"id": "dec-tree-view-img"}, dec_tree_view, style = "width:100%; height:1000px; overflow-x:auto; overflow-y:auto;"),
                    selector="#dec_tree_view",
                    where="beforeEnd",
                )
            else:
                view_tree_dec_tree_warning = ui.output_text("decision_tree_warning_view_txt"),
                ui.insert_ui(
                    ui.div({"id": "dec-tree-view-img"}, view_tree_dec_tree_warning, style="color:red; font-style:italic; margin-top:20px; padding: 10px; background: #f7f7f7; border-radius: 10px;"),
                    selector="#dec_tree_view",
                    where="beforeEnd",
                )
        else:
            ui.remove_ui("#dec-tree-view-img")

#################################### WIDGETS #################################################    

    # WIDGET DE LA IMPORTANCIA DE LAS VARIABLES DEL ÁRBOL DE DECISIÓN
    @output
    @render_widget
    def widget_dec_tree_var_imp():
        #Variables a las que reaccionar:
        decision_tree_execution_counter.get()
        input.view_variable_importance_dec_tree()

        if len(dec_tree_feat_imp_df) == 0:
            return go.Figure()
        
        fig = go.Figure(data=[go.Bar(x = dec_tree_feat_imp_df["Valor"],
                                     y = dec_tree_feat_imp_df["Característica"],
                                     orientation='h',
                                     name="",
                                     marker=dict(color = dec_tree_feat_imp_df["Valor"],
                                                 colorscale=px.colors.sequential.Viridis_r))
        ])

        fig.update_layout(autosize=True,
                          height=max(280, 40*len(dec_tree_feat_imp_df)),
                          margin=dict(l=20, r=20, t=40, b=20),)
        
        fig = fig.update_traces(hovertemplate='%{y} : %{x}%')

        fig.update_yaxes(autorange="reversed")
        
        return fig
    
    # WIDGET MATRIZ DE CONFUSIÓN ENTRENAMIENTO DEL ÁRBOL DE DECISIÓN
    @output
    @render_widget
    def widget_dec_tree_conf_mat_train():
        cm_map = tree_conf_mat_train.get()
        fig = go.Figure(data=[go.Heatmap(z=cm_map,
                                        xgap = 1,
                                        ygap = 1,
                                        colorscale=px.colors.sequential.Teal,
                                        name="")
        ])

        fig.update_xaxes(
            autorange="reversed",
        )

        fig.update_layout(title="Matriz de confusión: datos entrenamiento",
                          xaxis_title="Valores reales",
                          yaxis_title="Valores predichos",
                          xaxis = dict(
                                tickmode = 'array',
                                tickvals = [0, 1],
                                ticktext = ['0', '1']
                            ),
                          yaxis = dict(
                                scaleanchor = 'x',
                                tickmode = 'array',
                                tickvals = [0, 1],
                                ticktext = ['0', '1']
                            ),
                          autosize=True,
                          height=300,
                          width=400,
                          margin=dict(l=20, r=20, t=40, b=20),)
        
        fig = fig.update_traces(text=cm_map,
                                texttemplate="%{text}",
                                hovertemplate='Valor real: %{x}<br>Valor predicho: %{y}<br>Cantidad: %{z}')
                
        return fig
    
    # WIDGET MATRIZ DE CONFUSIÓN TESTING DEL ÁRBOL DE DECISIÓN
    @output
    @render_widget
    def widget_dec_tree_conf_mat_test():
        cm_map = tree_conf_mat_test.get()
        fig = go.Figure(data=[go.Heatmap(z=cm_map,
                                        xgap = 1,
                                        ygap = 1,
                                        colorscale=px.colors.sequential.Teal,
                                        name="") 
        ])

        fig.update_xaxes(
            autorange="reversed",
        )

        fig.update_layout(title="Matriz de confusión: datos test",
                          xaxis_title="Valores reales",
                          yaxis_title="Valores predichos",
                          xaxis = dict(
                                tickmode = 'array',
                                tickvals = [0, 1],
                                ticktext = ['0', '1']
                            ),
                          yaxis = dict(
                                scaleanchor = 'x',
                                tickmode = 'array',
                                tickvals = [0, 1],
                                ticktext = ['0', '1']
                            ),
                          autosize=True,
                          height=300,
                          width=400,
                          margin=dict(l=20, r=20, t=40, b=20),)
        
        fig = fig.update_traces(text=cm_map,
                                texttemplate="%{text}",
                                hovertemplate='Valor real: %{x}<br>Valor predicho: %{y}<br>Cantidad: %{z}')
                
        return fig
    
    # WIDGET VISUALIZACIÓN DEL ÁRBOL DE DECISIÓN
    @output
    @render_widget
    def widget_dec_tree_view():
        # Variables a las que reaccionar:
        decision_tree_execution_counter.get()

        img_path = str(Path(__file__).parent / "DecTrees") + "\\" + str(session.id) + "_dec_tree.jpg"
        img_src = Image.open( img_path )
        
        fig = go.Figure()

        fig.add_trace(
            go.Scatter(
                x=tree_plot_x_coords.get(),
                y=tree_plot_y_coords.get(),
                text=tree_plot_texts.get(),
                mode="markers",
                marker=dict(
                    color="white",
                    size=60,
                    opacity=0.1,
                ),
                name="",
            )
        )

        # Configurar ejes
        fig.update_xaxes(
            visible=False,
            range=[0,1],
        )

        fig.update_yaxes(
            visible=False,
            range=[0,1],
            # el atributo de scaleanchor asegura que la relación de aspecto no cambie
            scaleanchor="x"
        )

        fig.add_layout_image(
            dict(
                x=-0.02,
                sizex=1.04,
                y=1.01,
                sizey=1.02,
                xref="x",
                yref="y",
                opacity=1.0,
                layer="above",
                sizing="stretch",
                source=img_src)
        )

        fig = fig.update_traces(hovertemplate='%{text}')

        fig.update_layout(autosize=True,
                          height=1000,
                          margin=dict(l=20, r=20, t=40, b=20),)
        
        return fig

#################################### TEXTOS ##################################################

    # RESULTADOS
    @output
    @render.text
    def decision_tree_accuracy():
        if accuracy_decTree.get() == -1:
            return "Exactitud: "
        return "Exactitud: " + str(accuracy_decTree.get()) + "%"

    @output
    @render.text
    def decision_tree_recall():
        if recall_decTree.get() == -1:
            return "Sensibilidad o TVP: "
        return "Sensibilidad o TVP: " + str(recall_decTree.get()) + "%"
    
    @output
    @render.text
    def decision_tree_precision():
        if precision_decTree.get() == -1:
            return "Precisión: "
        return "Precisión: " + str(precision_decTree.get()) + "%"
    
    @output
    @render.text
    def decision_tree_f1():
        if f1_decTree.get() == -1:
            return "F1 Score: "
        return "F1 Score: " + str(f1_decTree.get()) + "%"
    
    @output
    @render.text
    def decision_tree_accuracy_test():
        if accuracy_decTree_test.get() == -1:
            return "Exactitud: "
        return "Exactitud: " + str(accuracy_decTree_test.get()) + "%"

    @output
    @render.text
    def decision_tree_recall_test():
        if recall_decTree_test.get() == -1:
            return "Sensibilidad o TVP: "
        return "Sensibilidad o TVP: " + str(recall_decTree_test.get()) + "%"
    
    @output
    @render.text
    def decision_tree_precision_test():
        if precision_decTree_test.get() == -1:
            return "Precisión: "
        return "Precisión: " + str(precision_decTree_test.get()) + "%"
    
    @output
    @render.text
    def decision_tree_f1_test():
        if f1_decTree_test.get() == -1:
            return "F1 Score: "
        return "F1 Score: " + str(f1_decTree_test.get()) + "%"
    
    # WARNING VISUALIZACIÓN ÁRBOL
    @output
    @render.text
    def decision_tree_warning_view_txt():
        return "No se puede mostrar el árbol de decisión sin haber creado el modelo!"

#################################### UPDATES Y OTROS #########################################   

    # ACTUALIZAR CHECKBOX ÁRBOL DE DECISIÓN
    def update_decTree_checkbox_group():
        column_dict = {}
        for col in clean_df.columns:
            if col != "diagnosis":
                column_dict[col] = col
        ui.update_checkbox_group("dec_tree_features_sel", choices=column_dict, selected=list(column_dict))



##############################################################################################
#################################### RANDOM FOREST ###########################################
##############################################################################################

#################################### IMPORTANTES #############################################

    # COMPROBACIONES PREVIAS RANDOM FOREST
    def ran_forest_previous_checks(test_size_split, df_len):
        if diagnosis_data_converted.get() == False:
            ui.insert_ui(
                ui.div({"id": "ran-forest-warning"}, diagnosis_warning_ui("ran_forest_warnings")),
                selector="#ran_forest_generator",
                where="beforeEnd",
            )
            return True

        if test_split_done.get() == False:
            ui.insert_ui(
                ui.div({"id": "ran-forest-warning"}, test_split_warning_ui("ran_forest_warnings")),
                selector="#ran_forest_generator",
                where="beforeEnd",
            )
            return True
        
        if len(list(input.ran_forest_features_sel())) == 0:
            ui.insert_ui(
                ui.div({"id": "ran-forest-warning"}, features_warning_ui("ran_forest_warnings")),
                selector="#ran_forest_generator",
                where="beforeEnd",
            )
            return True
        
        if df_len * test_size_split < 1.0:
            ui.insert_ui(
                ui.div({"id": "ran-forest-warning"}, test_split_low_warning_ui("ran_forest_warnings")),
                selector="#ran_forest_generator",
                where="beforeEnd",
            )
            return True
        
        if df_len * ( 1 - test_size_split ) < 1.0:
            ui.insert_ui(
                ui.div({"id": "ran-forest-warning"}, test_split_high_warning_ui("ran_forest_warnings")),
                selector="#ran_forest_generator",
                where="beforeEnd",
            )
            return True
        
        return False

    # FIT, PREDICCIÓN Y GUARDADO DE DATOS DEL RANDOM FOREST
    def classification_model_random_forest(model, data, size_test, predictors, outcome, n_estimators):
        # Crear la división de test y entrenamiento!
        data_train, data_test = train_test_split(data, test_size = size_test)
        
        # Fit del modelo:
        model.fit(data_train[predictors],data_train[outcome])

        # Hacer predicciones del set de entrenamiento:
        predictions = model.predict(data_train[predictors])

        # Setear los resultados del set de entrenamiento:
        accuracy_ranForest.set((metrics.accuracy_score(predictions,data_train[outcome]) * 100).round(decimals=3))
        recall_ranForest.set((metrics.recall_score(predictions,data_train[outcome]) * 100).round(decimals=3))
        precision_ranForest.set((metrics.precision_score(predictions,data_train[outcome]) * 100).round(decimals=3))
        f1_ranForest.set((metrics.f1_score(predictions,data_train[outcome]) * 100).round(decimals=3))

        # Hacer predicciones del set de test:
        predictions_test = model.predict(data_test[predictors])

        # Setear los resultados del set de test:
        accuracy_ranForest_test.set((metrics.accuracy_score(predictions_test,data_test[outcome]) * 100).round(decimals=3))
        recall_ranForest_test.set((metrics.recall_score(predictions_test,data_test[outcome]) * 100).round(decimals=3))
        precision_ranForest_test.set((metrics.precision_score(predictions_test,data_test[outcome]) * 100).round(decimals=3))
        f1_ranForest_test.set((metrics.f1_score(predictions_test,data_test[outcome]) * 100).round(decimals=3))

        # Creación y guardado de la matriz de confusión
        cm_train = metrics.confusion_matrix(predictions,data_train[outcome])
        cm_test = metrics.confusion_matrix(predictions_test,data_test[outcome])
        ranForest_tree_conf_mat_train.set(cm_train)
        ranForest_tree_conf_mat_test.set(cm_test)

        coords_x_list = list()
        coords_y_list = list()
        texts_list = list()

        # Creación de las figuras de árboles de decisión (máximo 5 para ahorrar espacio)
        for index in range(0, min(5, n_estimators)):
            plt.figure(figsize=(12,12))
            m_tree = plot_tree(model.estimators_[index], filled=True, feature_names=predictors, class_names=["Benign", "Malign"], rounded=True, fontsize=5)
            plt.savefig( str(ranForest_image_folder) + "\\" + str(session.id) + '_ran_forest' + str(index) + '.jpg',format='jpg',bbox_inches = "tight", dpi=600)
            # Cerrar todas las figuras para evitar llenar la memoria de información innecesaria
            plt.close('all')

            # Guardado de datos de la figura del árbol de decisión
            coords = list()
            coords_x = list()
            coords_y = list()
            texts = list()

            for node in m_tree:
                coords.append(list(node.get_position()))
                # Arreglo del problema generado por boostrap sampling en los random forest:
                new_texts = node.get_text().split("\n")
                first_value = 0
                second_value = 0
                value_index = 0
                for idx, string in enumerate(new_texts):
                    values_split = re.split('(\d+)', string)
                    if len(values_split) > 0 and values_split[0] == 'value = [':
                        first_value = int(values_split[1])
                        second_value = int(values_split[3])
                        value_index = idx

                if value_index != 0:
                    new_texts[value_index - 1] = 'samples = ' + str(first_value + second_value)

                final_string = '<br>'.join(new_texts)

                texts.append(final_string)

            for x, y in coords:
                coords_x.append(x)
                coords_y.append(y)

            coords_x_list.append(coords_x)
            coords_y_list.append(coords_y)
            texts_list.append(texts)
        
        ranForest_tree_plot_x_coords.set(coords_x_list)
        ranForest_tree_plot_y_coords.set(coords_y_list)
        ranForest_tree_plot_texts.set(texts_list)

        random_forest_last_estimators_num.set(n_estimators)

#################################### EFECTOS REACTIVOS #######################################
    
    # GENERAR EL MODELO DE RANDOM FOREST Y REALIZAR TODOS LOS CÁLCULOS
    @reactive.Effect
    @reactive.event(input.generate_random_forest)
    def _():
        ui.remove_ui("#ran-forest-warning")

        # Obtener el tamaño de la separación de entrenamiento y la longitud de la base de datos para comprobaciones:
        test_size_split = input.test_split_value()
        df_len = len(clean_df)

        # Comprobaciones previas. Si algo falla, el modelo no se calcula y se resetean los resultados anteriores para no causar confusión:
        if ran_forest_previous_checks(test_size_split, df_len) == True:
            # Cerrar todas las visualizaciones
            ui.update_switch("view_variable_importance_ran_forest", value=False)
            ui.update_switch("conf_mat_ran_forest_switch", value=False)
            ui.update_switch("view_tree_ran_forest_switch", value=False)
            # Resetear todos los resultados
            reset_ran_forest_result_values()
            empty_ran_forest_feature_importance_df()
            random_forest_last_estimators_num.set(0)
            random_forest_execution_counter.set(0)
            return

        # Arreglar valores None para poder ser aceptados por el modelo:
        max_depth_val = input.ran_forest_max_depth()
        if max_depth_val == 0:
            max_depth_val = None
        
        max_features_value = input.ran_forest_max_features()
        if max_features_value == 'None':
            max_features_value = None

        n_estimators_ran_forest = input.ran_forest_n_estimators()

        # Crear el modelo de random forest
        ran_forest_model = RandomForestClassifier(n_estimators=n_estimators_ran_forest,
                                                  criterion=input.ran_forest_criterion(),
                                                  max_depth=max_depth_val,
                                                  min_samples_split=input.ran_forest_min_samples_split(),
                                                  min_samples_leaf=input.ran_forest_min_samples_leaf(),
                                                  max_features=max_features_value)
        # bootstrap=False # Boostrap sampling causa problemas al representar los árboles, su número de samples no
        # corresponde a la suma de los valores de cada tipo. Sin embargo, si se desactiva, todos los árboles generados
        # son exactamente iguales.
        
        # Lista de las características que usamos:
        features_list = list(input.ran_forest_features_sel())

        # Fit y predicciónes del modelo. Guardado de todos los datos
        classification_model_random_forest(ran_forest_model,clean_df,test_size_split,features_list,outcome_var,n_estimators_ran_forest)

        # Variables importantes y guardado de sus resultados
        empty_ran_forest_feature_importance_df()
        ran_forest_feat_imp = pd.Series(ran_forest_model.feature_importances_, index=features_list).sort_values(ascending=False)
        ran_forest_feat_imp_df.insert(0, "Característica", ran_forest_feat_imp.index)
        ran_forest_feat_imp_df.insert(1, "Valor", ran_forest_feat_imp.values.round(decimals=3) * 100)

        random_forest_execution_counter.set(random_forest_execution_counter.get()+1)

    # MOSTRAR EL WIDGET DE IMPORTANCIA DE VARIABLES DEL RANDOM FOREST
    @reactive.Effect
    def _():
        var_imp_ran_forest_switch = input.view_variable_importance_ran_forest()
        if var_imp_ran_forest_switch == True:
            ui.remove_ui("#var-imp-ran-forest-plot")
            if random_forest_execution_counter.get() > 0:
                var_imp_ran_forest_plot = output_widget("widget_ran_forest_var_imp")
                ui.insert_ui(
                    ui.div({"id": "var-imp-ran-forest-plot"}, var_imp_ran_forest_plot, style = "width:100%; overflow-x:auto; overflow-y:auto;"),
                    selector="#var_imp_ran_forest",
                    where="beforeEnd",
                )
            else:
                ui.insert_ui(
                    ui.div({"id": "var-imp-ran-forest-plot"}, feat_imp_warning_ui("ran_forest_warnings")),
                    selector="#var_imp_ran_forest",
                    where="beforeEnd",
                )
        else:
            ui.remove_ui("#var-imp-ran-forest-plot")
    
    # DESELECCIONAR VARIABLES POCO IMPORTANTES DEL RANDOM FOREST
    @reactive.Effect
    @reactive.event(input.deselect_not_imp_vars_ran_forest)
    def _():
        minimum_importance = input.minimum_importance_ran_forest()
        important_columns_auto = [feature["Característica"] for idx, feature in ran_forest_feat_imp_df.iterrows() if (feature["Valor"] >= minimum_importance)]
        ui.update_checkbox_group("ran_forest_features_sel", selected=important_columns_auto)

    # MOSTRAR LA MATRIZ DE CONFUSIÓN DEL RANDOM FOREST
    @reactive.Effect
    def _():
        conf_mat_ran_forest_switch = input.conf_mat_ran_forest_switch()
        if conf_mat_ran_forest_switch == True:
            ui.remove_ui("#ran-forest-conf-mat-train")
            ui.remove_ui("#ran-forest-conf-mat-test")
            if random_forest_execution_counter.get() > 0:
                ran_forest_conf_mat_train = output_widget("widget_ran_forest_conf_mat_train")
                ui.insert_ui(
                    ui.div({"id": "ran-forest-conf-mat-train"}, ran_forest_conf_mat_train, style = "width:100%; height:300px; overflow-x:auto; overflow-y:auto;"),
                    selector="#ran_forest_conf_matrix_train",
                    where="beforeEnd",
                )
                ran_forest_conf_mat_test = output_widget("widget_ran_forest_conf_mat_test")
                ui.insert_ui(
                    ui.div({"id": "ran-forest-conf-mat-test"}, ran_forest_conf_mat_test, style = "width:100%; height:300px; overflow-x:auto; overflow-y:auto;"),
                    selector="#ran_forest_conf_matrix_test",
                    where="beforeEnd",
                )
            else:
                ui.insert_ui(
                    ui.div({"id": "ran-forest-conf-mat-train"}, conf_matrix_warning_ui("ran_forest_warnings")),
                    selector="#ran_forest_conf_matrix",
                    where="beforeEnd",
                )
        else:
            ui.remove_ui("#ran-forest-conf-mat-train")
            ui.remove_ui("#ran-forest-conf-mat-test")

    # MOSTRAR EL WIDGET DEL RANDOM FOREST
    @reactive.Effect
    def _():
        view_tree_ran_forest_switch = input.view_tree_ran_forest_switch()
        if view_tree_ran_forest_switch == True:
            ui.remove_ui("#ran-forest-view-img")
            ui.remove_ui("#ran-forest-view-img-foot")
            if random_forest_execution_counter.get() > 0:
                ran_forest_view = output_widget("widget_ran_forest_view")
                ui.insert_ui(
                    ui.div({"id": "ran-forest-view-img"}, ran_forest_view, style = "width:100%; height:1000px; overflow-x:auto; overflow-y:auto;"),
                    selector="#ran_forest_view",
                    where="beforeEnd",
                )
                ran_forest_view_foot = ui.output_text("random_forest_view_foot_txt")
                ui.insert_ui(
                    ui.div({"id": "ran-forest-view-img-foot"}, ran_forest_view_foot, style="color:grey; font-style:italic; text-align:center; font-size: 0.7em;"),
                    selector="#ran_forest_view",
                    where="beforeEnd",
                )
                
            else:
                view_tree_ran_forest_warning = ui.output_text("random_forest_warning_view_txt"),
                ui.insert_ui(
                    ui.div({"id": "ran-forest-view-img"}, view_tree_ran_forest_warning, style="color:red; font-style:italic; margin-top:20px; padding: 10px; background: #f7f7f7; border-radius: 10px;"),
                    selector="#ran_forest_view",
                    where="beforeEnd",
                )
        else:
            ui.remove_ui("#ran-forest-view-img")
            ui.remove_ui("#ran-forest-view-img-foot")

    # ACTUALIZAR EL SELECTOR DE ÁRBOL DE DECISIÓN PARA MOSTRAR
    @reactive.Effect
    def _():
        n_estimators = random_forest_last_estimators_num.get()
        new_list = list()
        for index in range(0, min(5, n_estimators)):
            new_list.append(index)
        ui.update_select("view_tree_ran_forest_number", choices=new_list)

#################################### WIDGETS #################################################    

    # WIDGET DE LA IMPORTANCIA DE LAS VARIABLES DEL RANDOM FOREST
    @output
    @render_widget
    def widget_ran_forest_var_imp():
        #Variables a las que reaccionar:
        random_forest_execution_counter.get()
        input.view_variable_importance_ran_forest()

        if len(ran_forest_feat_imp_df) == 0:
            return go.Figure()
        
        fig = go.Figure(data=[go.Bar(x = ran_forest_feat_imp_df["Valor"],
                                     y = ran_forest_feat_imp_df["Característica"],
                                     orientation='h',
                                     name="",
                                     marker=dict(color = ran_forest_feat_imp_df["Valor"],
                                                 colorscale=px.colors.sequential.Viridis_r))
        ])

        fig.update_layout(autosize=True,
                          height=max(280, 40*len(ran_forest_feat_imp_df)),
                          margin=dict(l=20, r=20, t=40, b=20),)
        
        fig = fig.update_traces(hovertemplate='%{y} : %{x}%')

        fig.update_yaxes(autorange="reversed")
        
        return fig
    
    # WIDGET MATRIZ DE CONFUSIÓN ENTRENAMIENTO DEL RANDOM FOREST
    @output
    @render_widget
    def widget_ran_forest_conf_mat_train():
        cm_map = ranForest_tree_conf_mat_train.get()
        fig = go.Figure(data=[go.Heatmap(z=cm_map,
                                        xgap = 1,
                                        ygap = 1,
                                        colorscale=px.colors.sequential.Teal,
                                        name="")
        ])

        fig.update_xaxes(
            autorange="reversed",
        )

        fig.update_layout(title="Matriz de confusión: datos entrenamiento",
                          xaxis_title="Valores reales",
                          yaxis_title="Valores predichos",
                          xaxis = dict(
                                tickmode = 'array',
                                tickvals = [0, 1],
                                ticktext = ['0', '1']
                            ),
                          yaxis = dict(
                                scaleanchor = 'x',
                                tickmode = 'array',
                                tickvals = [0, 1],
                                ticktext = ['0', '1']
                            ),
                          autosize=True,
                          height=300,
                          width=400,
                          margin=dict(l=20, r=20, t=40, b=20),)
        
        fig = fig.update_traces(text=cm_map,
                                texttemplate="%{text}",
                                hovertemplate='Valor real: %{x}<br>Valor predicho: %{y}<br>Cantidad: %{z}')
                
        return fig
    
    # WIDGET MATRIZ DE CONFUSIÓN TESTING DEL RANDOM FOREST
    @output
    @render_widget
    def widget_ran_forest_conf_mat_test():
        cm_map = ranForest_tree_conf_mat_test.get()
        fig = go.Figure(data=[go.Heatmap(z=cm_map,
                                        xgap = 1,
                                        ygap = 1,
                                        colorscale=px.colors.sequential.Teal,
                                        name="")
        ])

        fig.update_xaxes(
            autorange="reversed",
        )

        fig.update_layout(title="Matriz de confusión: datos test",
                          xaxis_title="Valores reales",
                          yaxis_title="Valores predichos",
                          xaxis = dict(
                                tickmode = 'array',
                                tickvals = [0, 1],
                                ticktext = ['0', '1']
                            ),
                          yaxis = dict(
                                scaleanchor = 'x',
                                tickmode = 'array',
                                tickvals = [0, 1],
                                ticktext = ['0', '1']
                            ),
                          autosize=True,
                          height=300,
                          width=400,
                          margin=dict(l=20, r=20, t=40, b=20),)
        
        fig = fig.update_traces(text=cm_map,
                                texttemplate="%{text}",
                                hovertemplate='Valor real: %{x}<br>Valor predicho: %{y}<br>Cantidad: %{z}')
                
        return fig
    
    # WIDGET VISUALIZACIÓN DEL RANDOM FOREST
    @output
    @render_widget
    def widget_ran_forest_view():
        # Variables a las que reaccionar:
        random_forest_execution_counter.get()

        num_tree = int(input.view_tree_ran_forest_number())

        img_path = str(Path(__file__).parent / "RanForests") + "\\" + str(session.id) + '_ran_forest' + str(num_tree) + '.jpg'
        img_src = Image.open( img_path )
        
        fig = go.Figure()

        fig.add_trace(
            go.Scatter(
                x=ranForest_tree_plot_x_coords.get()[num_tree],
                y=ranForest_tree_plot_y_coords.get()[num_tree],
                text=ranForest_tree_plot_texts.get()[num_tree],
                mode="markers",
                marker=dict(
                    color="white",
                    size=60,
                    opacity=0.1,
                ),
                name="",
            )
        )

        # Configurar ejes
        fig.update_xaxes(
            visible=False,
            range=[0,1],
        )

        fig.update_yaxes(
            visible=False,
            range=[0,1],
            # el atributo scaleanchor asegura que la relación de aspecto se mantiene constante
            scaleanchor="x"
        )

        fig.add_layout_image(
            dict(
                x=-0.02,
                sizex=1.04,
                y=1.01,
                sizey=1.02,
                xref="x",
                yref="y",
                opacity=1.0,
                layer="above",
                sizing="stretch",
                source=img_src)
        )

        fig = fig.update_traces(hovertemplate='%{text}')

        fig.update_layout(autosize=True,
                          height=1000,
                          margin=dict(l=20, r=20, t=40, b=20),)
        
        return fig

#################################### TEXTOS ##################################################

    # RESULTADOS
    @output
    @render.text
    def random_forest_accuracy():
        if accuracy_ranForest.get() == -1:
            return "Exactitud: "
        return "Exactitud: " + str(accuracy_ranForest.get()) + "%"

    @output
    @render.text
    def random_forest_recall():
        if recall_ranForest.get() == -1:
            return "Sensibilidad o TVP: "
        return "Sensibilidad o TVP: " + str(recall_ranForest.get()) + "%"
    
    @output
    @render.text
    def random_forest_precision():
        if precision_ranForest.get() == -1:
            return "Precisión: "
        return "Precisión: " + str(precision_ranForest.get()) + "%"
    
    @output
    @render.text
    def random_forest_f1():
        if f1_ranForest.get() == -1:
            return "F1 Score: "
        return "F1 Score: " + str(f1_ranForest.get()) + "%"
    
    @output
    @render.text
    def random_forest_accuracy_test():
        if accuracy_ranForest_test.get() == -1:
            return "Exactitud: "
        return "Exactitud: " + str(accuracy_ranForest_test.get()) + "%"

    @output
    @render.text
    def random_forest_recall_test():
        if recall_ranForest_test.get() == -1:
            return "Sensibilidad o TVP: "
        return "Sensibilidad o TVP: " + str(recall_ranForest_test.get()) + "%"
    
    @output
    @render.text
    def random_forest_precision_test():
        if precision_ranForest_test.get() == -1:
            return "Precisión: "
        return "Precisión: " + str(precision_ranForest_test.get()) + "%"
    
    @output
    @render.text
    def random_forest_f1_test():
        if f1_ranForest_test.get() == -1:
            return "F1 Score: "
        return "F1 Score: " + str(f1_ranForest_test.get()) + "%"
    
    
    # WARNING VISUALIZACIÓN ÁRBOL
    @output
    @render.text
    def random_forest_warning_view_txt():
        return "No se puede mostrar uno de los árboles de decisión sin haber creado el modelo!"
    
    @output
    @render.text
    def random_forest_view_foot_txt():
        return "Nota: Los valores de samples mostrados en la imagen son erroneos. En los bocadillos de información son correctos, son la suma de samples."

#################################### UPDATES Y OTROS #########################################   

    # ACTUALIZAR CHECKBOX ÁRBOL DE DECISIÓN
    def update_ranForest_checkbox_group():
        column_dict = {}
        for col in clean_df.columns:
            if col != "diagnosis":
                column_dict[col] = col
        ui.update_checkbox_group("ran_forest_features_sel", choices=column_dict, selected=list(column_dict))


##############################################################################################
################################### REGRESIÓN LOGÍSTICA ######################################
##############################################################################################

#################################### IMPORTANTES #############################################

    # COMPROBACIONES PREVIAS DE LA REGRESIÓN LOGÍSTICA
    def log_reg_previous_checks(test_size_split, df_len):
        if diagnosis_data_converted.get() == False:
            ui.insert_ui(
                ui.div({"id": "log-reg-warning"}, diagnosis_warning_ui("log_reg_warnings")),
                selector="#log_reg_generator",
                where="beforeEnd",
            )
            return True

        if test_split_done.get() == False:
            ui.insert_ui(
                ui.div({"id": "log-reg-warning"}, test_split_warning_ui("log_reg_warnings")),
                selector="#log_reg_generator",
                where="beforeEnd",
            )
            return True
        
        if len(list(input.log_reg_features_sel())) == 0:
            ui.insert_ui(
                ui.div({"id": "log-reg-warning"}, features_warning_ui("log_reg_warnings")),
                selector="#log_reg_generator",
                where="beforeEnd",
            )
            return True
        
        if df_len * test_size_split < 1.0:
            ui.insert_ui(
                ui.div({"id": "log-reg-warning"}, test_split_low_warning_ui("log_reg_warnings")),
                selector="#log_reg_generator",
                where="beforeEnd",
            )
            return True
        
        if df_len * ( 1 - test_size_split ) < 1.0:
            ui.insert_ui(
                ui.div({"id": "log-reg-warning"}, test_split_high_warning_ui("log_reg_warnings")),
                selector="#log_reg_generator",
                where="beforeEnd",
            )
            return True
        
        return False

    # FIT, PREDICCIÓN Y GUARDADO DE DATOS DE LA REGRESIÓN LOGÍSTICA
    def classification_model_log_reg(model, data, size_test, predictors, outcome, log_reg_max_iter):
        # Crear la división de test y entrenamiento!
        data_train, data_test = train_test_split(data, test_size = size_test)
        
        # Fit del modelo:
        model.fit(data_train[predictors],data_train[outcome])

        if log_reg_max_iter == model.n_iter_[0]:
            logistic_regression_warning = ui.output_text("logistic_regression_warning_iters_txt"),
            ui.insert_ui(
                ui.div({"id": "log-reg-warning"}, logistic_regression_warning, style="color:orange; font-style:italic; margin-top:20px; padding: 10px; background: #f7f7f7; border-radius: 10px;"),
                selector="#log_reg_generator",
                where="beforeEnd",
            )

        # Hacer predicciones del set de entrenamiento:
        predictions = model.predict(data_train[predictors])

        # Setear los resultados del set de entrenamiento:
        accuracy_logReg.set((metrics.accuracy_score(predictions,data_train[outcome]) * 100).round(decimals=3))
        recall_logReg.set((metrics.recall_score(predictions,data_train[outcome]) * 100).round(decimals=3))
        precision_logReg.set((metrics.precision_score(predictions,data_train[outcome]) * 100).round(decimals=3))
        f1_logReg.set((metrics.f1_score(predictions,data_train[outcome]) * 100).round(decimals=3))

        # Hacer predicciones del set de test:
        predictions_test = model.predict(data_test[predictors])

        # Setear los resultados del set des test:
        accuracy_logReg_test.set((metrics.accuracy_score(predictions_test,data_test[outcome]) * 100).round(decimals=3))
        recall_logReg_test.set((metrics.recall_score(predictions_test,data_test[outcome]) * 100).round(decimals=3))
        precision_logReg_test.set((metrics.precision_score(predictions_test,data_test[outcome]) * 100).round(decimals=3))
        f1_logReg_test.set((metrics.f1_score(predictions_test,data_test[outcome]) * 100).round(decimals=3))

        # Creación y guardado de la matriz de confusión
        cm_train = metrics.confusion_matrix(predictions,data_train[outcome])
        cm_test = metrics.confusion_matrix(predictions_test,data_test[outcome])
        logReg_conf_mat_train.set(cm_train)
        logReg_conf_mat_test.set(cm_test)

#################################### EFECTOS REACTIVOS #######################################

    # GENERAR EL MODELO DE LA REGRESIÓN LOGÍSTICA Y REALIZAR TODOS LOS CÁLCULOS
    @reactive.Effect
    @reactive.event(input.generate_logistic_regression)
    def _():
        ui.remove_ui("#log-reg-warning")

        # Obtener el tamaño de la separación de entrenamiento y la longitud de la base de datos para comprobaciones:
        test_size_split = input.test_split_value()
        df_len = len(clean_df)

        # Comprobaciones previas. Si algo falla, el modelo no se calcula y se resetean los resultados anteriores para no causar confusión:
        if log_reg_previous_checks(test_size_split, df_len) == True:
            # Cerrar todas las visualizaciones
            ui.update_switch("view_variable_importance_log_reg", value=False)
            ui.update_switch("conf_mat_log_reg_switch", value=False)
            ui.update_switch("view_tree_log_reg_switch", value=False)
            # Resetear todos los resultados
            reset_log_reg_result_values()
            empty_log_reg_feature_importance_df()
            logistic_regression_execution_counter.set(0)
            return

        # Arreglar valores None para poder ser aceptados por el modelo:
        log_reg_penalty = input.log_reg_penalty()
        if log_reg_penalty == 'None':
            log_reg_penalty = None

        log_reg_tolerance = 1 * pow(10, input.log_reg_tol())

        log_reg_max_iters = input.log_reg_max_iter()

        log_reg_l1_rat = None
        if log_reg_penalty == "elasticnet":
            log_reg_l1_rat = 0.5

        # Crear el modelo de regresión logística
        log_reg_model = LogisticRegression(penalty=log_reg_penalty,
                                           tol=log_reg_tolerance,
                                           C=input.log_reg_c(),
                                           solver=input.log_reg_solver(),
                                           max_iter=log_reg_max_iters,
                                           l1_ratio=log_reg_l1_rat)
        
        # Lista de las características que usamos:
        features_list = list(input.log_reg_features_sel())

        # Fit y predicciónes del modelo. Guardado de todos los datos
        classification_model_log_reg(log_reg_model,clean_df,test_size_split,features_list,outcome_var,log_reg_max_iters)

        # Variables importantes y guardado de sus resultados
        empty_log_reg_feature_importance_df()
        log_reg_feat_imp = pd.Series(np.abs(log_reg_model.coef_[0]), index=features_list).sort_values(ascending=False)
        # La importancia de las variables en regresión logística no suman 1, lo cambiamos a porcentaje
        sum_all_imp_values = log_reg_feat_imp.sum()
        log_reg_feat_imp_df.insert(0, "Característica", log_reg_feat_imp.index)
        log_reg_feat_imp_df.insert(1, "Valor", (log_reg_feat_imp.values / sum_all_imp_values).round(decimals=3) * 100)

        logistic_regression_execution_counter.set(logistic_regression_execution_counter.get()+1)

    # MOSTRAR EL WIDGET DE IMPORTANCIA DE VARIABLES DE LA REGRESIÓN LOGÍSTICA
    @reactive.Effect
    def _():
        var_imp_log_reg_switch = input.view_variable_importance_log_reg()
        if var_imp_log_reg_switch == True:
            ui.remove_ui("#var-imp-log-reg-plot")
            if logistic_regression_execution_counter.get() > 0:
                var_imp_log_reg_plot = output_widget("widget_log_reg_var_imp")
                ui.insert_ui(
                    ui.div({"id": "var-imp-log-reg-plot"}, var_imp_log_reg_plot, style = "width:100%; overflow-x:auto; overflow-y:auto;"),
                    selector="#var_imp_log_reg",
                    where="beforeEnd",
                )
            else:
                ui.insert_ui(
                    ui.div({"id": "var-imp-log-reg-plot"}, feat_imp_warning_ui("log_reg_warnings")),
                    selector="#var_imp_log_reg",
                    where="beforeEnd",
                )
        else:
            ui.remove_ui("#var-imp-log-reg-plot")
    
    # DESELECCIONAR VARIABLES POCO IMPORTANTES DE LA REGRESIÓN LOGÍSTICA
    @reactive.Effect
    @reactive.event(input.deselect_not_imp_vars_log_reg)
    def _():
        minimum_importance = input.minimum_importance_log_reg()
        important_columns_auto = [feature["Característica"] for idx, feature in log_reg_feat_imp_df.iterrows() if (feature["Valor"] >= minimum_importance)]
        ui.update_checkbox_group("log_reg_features_sel", selected=important_columns_auto)

    # MOSTRAR LA MATRIZ DE CONFUSIÓN DE LA REGRESIÓN LOGÍSTICA
    @reactive.Effect
    def _():
        conf_mat_log_reg_switch = input.conf_mat_log_reg_switch()
        if conf_mat_log_reg_switch == True:
            ui.remove_ui("#log-reg-conf-mat-train")
            ui.remove_ui("#log-reg-conf-mat-test")
            if logistic_regression_execution_counter.get() > 0:
                log_reg_conf_mat_train = output_widget("widget_log_reg_conf_mat_train")
                ui.insert_ui(
                    ui.div({"id": "log-reg-conf-mat-train"}, log_reg_conf_mat_train, style = "width:100%; height:300px; overflow-x:auto; overflow-y:auto;"),
                    selector="#log_reg_conf_matrix_train",
                    where="beforeEnd",
                )
                log_reg_conf_mat_test = output_widget("widget_log_reg_conf_mat_test")
                ui.insert_ui(
                    ui.div({"id": "log-reg-conf-mat-test"}, log_reg_conf_mat_test, style = "width:100%; height:300px; overflow-x:auto; overflow-y:auto;"),
                    selector="#log_reg_conf_matrix_test",
                    where="beforeEnd",
                )
            else:
                ui.insert_ui(
                    ui.div({"id": "log-reg-conf-mat-train"}, conf_matrix_warning_ui("log_reg_warnings")),
                    selector="#log_reg_conf_matrix",
                    where="beforeEnd",
                )
        else:
            ui.remove_ui("#log-reg-conf-mat-train")
            ui.remove_ui("#log-reg-conf-mat-test")

    # ACTUALIZAR PENALTY SEGÚN SOLVER DE LA REGRESIÓN LOGÍSTICA
    @reactive.Effect
    def _():
        solver = input.log_reg_solver()
        if solver == "saga":
            ui.update_select("log_reg_penalty", choices={"elasticnet": "Elasticnet (L1 + L2)", "l1": "L1", "l2": "L2 (default)", "None": "None"})
        elif solver == "liblinear":
            ui.update_select("log_reg_penalty", choices={"l1": "L1", "l2": "L2 (default)"})
        else:
            ui.update_select("log_reg_penalty", choices={"l2": "L2 (default)", "None": "None"})
            
#################################### WIDGETS #################################################    

    # WIDGET DE LA IMPORTANCIA DE LAS VARIABLES DE LA REGRESIÓN LOGÍSTICA
    @output
    @render_widget
    def widget_log_reg_var_imp():
        #Variables a las que reaccionar:
        logistic_regression_execution_counter.get()
        input.view_variable_importance_log_reg()

        if len(log_reg_feat_imp_df) == 0:
            return go.Figure()
        
        fig = go.Figure(data=[go.Bar(x = log_reg_feat_imp_df["Valor"],
                                     y = log_reg_feat_imp_df["Característica"],
                                     orientation='h',
                                     name="",
                                     marker=dict(color = log_reg_feat_imp_df["Valor"],
                                                 colorscale=px.colors.sequential.Viridis_r))
        ])

        fig.update_layout(autosize=True,
                          height=max(280, 40*len(log_reg_feat_imp_df)),
                          margin=dict(l=20, r=20, t=40, b=20),)
        
        fig = fig.update_traces(hovertemplate='%{y} : %{x}%')

        fig.update_yaxes(autorange="reversed")
        
        return fig
    
    # WIDGET MATRIZ DE CONFUSIÓN ENTRENAMIENTO DE LA REGRESIÓN LOGÍSTICA
    @output
    @render_widget
    def widget_log_reg_conf_mat_train():
        cm_map = logReg_conf_mat_train.get()
        fig = go.Figure(data=[go.Heatmap(z=cm_map,
                                        xgap = 1,
                                        ygap = 1,
                                        colorscale=px.colors.sequential.Teal,
                                        name="")
        ])

        fig.update_xaxes(
            autorange="reversed",
        )

        fig.update_layout(title="Matriz de confusión: datos entrenamiento",
                          xaxis_title="Valores reales",
                          yaxis_title="Valores predichos",
                          xaxis = dict(
                                tickmode = 'array',
                                tickvals = [0, 1],
                                ticktext = ['0', '1']
                            ),
                          yaxis = dict(
                                scaleanchor = 'x',
                                tickmode = 'array',
                                tickvals = [0, 1],
                                ticktext = ['0', '1']
                            ),
                          autosize=True,
                          height=300,
                          width=400,
                          margin=dict(l=20, r=20, t=40, b=20),)
        
        fig = fig.update_traces(text=cm_map,
                                texttemplate="%{text}",
                                hovertemplate='Valor real: %{x}<br>Valor predicho: %{y}<br>Cantidad: %{z}')
                
        return fig
    
    # WIDGET MATRIZ DE CONFUSIÓN TESTING DE LA REGRESIÓN LOGÍSTICA
    @output
    @render_widget
    def widget_log_reg_conf_mat_test():
        cm_map = logReg_conf_mat_test.get()
        fig = go.Figure(data=[go.Heatmap(z=cm_map,
                                        xgap = 1,
                                        ygap = 1,
                                        colorscale=px.colors.sequential.Teal,
                                        name="")
        ])

        fig.update_xaxes(
            autorange="reversed",
        )

        fig.update_layout(title="Matriz de confusión: datos test",
                          xaxis_title="Valores reales",
                          yaxis_title="Valores predichos",
                          xaxis = dict(
                                tickmode = 'array',
                                tickvals = [0, 1],
                                ticktext = ['0', '1']
                            ),
                          yaxis = dict(
                                scaleanchor = 'x',
                                tickmode = 'array',
                                tickvals = [0, 1],
                                ticktext = ['0', '1']
                            ),
                          autosize=True,
                          height=300,
                          width=400,
                          margin=dict(l=20, r=20, t=40, b=20),)
        
        fig = fig.update_traces(text=cm_map,
                                texttemplate="%{text}",
                                hovertemplate='Valor real: %{x}<br>Valor predicho: %{y}<br>Cantidad: %{z}')
                
        return fig
    

#################################### TEXTOS ##################################################

    # WARNINGS DE LA REGRESIÓN LOGÍSTICA    
    @output
    @render.text
    def logistic_regression_warning_iters_txt():
        return "El modelo ha parado porque ha llegado al máximo de iteraciones! Modifica los datos de entrada o aumenta el número máximo de iteraciones."

    # RESULTADOS DE LA REGRESIÓN LOGÍSTICA
    @output
    @render.text
    def logistic_regression_accuracy():
        if accuracy_logReg.get() == -1:
            return "Exactitud: "
        return "Exactitud: " + str(accuracy_logReg.get()) + "%"

    @output
    @render.text
    def logistic_regression_recall():
        if recall_logReg.get() == -1:
            return "Sensibilidad o TVP: "
        return "Sensibilidad o TVP: " + str(recall_logReg.get()) + "%"
    
    @output
    @render.text
    def logistic_regression_precision():
        if precision_logReg.get() == -1:
            return "Precisión: "
        return "Precisión: " + str(precision_logReg.get()) + "%"
    
    @output
    @render.text
    def logistic_regression_f1():
        if f1_logReg.get() == -1:
            return "F1 Score: "
        return "F1 Score: " + str(f1_logReg.get()) + "%"
    
    @output
    @render.text
    def logistic_regression_accuracy_test():
        if accuracy_logReg_test.get() == -1:
            return "Exactitud: "
        return "Exactitud: " + str(accuracy_logReg_test.get()) + "%"

    @output
    @render.text
    def logistic_regression_recall_test():
        if recall_logReg_test.get() == -1:
            return "Sensibilidad o TVP: "
        return "Sensibilidad o TVP: " + str(recall_logReg_test.get()) + "%"
    
    @output
    @render.text
    def logistic_regression_precision_test():
        if precision_logReg_test.get() == -1:
            return "Precisión: "
        return "Precisión: " + str(precision_logReg_test.get()) + "%"
    
    @output
    @render.text
    def logistic_regression_f1_test():
        if f1_logReg_test.get() == -1:
            return "F1 Score: "
        return "F1 Score: " + str(f1_logReg_test.get()) + "%"
    
    # WARNING MATRIZ DE CONFUSIÓN DE LA REGRESIÓN LOGÍSTICA
    @output
    @render.text
    def logistic_regression_warning_conf_matrix_txt():
        return "No se puede mostrar la matriz de confusión de la regresión logística sin haber creado el modelo!"

#################################### UPDATES Y OTROS #########################################   

    # ACTUALIZAR CHECKBOX DE LA REGRESIÓN LOGÍSTICA
    def update_logReg_checkbox_group():
        column_dict = {}
        for col in clean_df.columns:
            if col != "diagnosis":
                column_dict[col] = col
        ui.update_checkbox_group("log_reg_features_sel", choices=column_dict, selected=list(column_dict))
    

##############################################################################################
######################## RESETEAR CLEAN DATAFRAME Y ACCIONES GENERALES #######################
##############################################################################################

    # RESETEO GENERAL
    @reactive.Effect
    @reactive.event(input.reset_clean_df)
    def _():
        # Resetear la dataframe de datos limpios
        reset_clean_df()

        # Vaciar las dataframes de importancia de variables
        empty_all_feature_importance_df()

        # Actualizar y cerrar partes necesarias
        update_all_selectors()
        close_every_ui_after_reset()

        reset_dataframe_counter.set( reset_dataframe_counter.get() + 1 )
        diagnosis_data_converted.set(False)
        correlation_execution_counter.set(0)
        test_split_done.set(False)

        decision_tree_execution_counter.set(0)
        random_forest_execution_counter.set(0)
        logistic_regression_execution_counter.set(0)
        reset_all_result_values()

    # RESETEAR CLEAN_DF
    def reset_clean_df():
        # Vaciar la dataframe limpia
        for columnName in clean_df.columns:
            clean_df.drop(columnName, axis = 1, inplace=True)
        clean_df.drop(clean_df.index, inplace=True)

        # Rellenar la dataframe limpia con los datos originales
        for columnName in original_df.columns:
            clean_df[columnName] = original_df[columnName]

    # VACIAR TODAS LAS DATAFRAMES DE IMPORTANCIA DE VARIABLES
    def empty_all_feature_importance_df():
        # Vaciar la tabla de importancia de variables si es necesario
        empty_dec_tree_feature_importance_df()
        empty_ran_forest_feature_importance_df()
        empty_log_reg_feature_importance_df()

    # Para árbol de decisión
    def empty_dec_tree_feature_importance_df():
        if len(dec_tree_feat_imp_df) > 0:
            dec_tree_feat_imp_df.drop(["Característica", "Valor"], axis = 1, inplace=True)
            dec_tree_feat_imp_df.drop(dec_tree_feat_imp_df.index, inplace=True)
    
    # Para bosque aleatorio
    def empty_ran_forest_feature_importance_df():
        if len(ran_forest_feat_imp_df) > 0:
            ran_forest_feat_imp_df.drop(["Característica", "Valor"], axis = 1, inplace=True)
            ran_forest_feat_imp_df.drop(ran_forest_feat_imp_df.index, inplace=True)

    # Para regresión logística
    def empty_log_reg_feature_importance_df():
        if len(log_reg_feat_imp_df) > 0:
            log_reg_feat_imp_df.drop(["Característica", "Valor"], axis = 1, inplace=True)
            log_reg_feat_imp_df.drop(log_reg_feat_imp_df.index, inplace=True)
    
    # ACTUALIZAR SELECTORES Y CHECKBOXES NECESARIOS
    def update_all_selectors():
        update_dropIdSelector()
        update_decTree_checkbox_group()
        update_ranForest_checkbox_group()
        update_logReg_checkbox_group()

    # RESET TODAS LAS VARIABLES DE RESULTADOS
    def reset_all_result_values():
        #Decision Tree:
        reset_dec_tree_result_values()
        #Random Forest:
        reset_ran_forest_result_values()
        #Logistic regression:
        reset_log_reg_result_values()

    # RESET TODAS LAS VARIABLES DE RESULTADOS ÁRBOL DECISION
    def reset_dec_tree_result_values():
        accuracy_decTree.set(-1)
        recall_decTree.set(-1)
        precision_decTree.set(-1)
        f1_decTree.set(-1)

        accuracy_decTree_test.set(-1)
        recall_decTree_test.set(-1)
        precision_decTree_test.set(-1)
        f1_decTree_test.set(-1)

    # RESET TODAS LAS VARIABLES DE RESULTADOS BOSQUE ALEATORIO
    def reset_ran_forest_result_values():
        accuracy_ranForest.set(-1)
        recall_ranForest.set(-1)
        precision_ranForest.set(-1)
        f1_ranForest.set(-1)

        accuracy_ranForest_test.set(-1)
        recall_ranForest_test.set(-1)
        precision_ranForest_test.set(-1)
        f1_ranForest_test.set(-1)

    # RESET TODAS LAS VARIABLES DE RESULTADOS REGRESIÓN LOGÍSTICA
    def reset_log_reg_result_values():
        accuracy_logReg.set(-1)
        recall_logReg.set(-1)
        precision_logReg.set(-1)
        f1_logReg.set(-1)

        accuracy_logReg_test.set(-1)
        recall_logReg_test.set(-1)
        precision_logReg_test.set(-1)
        f1_logReg_test.set(-1)

    # CERRAR LAS PARTES DE UI NECESARIAS AL HACER RESET
    def close_every_ui_after_reset():
        ui.remove_ui("#correlation-plot")
        ui.update_switch("view_correlation", value=False)
        
        ui.remove_ui("#var-imp-dec-tree-plot")
        ui.update_switch("view_variable_importance_dec_tree", value=False)
        
        ui.remove_ui("#dec-tree-conf-mat-train")
        ui.remove_ui("#dec-tree-conf-mat-test")
        ui.update_switch("conf_mat_dec_tree_switch", value=False)
        
        ui.remove_ui("#dec-tree-view-img")
        ui.update_switch("view_tree_dec_tree_switch", value=False)
        
        ui.remove_ui("#var-imp-ran-forest-plot")
        ui.update_switch("view_variable_importance_ran_forest", value=False)
        
        ui.remove_ui("#ran-forest-conf-mat-train")
        ui.remove_ui("#ran-forest-conf-mat-test")
        ui.update_switch("conf_mat_ran_forest_switch", value=False)
        
        ui.remove_ui("#ran-forest-view-img")
        ui.remove_ui("#ran-forest-view-img-foot")
        ui.update_switch("view_tree_ran_forest_switch", value=False)
        
        ui.remove_ui("#var-imp-log-reg-plot")
        ui.update_switch("view_variable_importance_log_reg", value=False)
        
        ui.remove_ui("#log-reg-conf-mat-train")
        ui.remove_ui("#log-reg-conf-mat-test")
        ui.update_switch("conf_mat_log_reg_switch", value=False)

        # Resetear manualmente el select del número de árbol a mostrar de random forest
        ui.update_select("view_tree_ran_forest_number", choices=empty_column_dict),

    def resetAndDeleteEverything():
        delete_imgs_from_disk()
        reset_dataframes_if_changed()
    
    # RESETEAR DATAFRAMES NECESARIAS (POR SI SE HA REALIZADO UNA RECARGA DE LA PÁGINA)
    def reset_dataframes_if_changed():
        if len(original_df.columns) != len(clean_df.columns) or ( clean_df["diagnosis"][0] == 1 or clean_df["diagnosis"][0] == 0 ):
            reset_clean_df()

        # Vaciar las tablas de importancia de variables si es necesario
        empty_all_feature_importance_df()
    
    # BORRAR LAS IMÁGENES GUARDADAS EN DISCO (USADO AL CERRAR SESIÓN)
    def delete_imgs_from_disk():
        img_dec_tree_path = str(decTree_image_folder) + "\\" + str(session.id) + "_dec_tree.jpg"
        Path(img_dec_tree_path).unlink(missing_ok=True)

        img_custom_dec_tree_path = str(decTree_image_folder)  + "\\" + str(session.id) + "custom_dec_tree.jpg"
        Path(img_custom_dec_tree_path).unlink(missing_ok=True)

        for index in range(5):
            img_ran_forest_path = str(ranForest_image_folder) + "\\" + str(session.id) + '_ran_forest' + str(index) + '.jpg'
            Path(img_ran_forest_path).unlink(missing_ok=True)

            img_custom_ran_forest_path = str(ranForest_image_folder) + "\\" + str(session.id) + 'custom_ran_forest' + str(index) + '.jpg'
            Path(img_custom_ran_forest_path).unlink(missing_ok=True)

        return
    
    # LLAMAR A LA FUNCIÓN QUE RESETEE Y LIMPIE LOS DATOS DEL DISCO AL CERRAR UNA SESIÓN
    session.on_ended(resetAndDeleteEverything)
    