from shiny import reactive, render, ui, module
from shinywidgets import output_widget, render_widget

import plotly.graph_objects as go
import plotly.express as px
from plotly.subplots import make_subplots

from pathlib import Path
from PIL import Image

import numpy as np
import pandas as pd
import math as math
import re
import matplotlib.pyplot as plt

# Importar los algoritmos y modelss desde scikit learn:
from sklearn.model_selection import train_test_split
from sklearn.tree import DecisionTreeClassifier, plot_tree
from sklearn.ensemble import RandomForestClassifier
from sklearn.linear_model import LogisticRegression
from sklearn import metrics

# Importar todos los paquetes de información, ui y warnings necesarios
from UIMessages.warningsCustomNoData import custom_new_dataframe_warning_ui, custom_new_dataframe_warning_empty_ui, custom_new_dataframe_accepted_ui, custom_missing_data_table_warning_ui, custom_missing_data_types_warning_ui, custom_missing_data_histogram_warning_ui, warnings_custom_no_data_server
from UIMessages.warningsCustomGeneral import custom_missing_data_clean_table_warning_ui, custom_missing_data_clean_types_warning_ui, custom_missing_data_clean_hist_warning_ui, custom_missing_data_clean_hist_div_outcome_warning_ui, custom_too_many_unique_clean_hist_div_outcome_warning_ui, custom_correlation_warning_ui, custom_correlation_no_data_warning_ui, warnings_custom_general_server
from UIMessages.warningsCustomAlgorithms import custom_no_data_warning_ui, custom_outcome_warning_ui, custom_features_warning_ui, custom_features_non_numeric_warning_ui, custom_features_nan_warning_ui, custom_test_split_warning_ui, custom_test_split_low_warning_ui, custom_test_split_high_warning_ui, warnings_custom_algorithms_server



################################## DATAFRAMES Y DICCIONARIOS #################################

empty_column_dict = {}

#Datos del dataframe custom 
custom_df = pd.DataFrame()
clean_custom_df = pd.DataFrame()

#Datos para Decision tree custom
custom_dec_tree_feat_imp_df = pd.DataFrame()

#Datos para Random Forest custom
custom_ran_forest_feat_imp_df = pd.DataFrame()

#Datos para Logistic regression custom
custom_log_reg_feat_imp_df = pd.DataFrame()

# Paths para guardado de imágenes
decTree_image_folder = Path(Path(__file__).parent / 'DecTrees')
ranForest_image_folder = Path(Path(__file__).parent / 'RanForests')



##############################################################################################
##############################################################################################
####################################### MÓDULO DE UI #########################################
##############################################################################################
##############################################################################################

@module.ui
def customTool_ui():
    return ui.div(
        ui.panel_main(
            ui.tags.h1("PRUEBA CON TUS PROPIOS DATOS"),
            ui.markdown("Introduce un dataset válido y... **¡prueba por tu cuenta!** En esta sección puedes probar a generar modelos de aprendizaje supervisado y obtener resultados con tus datos, pero no dispondrás de ninguna guía o ayuda."),
            width=12, style="text-align: justify; text-justify: inter-word;"
        ),
        ui.panel_main(
            {"id": "new_dataframe_input"},
            ui.input_file("new_dataframe", "Elige el archivo a subir (Csv o Excel):", multiple=False, width="60%"),
            width=12, style="padding-top:10px; font-weight: bold;"
        ),
        ui.panel_main(
            {"id": "new_dataframe_input_msg"},
        ),
        ui.panel_main(
            ui.tags.hr(),
            width=12, style="padding-top:10px;"
        ),

#################################### CUSTOM: DATOS INICIALES ###########################
        ui.panel_main(
            ui.tags.h3("OBSERVACIÓN DE LOS DATOS IMPORTADOS"),
            ui.tags.p("Tabla de datos:", style="font-weight: bold;"),
            width=12
        ),
        ui.panel_main(
            {"id": "custom-table"},
            ui.input_switch("view_custom_table", "¡Ver los datos!"),
            width=12
        ),
        ui.panel_main(
            {"id": "custom-table-types"},
            ui.tags.p("Tipo de variables de los datos:", style="font-weight: bold;"),
            ui.input_switch("view_custom_table_types", "¡Ver los tipos de los datos!"),
            width=12, style="padding-top:20px;"
        ),
        ui.panel_main(
            {"id": "custom-table-histogram"},
            ui.tags.p("Histograma de los datos:", style="font-weight: bold;"),
            ui.input_switch("view_custom_table_histogram", "¡Ver el histograma de los datos!"),
            width=12, style="padding-top:20px;"
        ),              
        ui.panel_main(
            ui.tags.hr(),
            width=12, style="padding-top:20px;"
        ),

#################################### CUSTOM: SELECCIÓN VARIABLE A PREDECIR ###########################
        ui.panel_main(
            ui.tags.h3("VARIABLE A PREDECIR"),
            ui.tags.p("Selección y preparación de la variable a predecir. Si la variable no es numérica, ¡debe ser convertida!"),
            width=12, style="text-align: justify; text-justify: inter-word;"
        ),
        ui.div(
            ui.div(
                ui.input_select("outcomeSelectorCustom", "Selecciona la columna a predecir:", empty_column_dict),
                style="font-weight: bold;"
            ),
            ui.tags.hr(style="color: Gainsboro;"),
            ui.row(
                ui.column(4,
                    ui.input_action_button("convert_custom_outcome", "Convertir datos de la variable a predecir a numérica"),
                ),
                ui.column(4,
                    ui.input_action_button("convert_custom_outcome_higher_0", "Convertir datos de la variable a predecir (0=0, >0=1)"),
                ),
            ),
            style="width: 66%; border: solid 1px WhiteSmoke; border-radius: 10px; background:white; padding:15px 20px 10px 20px; text-align: justify; text-justify: inter-word;"
        ),
        ui.panel_main(
            ui.tags.hr(),
            width=12, style="padding-top:30px;"
        ),

#################################### CUSTOM: LIMPIEZA DE LOS DATOS ###########################
        ui.panel_main(
            ui.tags.h3("LIMPIEZA DE DATOS"),
            ui.tags.h5("Operaciones generales:", style="padding-top:10px;"),width=12
        ),
        ##### LIMPIEZA GENERAL #####
        ui.div(
            ui.row(
                ui.column(6,
                    ui.input_action_button("fillNA_all_column_custom_clean", "Convertir valores nulos a 0 de todos los datos", width="100%"),
                ),
                ui.column(6,
                    ui.input_action_button("convert_numeric_all_column_custom_clean", "Convertir a valores numéricos todos los datos", width="100%"),
                ),
            ),
            style="width: 66%; border: solid 1px WhiteSmoke; border-radius: 10px; background:white; padding:15px 20px 10px 20px; text-align: justify; text-justify: inter-word;"
        ),
        ##### OPERACIONES CON COLUMNAS #####
        ui.panel_main(
            ui.tags.h5("Operaciones con columnas:", style="padding-top:15px;"),
            width=12 , style="padding-top:15px;"
        ),
        ui.div(
            ui.div(
                ui.input_select("dropIdSelectorCustom", "Selecciona la columna a tratar:", empty_column_dict),
                style="font-weight: bold;"
            ),
            ui.tags.hr(style="color: Gainsboro;"),
            ui.row(
                ui.column(4,
                    ui.tags.p("Eliminación:"),
                    ui.input_action_button("drop_selected_column_custom_clean", "Eliminar la columna seleccionada", width="100%"),
                ),
                ui.column(4,
                    ui.tags.p("Operaciones básicas:"),
                    ui.input_action_button("fillNA_selected_column_custom_clean", "Convertir valores nulos a 0 de la columna seleccionada", width="100%"),
                    ui.tags.p(),
                    ui.input_action_button("convert_numeric_selected_column_custom_clean", "Convertir valores a numéricos de la columna seleccionada", width="100%"),
                ),
                ui.column(4,
                    ui.tags.p("Operaciones personalizadas:"),
                    ui.input_numeric("numeric_selected_column_custom_clean", "", value=0),
                    ui.input_action_button("convert_custom_numeric_selected_column_custom_clean", "Convertir valores nulos a número de la columna seleccionada", width="100%"),
                    ui.tags.p(),
                    ui.input_text("text_selected_column_custom_clean", "", value=""),
                    ui.input_action_button("convert_custom_text_selected_column_custom_clean", "Convertir valores nulos a texto de la columna seleccionada", width="100%"),
                ),
            ),
            style="border: solid 1px WhiteSmoke; border-radius: 10px; background:white; padding:15px 20px 10px 20px; text-align: justify; text-justify: inter-word;"
        ),
        
        ##### OPERACIONES CON FILAS #####
        ui.panel_main(
            ui.tags.h5("Operaciones con filas"),
            
            width=12, style="padding-top: 20px;"
        ),
        ui.div(
            ui.div(
                ui.input_numeric("selected_row_to_drop_custom_clean", "Selecciona el índice de la fila a tratar:", value=0, min=0, max=1),
                style="font-weight: bold;"
            ),
            ui.tags.hr(style="color: Gainsboro;"),
            ui.row(
                ui.column(6,
                    {"id": "custom-clean-row-buttons-drop"},
                    ui.tags.p("Eliminación:"),
                    ui.input_action_button("drop_row_selected_custom_clean", "Eliminar la fila seleccionada", width="100%"),
                ),
                ui.column(6,
                    {"id": "custom-clean-row-buttons"},
                    ui.tags.p("Operaciones generales con filas:"),
                    ui.input_action_button("drop_all_NA_rows_custom_clean", "Eliminar todas las filas con valores nulos", width="100%"),
                ),
            ),
            style="width: 66%; border: solid 1px WhiteSmoke; border-radius: 10px; background:white; padding:15px 20px 10px 20px; text-align: justify; text-justify: inter-word;"
        ),
        ##### VISUALIZACIONES DE DATOS LIMPIOS #####
        ui.panel_main(
            {"id": "custom-clean-table"},
            ui.tags.h5("Visualizaciones de los datos limpios"),
            ui.tags.p(),
            ui.div(
                ui.input_switch("view_custom_clean_table", "¡Ver los datos limpios!", width="100%"),
                style="font-weight: bold;"
            ),
            width=12, style="padding-top:30px;"
        ),
        ui.panel_main(
            {"id": "custom-clean-table-types"},
            ui.div(
                ui.input_switch("view_custom_clean_table_types", "¡Ver los tipos de los datos limpios!", width="100%"),
                style="font-weight: bold;"
            ),
            width=12, style="padding-top:20px;"
        ),
        ui.panel_main(
            {"id": "custom-clean-table-histogram"},
            ui.div(
                ui.input_switch("view_custom_clean_table_histogram", "¡Ver el histograma de los datos limpios!", width="100%"),
                style="font-weight: bold;"
            ),
            width=12, style="padding-top:20px;"
        ),
        ui.panel_main(
            {"id": "custom-clean-table-histogram-div-outcome"},
            ui.div(
                ui.input_switch("view_custom_clean_table_histogram_div_outcome", "¡Ver el histograma de los datos limpios divididos según la variable a predecir!", width="100%"),
                style="font-weight: bold;"
            ),
            width=12, style="padding-top:20px;"
        ), 
        ui.panel_main(
            ui.tags.hr(),
            width=12, style="padding-top:20px;"
        ),
        

#################################### CUSTOM: CORRELACIÓN ###########################
        ui.panel_main(
            ui.tags.h3("CORRELACIÓN DE DATOS"),
            width=12
        ),
        ui.panel_main(
            {"id": "custom-correlation"},
            ui.input_slider("custom_maximum_correlation", "Máxima correlación permitida:", min=0, max=1, value=0.7, step=0.01),
            ui.input_action_button("custom_drop_correlation", "Eliminar columnas con correlación superior a la seleccionada"),
            ui.tags.p(style="padding-bottom: 10px;"),
            ui.div(
                ui.input_switch("custom_view_correlation", "¡Ver la correlación entre datos!", width="100%"),
                style="font-weight: bold;"
            ),
            
            width=12
        ),
        ui.panel_main(
            ui.tags.hr(),
            width=12, style="padding-top:10px;"
        ),

#################################### CUSTOM: TABS DE ALGORITMOS ###########################
        ui.panel_main(
            ui.tags.h3("ALGORITMOS DE PREDICCIÓN"),
            width=12, style="padding-bottom:10px;"
        ),
        ui.panel_main(
            {"id": "custom_test_split"},
            ui.tags.h5("División de los datos en sets de entrenamiento y test"),
            ui.input_slider("custom_test_split_value", "Tamaño del subset de testeo:", min=0, max=1, value=0.2, step=0.01),
            ui.input_action_button("custom_make_test_split", "Divide los datos en subset de entrenamiento y testeo"),
            width=12
        ),
        ui.panel_main(
            ui.tags.hr(),
            width=12, style="padding-top:20px; padding-bottom:5px;"
        ),
        ui.navset_tab(
            ######################################################################
            ################ CUSTOM: ÁRBOL DE DECISION ###########################
            ######################################################################
            ui.nav(
                "Árbol de decisión",
                ui.tags.h3("Árbol de decisión", style="padding-top:20px; padding-bottom:20px;"),
                ######### AD_C: AJUSTES, CARACTERÍSTICAS Y CREACIÓN #########
                ui.row(
                    ui.column(
                        3,
                        ui.panel_well(
                            ui.tags.h5("Ajustes:"),
                            ui.tags.hr(),
                            ui.input_select("custom_dec_tree_criterion","Criterion", {"gini": "Gini (default)", "entropy": "Entropy", "log_loss": "Log_loss"}),
                            ui.input_select("custom_dec_tree_splitter","Splitter", {"best": "Best (default)", "random": "Random"}),
                            ui.input_slider("custom_dec_tree_max_depth", "Max Depth (0 = None / default)", 0, 32, 0, step=1),
                            ui.input_slider("custom_dec_tree_min_samples_split", "Min samples split (default = 2)", 1, 6, 2, step=1),
                            ui.input_slider("custom_dec_tree_min_samples_leaf", "Min samples leaf (default = 1)", 1, 5, 1, step=1),
                            ui.input_select("custom_dec_tree_max_features","Max features", {"None": "None (default)", "sqrt": "Sqrt", "log2": "Log2"}),
                        ),
                    ),
                    ui.column(
                        3,
                        ui.panel_well(
                            ui.tags.h5("Características:"),
                            ui.tags.hr(),
                            ui.input_checkbox_group("custom_dec_tree_features_sel", "", choices=empty_column_dict),
                        ),
                    ),
                    ui.column(
                        6,
                        ui.panel_main(
                            {"id": "custom_dec_tree_generator"},
                            ui.tags.h5("¡Crea el modelo de predicción!"),
                            ui.input_action_button("custom_generate_decission_tree", "Generar el modelo de árbol de decisión"),
                            width=12
                        ),
                        ui.panel_main(
                            {"id": "custom_var_imp_dec_tree"},
                            ui.tags.hr(),
                            ui.tags.h5("Importancia de las características para el modelo:"),
                            ui.div(
                                ui.input_switch("custom_view_variable_importance_dec_tree", "¡Ver la importancia de las características!", width="100%"),
                                style="font-weight: bold;"
                            ),
                            width=12
                        ),
                        ui.panel_main(
                            {"id": "custom_var_imp_slider_dec_tree"},
                            ui.input_slider("custom_minimum_importance_dec_tree", "Mínima importancia:", min=0, max=100, value=5.0, step=0.1),
                            ui.input_action_button("custom_deselect_not_imp_vars_dec_tree", "Deseleccionar características poco importantes automaticamente"),
                            width=12
                        ),
                    ),
                ),
                ui.panel_main(
                    ui.tags.hr(),
                    width=12
                ),
                ######### AD_C: MATRIZ DE CONFUSIÓN #########
                ui.panel_main(
                    ui.tags.h5("Resultados del modelo: matriz de confusión y métricas básicas"),
                    width=12, style="padding-bottom: 10px;"
                ),
                ui.panel_main(
                    {"id": "custom_dec_tree_conf_matrix"},
                    ui.div(
                        ui.input_switch("custom_conf_mat_dec_tree_switch", "¡Ver la matriz de confusión del árbol de decisión generado!", width="60%"),
                        style="font-weight: bold;"
                    ),
                    width=12
                ),
                ui.row(
                    ui.column(6,
                        ui.panel_main(
                            {"id": "custom_dec_tree_conf_matrix_train"},
                            width=12
                        ),
                    ),
                    ui.column(6,
                        ui.panel_main(
                            {"id": "custom_dec_tree_conf_matrix_test"},
                            width=12
                        ),
                    ),
                ),
                ######### AD_C: RESULTADOS CON ENTRENAMIENTO Y TEST #########
                ui.row(
                    ui.column(6,
                        ui.tags.p("Resultados con los datos de entrenamiento:", style="font-weight: bold;"),
                        ui.panel_main(
                            ui.output_text_verbatim("custom_decision_tree_precision"),
                            ui.output_text_verbatim("custom_decision_tree_recall"),
                            ui.output_text_verbatim("custom_decision_tree_f1"),
                            ui.output_text_verbatim("custom_decision_tree_accuracy"),
                            width=7
                        ),
                    ),
                    ui.column(6,
                        ui.tags.p("Resultados con los datos de prueba:", style="font-weight: bold;"),
                        ui.panel_main(
                            ui.output_text_verbatim("custom_decision_tree_precision_test"),
                            ui.output_text_verbatim("custom_decision_tree_recall_test"),
                            ui.output_text_verbatim("custom_decision_tree_f1_test"),
                            ui.output_text_verbatim("custom_decision_tree_accuracy_test"),
                            width=7
                        ),
                    ),
                    style="padding-top:30px;"
                ),                       
                ui.panel_main(
                    ui.tags.p(style="padding-bottom:20px;"),
                    ui.tags.hr(),
                    width=12
                ),
                ######### AD_C: VISUALIZACIÓN DEL ÁRBOL #########
                ui.panel_main(
                    {"id": "custom_dec_tree_view"},
                    ui.tags.h5("Representación del árbol"),
                    ui.div(
                        ui.input_switch("custom_view_tree_dec_tree_switch", "¡Ver la representación del árbol de decisión generado!", width="80%"),
                        style="font-weight: bold; padding-top: 10px;"
                    ),
                    width=12
                ),
            ),
            ######################################################################
            ################ CUSTOM: RANDOM FOREST ###############################
            ######################################################################
            ui.nav("Bosque aleatorio",
                ui.tags.h3("Bosque aleatorio", style="padding-top:20px; padding-bottom:20px;"),
                ######### RF_C: AJUSTES, CARACTERÍSTICAS Y CREACIÓN #########
                ui.row(
                    ui.column(
                        3,
                        ui.panel_well(
                            ui.tags.h5("Ajustes:"),
                            ui.tags.hr(),
                            ui.input_slider("custom_ran_forest_n_estimators", "Num Estimators (default = 100)", 1, 200, 10, step=1),
                            ui.input_select("custom_ran_forest_criterion","Criterion", {"gini": "Gini (default)", "entropy": "Entropy", "log_loss": "Log_loss"}),
                            ui.input_slider("custom_ran_forest_max_depth", "Max Depth (0 = None / default)", 0, 32, 0, step=1),
                            ui.input_slider("custom_ran_forest_min_samples_split", "Min samples split (default = 2)", 1, 6, 2, step=1),
                            ui.input_slider("custom_ran_forest_min_samples_leaf", "Min samples leaf (default = 1)", 1, 5, 1, step=1),
                            ui.input_select("custom_ran_forest_max_features","Max features", {"None": "None (default)", "sqrt": "Sqrt", "log2": "Log2"}),
                        ),
                    ),
                    ui.column(
                        3,
                        ui.panel_well(
                            ui.tags.h5("Características:"),
                            ui.tags.hr(),
                            ui.input_checkbox_group("custom_ran_forest_features_sel", "", choices=empty_column_dict),
                        ),
                    ),
                    ui.column(
                        6,
                        ui.panel_main(
                            {"id": "custom_ran_forest_generator"},
                            ui.tags.h5("¡Crea el modelo de predicción!"),
                            ui.input_action_button("generate_custom_random_forest", "Generar el modelo de bosque aletorio"),
                            width=12
                        ),
                        ui.panel_main(
                            {"id": "var_imp_custom_ran_forest"},
                            ui.tags.hr(),
                            ui.tags.h5("Importancia de las características para el modelo:"),
                            ui.div(
                                ui.input_switch("view_variable_importance_custom_ran_forest", "¡Ver la importancia de las características!", width="100%"),
                                style="font-weight: bold;"
                            ),
                            width=12
                        ),
                        ui.panel_main(
                            {"id": "var_imp_slider_custom_ran_forest"},
                            ui.input_slider("minimum_importance_custom_ran_forest", "Mínima importancia:", min=0, max=100, value=5.0, step=0.1),
                            ui.input_action_button("deselect_not_imp_vars_custom_ran_forest", "Deseleccionar características poco importantes automaticamente"),
                            width=12
                        ),
                    ),
                ),
                ui.panel_main(
                    ui.tags.hr(),
                    width=12
                ),


                ######### RF_C: MATRIZ DE CONFUSIÓN #########
                ui.panel_main(
                    ui.tags.h5("Resultados del modelo: matriz de confusión y métricas básicas"),
                    width=12, style="padding-bottom: 10px;"
                ),
                ui.panel_main(
                    {"id": "custom_ran_forest_conf_matrix"},
                    ui.div(
                        ui.input_switch("conf_mat_custom_ran_forest_switch", "¡Ver la matriz de confusión del random forest generado!", width="60%"),
                        style="font-weight: bold;"
                    ),
                    width=12
                ),
                ui.row(
                    ui.column(6,
                        ui.panel_main(
                            {"id": "custom_dec_tree_conf_matrix_train"},
                            width=12
                        ),
                    ),
                    ui.column(6,
                        ui.panel_main(
                            {"id": "custom_dec_tree_conf_matrix_test"},
                            width=12
                        ),
                    ),
                ),
                ######### RF_C: RESULTADOS CON ENTRENAMIENTO Y TEST #########
                ui.row(
                    ui.column(6,
                        ui.tags.p("Resultados con los datos de entrenamiento:", style="font-weight: bold;"),
                        ui.panel_main(
                            ui.output_text_verbatim("custom_random_forest_precision"),
                            ui.output_text_verbatim("custom_random_forest_recall"),
                            ui.output_text_verbatim("custom_random_forest_f1"),
                            ui.output_text_verbatim("custom_random_forest_accuracy"),
                            width=7
                        ),
                    ),
                    ui.column(6,
                        ui.tags.p("Resultados con los datos de prueba:", style="font-weight: bold;"),
                        ui.panel_main(
                            ui.output_text_verbatim("custom_random_forest_precision_test"),
                            ui.output_text_verbatim("custom_random_forest_recall_test"),
                            ui.output_text_verbatim("custom_random_forest_f1_test"),
                            ui.output_text_verbatim("custom_random_forest_accuracy_test"),
                            width=7
                        ),
                    ),
                    style="padding-top:30px;"
                ),
                ui.panel_main(
                    ui.tags.p(style="padding-bottom:20px;"),
                    ui.tags.hr(),
                    width=12
                ),
                ########## RF_C: REPRESENTACIÓN ÁRBOL ##########
                ui.panel_main(
                    {"id": "custom_ran_forest_view"},
                    ui.tags.h5("Representación de los árboles", style="padding-bottom:10px;"),
                    ui.input_select("view_tree_custom_ran_forest_number", "Selecciona el árbol que quieres mostrar", empty_column_dict),
                    ui.div(
                        ui.input_switch("view_tree_custom_ran_forest_switch", "¡Ver la representación de los árboles de decisión generados!", width="80%"),
                        style="font-weight: bold;"
                    ),
                    width=12, 
                ),
            ),
            ##############################################################
            ################ CUSTOM: REGRESIÓN LOGÍSTICA #################
            ##############################################################
            ui.nav("Regresión logística",
                ui.tags.h3("Regresión logística", style="padding-top:20px; padding-bottom:20px;"),
                ######### RL_C: AJUSTES, CARACTERÍSTICAS Y CREACIÓN #########
                ui.row(
                    ui.column(
                        3,
                        ui.panel_well(
                            ui.tags.h5("Ajustes:"),
                            ui.tags.hr(),
                            ui.input_select("custom_log_reg_solver","Solver", {"lbfgs": "Lbfgs (default)", "liblinear": "Liblinear", "newton-cg": "Newton-cg", "newton-cholesky": "Newton-cholesky", "sag": "Sag", "saga": "Saga"}, selected="lbfgs"),
                            ui.input_select("custom_log_reg_penalty","Penalty", {"l2": "L2 (default)", "None": "None"}, selected="l2"),
                            ui.input_slider("custom_log_reg_tol", "Tolerance (default = 1e-4) - 1e(valor seleccionado)", -10, 0, -4, step=1),
                            ui.input_slider("custom_log_reg_c", "C (default = 1)", 1, 3000, 1, step=1),
                            ui.input_slider("custom_log_reg_max_iter", "Max iterations (default = 100)", 100, 5000, 100, step=10),
                        ),
                    ),
                    ui.column(
                        3,
                        ui.panel_well(
                            ui.tags.h5("Características:"),
                            ui.tags.hr(),
                            ui.input_checkbox_group("custom_log_reg_features_sel", "", choices=empty_column_dict),
                        ),
                    ),
                    ui.column(
                        6,
                        ui.panel_main(
                            {"id": "custom_log_reg_generator"},
                            ui.tags.h5("¡Crea el modelo de predicción!"),
                            ui.input_action_button("custom_generate_logistic_regression", "Generar el modelo de Regresión logística"),
                            width=12
                        ),
                        ui.panel_main(
                            {"id": "custom_var_imp_log_reg"},
                            ui.tags.hr(),
                            ui.tags.h5("Importancia de las características para el modelo:"),
                            ui.div(
                                ui.input_switch("custom_view_variable_importance_log_reg", "¡Ver la importancia de las características!", width="100%"),
                                style="font-weight: bold;"
                            ),
                            width=12
                        ),
                        ui.panel_main(
                            {"id": "custom_var_imp_slider_log_reg"},
                            ui.input_slider("custom_minimum_importance_log_reg", "Mínima importancia:", min=0, max=100, value=5.0, step=0.1),
                            ui.input_action_button("custom_deselect_not_imp_vars_log_reg", "Deseleccionar características poco importantes automaticamente"),
                            width=12
                        ),
                    ),
                ),
                ui.panel_main(
                    ui.tags.hr(),
                    width=12
                ),
                ######### RL_C: MATRIZ DE CONFUSIÓN #########
                ui.panel_main(
                    ui.tags.h5("Resultados del modelo: matriz de confusión y métricas básicas"),
                    width=12, style="padding-bottom: 10px;"
                ),
                ui.panel_main(
                    {"id": "custom_log_reg_conf_matrix"},
                    ui.div(
                        ui.input_switch("custom_conf_mat_log_reg_switch", "¡Ver la matriz de confusión de la regresión logística generada!", width="60%"),
                        style="font-weight: bold;"
                    ),
                    width=12
                ),
                ui.row(
                    ui.column(6,
                        ui.panel_main(
                            {"id": "custom_log_reg_conf_matrix_train"},
                            width=12
                        ),
                    ),
                    ui.column(6,
                        ui.panel_main(
                            {"id": "custom_log_reg_conf_matrix_test"},
                            width=12
                        ),
                    ),
                ),
                ######### RL_C: RESULTADOS CON ENTRENAMIENTO Y TEST #########
                ui.row(
                    ui.column(6,
                        ui.tags.p("Resultados con los datos de entrenamiento:", style="font-weight: bold;"),
                        ui.panel_main(
                            ui.output_text_verbatim("custom_logistic_regression_precision"),
                            ui.output_text_verbatim("custom_logistic_regression_recall"),
                            ui.output_text_verbatim("custom_logistic_regression_f1"),
                            ui.output_text_verbatim("custom_logistic_regression_accuracy"),
                            width=7
                        ),
                    ),
                    ui.column(6,
                        ui.tags.p("Resultados con los datos de prueba:", style="font-weight: bold;"),
                        ui.panel_main(
                            ui.output_text_verbatim("custom_logistic_regression_precision_test"),
                            ui.output_text_verbatim("custom_logistic_regression_recall_test"),
                            ui.output_text_verbatim("custom_logistic_regression_f1_test"),
                            ui.output_text_verbatim("custom_logistic_regression_accuracy_test"),
                            width=7
                        ),
                    ),
                    style="padding-top:30px;"
                ),                   
            ),
        ),
        ui.panel_main(
            width=12, style="padding-top:70px;"
        ),
    )



##############################################################################################
##############################################################################################
#################################### MÓDULO DE SERVIDOR ######################################
##############################################################################################
##############################################################################################

@module.server
def customTool_server(input, output, session):
  
################# VARIABLES REACTIVAS Y DE CONTROL DE LA HERRAMIENTA #########################
    
    ######## CUSTOM #########
    #Controles generales custom:
    custom_df_counter = reactive.Value(0)
    custom_rows_deleted = reactive.Value(0)
    custom_correlation_execution_counter = reactive.Value(0)
    custom_test_split_done = reactive.Value(False)
    #custom_reset_dataframe_counter = reactive.Value(0)
    
    #Custom Dec-Tree:
    custom_decision_tree_execution_counter = reactive.Value(0)

    custom_accuracy_decTree = reactive.Value(-1)
    custom_recall_decTree = reactive.Value(-1)
    custom_precision_decTree = reactive.Value(-1)
    custom_f1_decTree = reactive.Value(-1)

    custom_accuracy_decTree_test = reactive.Value(-1)
    custom_recall_decTree_test = reactive.Value(-1)
    custom_precision_decTree_test = reactive.Value(-1)
    custom_f1_decTree_test = reactive.Value(-1)

    custom_tree_plot_x_coords = reactive.Value()
    custom_tree_plot_y_coords = reactive.Value()
    custom_tree_plot_texts = reactive.Value()

    custom_tree_conf_mat_train = reactive.Value()
    custom_tree_conf_mat_test = reactive.Value()

    #Custom Random forest:
    custom_random_forest_execution_counter = reactive.Value(0)
    custom_random_forest_last_estimators_num = reactive.Value(0)

    custom_accuracy_ranForest = reactive.Value(-1)
    custom_recall_ranForest = reactive.Value(-1)
    custom_precision_ranForest = reactive.Value(-1)
    custom_f1_ranForest = reactive.Value(-1)

    custom_accuracy_ranForest_test = reactive.Value(-1)
    custom_recall_ranForest_test = reactive.Value(-1)
    custom_precision_ranForest_test = reactive.Value(-1)
    custom_f1_ranForest_test = reactive.Value(-1)

    custom_ranForest_tree_plot_x_coords = reactive.Value()
    custom_ranForest_tree_plot_y_coords = reactive.Value()
    custom_ranForest_tree_plot_texts = reactive.Value()

    custom_ranForest_tree_conf_mat_train = reactive.Value()
    custom_ranForest_tree_conf_mat_test = reactive.Value()

    #Custom Logistic Regression:
    custom_logistic_regression_execution_counter = reactive.Value(0)

    custom_accuracy_logReg = reactive.Value(-1)
    custom_recall_logReg = reactive.Value(-1)
    custom_precision_logReg = reactive.Value(-1)
    custom_f1_logReg = reactive.Value(-1)

    custom_accuracy_logReg_test = reactive.Value(-1)
    custom_recall_logReg_test = reactive.Value(-1)
    custom_precision_logReg_test = reactive.Value(-1)
    custom_f1_logReg_test = reactive.Value(-1)

    custom_logReg_conf_mat_train = reactive.Value()
    custom_logReg_conf_mat_test = reactive.Value()


################# MODULOS DE SERVIDORES AUXILIARES DE LA HERRAMIENTA #########################

    warnings_custom_no_data_server("custom_tool_warnings_no_data")
    warnings_custom_general_server("custom_tool_warnings_general")

    warnings_custom_algorithms_server("custom_tool_warnings_dec_tree")
    warnings_custom_algorithms_server("custom_tool_warnings_ran_forest")
    warnings_custom_algorithms_server("custom_tool_warnings_log_reg")


##############################################################################################
#################################### SUBIDA DE DATOS #########################################
##############################################################################################

#################################### IMPORTANTES #############################################

    # SUBIR DATOS NUEVOS, RESETEAR BASE DE DATOS CUSTOM Y UI NECESARIA
    @reactive.Effect
    @reactive.event(input.new_dataframe)
    def load_data():
        file = input.new_dataframe()

        # Comprobar el archivo
        if not file:
            print("Invalid File")
            return
        
        # Si el archivo es válido y es un csv o excel se acepta
        else:
            #print(file[0]["name"])
            new_df = pd.DataFrame()
            if "csv" in file[0]["name"]:
                new_df = pd.read_csv(file[0]["datapath"])
                
            elif "xls" in file[0]["name"] or "xlsx" in file[0]["name"]:
                new_df = pd.read_excel(file[0]["datapath"])
            
            else:
                ui.remove_ui("#new-df-warning")
                ui.insert_ui(
                    ui.div({"id": "new-df-warning"}, custom_new_dataframe_warning_ui("custom_tool_warnings_no_data")),
                    selector="#new_dataframe_input_msg",
                    where="beforeEnd",
                )
                return
            
            # Si hay datos en la dataframe utilizada la podemos usar
            if len(new_df) > 0 and len(new_df.columns) > 0:
                # Vaciar la dataframe custom y rellenarla con los nuevos datos
                empty_custom_dataframes()

                # Rellenar ambas dataframes custom
                for columnName in new_df.columns:
                    custom_df[columnName] = new_df[columnName]
                
                for columnName in new_df.columns:
                    clean_custom_df[columnName] = new_df[columnName]

                # Actualizar los elementos de UI necesarios
                update_all_selectors_custom()
                update_dropRowSelector_custom()

                # Resetear valores necesarios y vaciar dataframes necesarias
                reset_custom_values_and_feat_imp_dfs()
                reset_all_custom_algoritms_reactive_values()
                remove_and_update_ui_elements_when_new_custom_df()

                # Dataframe es correcta
                ui.remove_ui("#new-df-warning")
                ui.insert_ui(
                    ui.div({"id": "new-df-warning"}, custom_new_dataframe_accepted_ui("custom_tool_warnings_no_data")),
                    selector="#new_dataframe_input_msg",
                    where="beforeEnd",
                )

                custom_df_counter.set( custom_df_counter.get() + 1 )
                return
            
            # Si la dataframe no es correcta se enseña un mesnaje de aviso
            else:
                ui.remove_ui("#new-df-warning")
                ui.insert_ui(
                    ui.div({"id": "new-df-warning"}, custom_new_dataframe_warning_empty_ui("custom_tool_warnings_no_data")),
                    selector="#new_dataframe_input_msg",
                    where="beforeEnd",
                )
                print("Valid file type but empty")
                return

    # VACIAR DATAFRAMES CUSTOM
    def empty_custom_dataframes():
        # Vaciar la dataframe custom 
        if len(custom_df.columns) > 0:
            for columnName in custom_df.columns:
                custom_df.drop(columnName, axis = 1, inplace=True)
            custom_df.drop(custom_df.index, inplace=True)

        # Repetir el proceso con la dataframe custom de datos limpios 
        if len(clean_custom_df.columns) > 0:
            for columnName in clean_custom_df.columns:
                clean_custom_df.drop(columnName, axis = 1, inplace=True)
            clean_custom_df.drop(clean_custom_df.index, inplace=True)
    
    # RESETEAR VALORES DE CUSTOM ALGORITMOS Y SUS DATAFRAMES DE IMPORTANCIAS DE CARACTERÍSTICAS
    def reset_custom_values_and_feat_imp_dfs():
        custom_correlation_execution_counter.set(0)
        custom_test_split_done.set(False)
        
        custom_empty_dec_tree_feature_importance_df()
        custom_decision_tree_execution_counter.set(0)

        custom_empty_ran_forest_feature_importance_df()
        custom_random_forest_execution_counter.set(0)

        custom_empty_log_reg_feature_importance_df()
        custom_logistic_regression_execution_counter.set(0)

    # VACIAR LA DATAFRAME DE IMPORTANCIAS DE CARACTERÍSTICAS DE ÁRBOL DE DECISIÓN
    def custom_empty_dec_tree_feature_importance_df():
        if custom_decision_tree_execution_counter.get() > 0:
            custom_dec_tree_feat_imp_df.drop(["Característica", "Valor"], axis = 1, inplace=True)
            custom_dec_tree_feat_imp_df.drop(custom_dec_tree_feat_imp_df.index, inplace=True)
    
    # VACIAR LA DATAFRAME DE IMPORTANCIAS DE CARACTERÍSTICAS DE BOSQUE ALEATORIO
    def custom_empty_ran_forest_feature_importance_df():
        if custom_random_forest_execution_counter.get() > 0:
            custom_ran_forest_feat_imp_df.drop(["Característica", "Valor"], axis = 1, inplace=True)
            custom_ran_forest_feat_imp_df.drop(custom_ran_forest_feat_imp_df.index, inplace=True)

    # VACIAR LA DATAFRAME DE IMPORTANCIAS DE CARACTERÍSTICAS DE REGRESIÓN LOGÍSTICA
    def custom_empty_log_reg_feature_importance_df():
        if custom_logistic_regression_execution_counter.get() > 0:
            custom_log_reg_feat_imp_df.drop(["Característica", "Valor"], axis = 1, inplace=True)
            custom_log_reg_feat_imp_df.drop(custom_log_reg_feat_imp_df.index, inplace=True)

    # CERRAR Y ELIMINAR LOS ELEMENTOS DE UI CUSTOM NECESARIOS AL CAMBIAR DATAFRAME
    def remove_and_update_ui_elements_when_new_custom_df():
        custom_rows_deleted.set(0)
        ui.remove_ui("#rows-deleted-custom")
        ui.remove_ui("#inserted-custom-clean-table")
        ui.remove_ui("#inserted-custom-clean-table-types")
        ui.remove_ui("#custom-clean-hist-plot")
        ui.remove_ui("#custom-clean-hist-plot-div-outcome")

        ui.update_switch("view_custom_clean_table", value=False)
        ui.update_switch("view_custom_clean_table_types", value=False)
        ui.update_switch("view_custom_clean_table_histogram", value=False)
        ui.update_switch("view_custom_clean_table_histogram_div_outcome", value=False)
        
        ui.remove_ui("#custom-correlation-plot")
        ui.update_switch("custom_view_correlation", value=False)
        
        ui.remove_ui("#custom-var-imp-dec-tree-plot")
        ui.update_switch("custom_view_variable_importance_dec_tree", value=False)
        
        ui.remove_ui("#custom-dec-tree-conf-mat-train")
        ui.remove_ui("#custom-dec-tree-conf-mat-test")
        ui.update_switch("custom_conf_mat_dec_tree_switch", value=False)
        
        ui.remove_ui("#custom-dec-tree-view-img")
        ui.update_switch("custom_view_tree_dec_tree_switch", value=False)
        
        ui.remove_ui("#custom-var-imp-ran-forest-plot")
        ui.update_switch("view_variable_importance_custom_ran_forest", value=False)
        
        ui.remove_ui("#custom-ran-forest-conf-mat-train")
        ui.remove_ui("#custom-ran-forest-conf-mat-test")
        ui.update_switch("conf_mat_custom_ran_forest_switch", value=False)
        
        ui.remove_ui("#custom-ran-forest-view-img")
        ui.remove_ui("#custom-ran-forest-view-img-foot")
        ui.update_switch("view_tree_custom_ran_forest_switch", value=False)
        
        ui.remove_ui("#custom-var-imp-log-reg-plot")
        ui.update_switch("custom_view_variable_importance_log_reg", value=False)
        
        ui.remove_ui("#custom-log-reg-conf-mat-train")
        ui.remove_ui("#custom-log-reg-conf-mat-test")
        ui.update_switch("custom_conf_mat_log_reg_switch", value=False)

        # Resetear manualmente el select del número de árbol a mostrar de random forest
        ui.update_select("custom_view_tree_ran_forest_number", choices=empty_column_dict)

    # RESETEAR TODAS LAS VARIABLES REACTIVAS DE LOS ALGORITMOS CUSTOM
    def reset_all_custom_algoritms_reactive_values():
        #Resetear todas las variables reactivas restantes:
        #Custom DecTree:
        reset_results_custom_dec_tree()
        #Custom Random forest:
        reset_results_custom_ran_forest()
        #Custom Logistic Regression:
        reset_results_custom_log_reg()

    # RESETEAR TODAS LAS VARIABLES REACTIVAS DE ÁRBOL DE DECISIÓN CUSTOM
    def reset_results_custom_dec_tree():
        custom_accuracy_decTree.set(-1)
        custom_recall_decTree.set(-1)
        custom_precision_decTree.set(-1)
        custom_f1_decTree.set(-1)

        custom_accuracy_decTree_test.set(-1)
        custom_recall_decTree_test.set(-1)
        custom_precision_decTree_test.set(-1)
        custom_f1_decTree_test.set(-1)

    # RESETEAR TODAS LAS VARIABLES REACTIVAS DE BOSQUE ALEATORIO CUSTOM
    def reset_results_custom_ran_forest():
        custom_accuracy_ranForest.set(-1)
        custom_recall_ranForest.set(-1)
        custom_precision_ranForest.set(-1)
        custom_f1_ranForest.set(-1)

        custom_accuracy_ranForest_test.set(-1)
        custom_recall_ranForest_test.set(-1)
        custom_precision_ranForest_test.set(-1)
        custom_f1_ranForest_test.set(-1)

    # RESETEAR TODAS LAS VARIABLES REACTIVAS DE REGRESIÓN LOGÍSTICA CUSTOM
    def reset_results_custom_log_reg():
        custom_accuracy_logReg.set(-1)
        custom_recall_logReg.set(-1)
        custom_precision_logReg.set(-1)
        custom_f1_logReg.set(-1)

        custom_accuracy_logReg_test.set(-1)
        custom_recall_logReg_test.set(-1)
        custom_precision_logReg_test.set(-1)
        custom_f1_logReg_test.set(-1)


#################################### TABLAS ##################################################

    #### DATAFRAME CUSTOM SIN MODIFICAR
    @output
    @render.table(index=True)
    def customTable():
        # Variables a las que reaccionar:
        custom_df_counter.get()

        return custom_df
    
    # TIPOS DE DATOS TABLA CUSTOM
    @output
    @render.table
    def customTableTypes():
        # Variables a las que reaccionar:
        custom_df_counter.get()
        
        custom_table_types = custom_df.dtypes.to_frame().reset_index().transpose().reset_index(drop=True)
        headers = custom_table_types.iloc[0]
        custom_table_types  = pd.DataFrame(custom_table_types.values[1:], columns=headers)
        custom_table_types = custom_table_types.replace(['int64', 'int32', 'float64', 'object'],['numérico', 'numérico', 'numérico', 'categórico'])
        return custom_table_types
    

#################################### EFECTOS REACTIVOS #######################################

    # MOSTRAR LA TABLA DE DATOS CUSTOM
    @reactive.Effect
    def _():
        # Variables a las que reaccionar:
        custom_df_counter.get()
        
        custom_table_switch = input.view_custom_table()
        if custom_table_switch == True:
            if len(custom_df) > 0:
                ui.remove_ui("#inserted-custom-table")
                custom_table = ui.output_table("customTable", style = "overflow-x:scroll; height:260px; overflow-y:auto;"),
                ui.insert_ui(
                    ui.div({"id": "inserted-custom-table"}, custom_table, style = "width:100%; overflow-x:auto;"),
                    selector="#custom-table",
                    where="beforeEnd",
                )
            else:
                ui.remove_ui("#inserted-custom-table")
                ui.insert_ui(
                    ui.div({"id": "inserted-custom-table"}, custom_missing_data_table_warning_ui("custom_tool_warnings_no_data")),
                    selector="#custom-table",
                    where="beforeEnd",
                )
        else:
            ui.remove_ui("#inserted-custom-table")

    # MOSTRAR LOS TIPOS DE DATOS DE LA TABLA DE DATOS CUSTOM
    @reactive.Effect
    def _():
        # Variables a las que reaccionar:
        custom_df_counter.get()

        custom_table_types_switch = input.view_custom_table_types()
        if custom_table_types_switch == True:
            if len(custom_df) > 0:
                ui.remove_ui("#inserted-custom-table-types")
                custom_table_types = ui.output_table("customTableTypes", style = "overflow-x:auto;"),
                ui.insert_ui(
                    ui.div({"id": "inserted-custom-table-types"}, custom_table_types),
                    selector="#custom-table-types",
                    where="beforeEnd",
                )
            else:
                ui.remove_ui("#inserted-custom-table-types")
                ui.insert_ui(
                    ui.div({"id": "inserted-custom-table-types"}, custom_missing_data_types_warning_ui("custom_tool_warnings_no_data")),
                    selector="#custom-table-types",
                    where="beforeEnd"
                )
        else:
            ui.remove_ui("#inserted-custom-table-types")

    # MOSTRAR EL HISTOGRAMA DE LA TABLA DE DATOS CUSTOM
    @reactive.Effect
    def _():
        # Variables a las que reaccionar:
        custom_df_counter.get()

        custom_hist_switch = input.view_custom_table_histogram()
        if custom_hist_switch == True:
            if len(custom_df) > 0:
                ui.remove_ui("#custom-hist-plot")
                custom_hist_plot = output_widget("widget_customObservation")
                ui.insert_ui(
                    ui.div({"id": "custom-hist-plot"}, custom_hist_plot, style = "width:100%; overflow-x:auto;"),
                    selector="#custom-table-histogram",
                    where="beforeEnd",
                )
            else:
                ui.remove_ui("#custom-hist-plot")
                ui.insert_ui(
                    ui.div({"id": "custom-hist-plot"}, custom_missing_data_histogram_warning_ui("custom_tool_warnings_no_data")),
                    selector="#custom-table-histogram",
                    where="beforeEnd",
                )
        else:
            ui.remove_ui("#custom-hist-plot")


#################################### WIDGETS #################################################

    # WIDGET HISTOGRAMA DE DATOS DE LA DATAFRAME CUSTOM
    @output
    @render_widget
    def widget_customObservation():
        # Variables a las que reaccionar:
        custom_df_counter.get()
        
        aux_df = custom_df.copy()
        selected_cols = aux_df.columns
        for columnName in aux_df.columns:
            if not pd.api.types.is_numeric_dtype(aux_df[columnName]):
                aux_df[columnName] = aux_df[columnName].astype(str)
        
        # Mantener el diseño de colores del resto de widgets
        num_colors = len(selected_cols)
        color_array = px.colors.sample_colorscale("viridis_r", [n/(num_colors - 1) for n in range(num_colors)])

        # Dividir los subplots en columnas
        subplot_cols_number = 4
        subplot_rows_number=math.ceil(len(selected_cols) / subplot_cols_number)
        
        fig = make_subplots(rows=subplot_rows_number, cols=subplot_cols_number, 
                            subplot_titles=selected_cols,
        )
        
        for idx,curr_col in enumerate(selected_cols):
            fig.add_trace(go.Histogram(x=aux_df[curr_col], opacity=0.7, name=curr_col, marker_color=color_array[idx]),
                          row=math.floor(idx/subplot_cols_number)+1, col=(idx%subplot_cols_number)+1)
        
        fig.update_layout(autosize=True,
                          barmode='overlay',
                          showlegend=True,
                          height=subplot_rows_number*180,
                          margin=dict(l=20, r=20, t=40, b=20))

        fig.update_traces(hovertemplate='%{y}<br>Rango: %{x}')  
        
        return fig

    
##############################################################################################
############################## CUSTOM: SELECTOR VARIABLE A PREDECIR ##########################
##############################################################################################

#################################### EFECTOS REACTIVOS #######################################

    # CONVERTIR LA VARIABLE A PREDECIR A NUMÉRICA
    @reactive.Effect
    @reactive.event(input.convert_custom_outcome)
    def _():
        column_outcome = input.outcomeSelectorCustom()
        if column_outcome != None:
            if pd.api.types.is_numeric_dtype(clean_custom_df[column_outcome]):
                #print("Los datos de " + column_outcome + " ya son numéricos")
                return
            if column_outcome in clean_custom_df.columns:
                clean_custom_df[column_outcome] = pd.factorize(clean_custom_df[column_outcome])[0]
                #print("Datos de " + column_outcome + " convertidos a numéricos")

    # CONVERTIR LA VARIABLE A PREDECIR A 0 SI = 0 Y 1 SI ES < 0
    @reactive.Effect
    @reactive.event(input.convert_custom_outcome_higher_0)
    def _():
        column_outcome = input.outcomeSelectorCustom()
        if column_outcome != None:
            if pd.api.types.is_numeric_dtype(clean_custom_df[column_outcome]):
                clean_custom_df[column_outcome] = np.where(clean_custom_df[column_outcome]>0,1,0)
                #print("Los datos de " + column_outcome + " se ha convertido a numéricos 0-1")


#################################### UPDATES Y OTROS #########################################

    # ACTUALIZAR EL SELECTOR DE LA VARIABLE A PREDECIR
    def update_outcomeSelector_custom():
        column_dict = {}
        for col in clean_custom_df.columns:
            column_dict[col] = col
        ui.update_select("outcomeSelectorCustom", choices=column_dict, selected=None)


##############################################################################################
############################## CUSTOM: LIMPIEZA DE LOS DATOS #################################
##############################################################################################

#################################### TABLAS ##################################################

    # TABLA DE DATOS CUSTOM LIMPIA
    @output
    @render.table(index=True)
    def customcleanTable():
        # Variables a las que reaccionar:
        custom_df_counter.get()
        input.convert_custom_outcome()
        input.convert_custom_outcome_higher_0()
        input.drop_selected_column_custom_clean()
        input.fillNA_selected_column_custom_clean()
        input.convert_numeric_selected_column_custom_clean()
        input.fillNA_all_column_custom_clean()
        input.convert_numeric_all_column_custom_clean()
        input.convert_custom_text_selected_column_custom_clean()
        input.convert_custom_numeric_selected_column_custom_clean()
        input.drop_row_selected_custom_clean()
        input.drop_all_NA_rows_custom_clean()
        custom_correlation_execution_counter.get()
        
        return clean_custom_df
    
    # TABLA DE TIPOS DE DATOS DE LA DATAFRAME CUSTOM LIMPIA
    @output
    @render.table
    def customCleanTableTypes():
        # Variables a las que reaccionar:
        custom_df_counter.get()
        input.convert_custom_outcome()
        input.convert_custom_outcome_higher_0()
        input.drop_selected_column_custom_clean()
        input.fillNA_selected_column_custom_clean()
        input.convert_numeric_selected_column_custom_clean()
        input.fillNA_all_column_custom_clean()
        input.convert_numeric_all_column_custom_clean()
        input.convert_custom_text_selected_column_custom_clean()
        input.convert_custom_numeric_selected_column_custom_clean()
        input.drop_row_selected_custom_clean()
        input.drop_all_NA_rows_custom_clean()
        custom_correlation_execution_counter.get()

        clean_custom_table_types = clean_custom_df.dtypes.to_frame().reset_index().transpose().reset_index(drop=True)
        headers = clean_custom_table_types.iloc[0]
        clean_custom_table_types  = pd.DataFrame(clean_custom_table_types.values[1:], columns=headers)
        clean_custom_table_types = clean_custom_table_types.replace(['int64', 'int32', 'float64', 'object'],['numérico', 'numérico', 'numérico', 'categórico'])
        return clean_custom_table_types

#################################### EFECTOS REACTIVOS #######################################

# ELIMINAR COLUMNA SELECCIONADA
    @reactive.Effect
    @reactive.event(input.drop_selected_column_custom_clean)
    def _():
        if  len(custom_df) > 0:
            column_selected = input.dropIdSelectorCustom()
            if column_selected in clean_custom_df.columns:
                outcome_column_name = input.outcomeSelectorCustom()
                clean_custom_df.drop(column_selected,axis=1,inplace=True)
                update_all_selectors_custom()
                ui.update_select("outcomeSelectorCustom", selected=outcome_column_name)
    
    # RELLENAR DE 0 VALORES NULOS DE LA COLUMNA SELECCIONADA
    @reactive.Effect
    @reactive.event(input.fillNA_selected_column_custom_clean)
    def _():
        if  len(custom_df) > 0:
            column_selected = input.dropIdSelectorCustom()
            if column_selected in clean_custom_df.columns:
                clean_custom_df[column_selected].fillna(0, inplace=True)

    # CONVERTIR A NUMÉRICOS LOS VALORES NULOS DE LA COLUMNA SELECCIONADA
    @reactive.Effect
    @reactive.event(input.convert_numeric_selected_column_custom_clean)
    def _():
        if  len(custom_df) > 0:
            column_selected = input.dropIdSelectorCustom()
            if pd.api.types.is_numeric_dtype(clean_custom_df[column_selected]):
                return
            if column_selected in clean_custom_df.columns:
                clean_custom_df[column_selected] = pd.factorize(clean_custom_df[column_selected])[0]

    # RELLENAR DE 0 VALORES NULOS DE TODA LA DATAFRAME
    @reactive.Effect
    @reactive.event(input.fillNA_all_column_custom_clean)
    def _():
        if  len(custom_df) > 0:
            clean_custom_df.fillna(0, inplace=True)

    # CONVERTIR A NUMÉRICOS LOS VALORES NULOS DE TODA LA DATAFRAME
    @reactive.Effect
    @reactive.event(input.convert_numeric_all_column_custom_clean)
    def _():
        if  len(custom_df) > 0:
            outcome_column_name = input.outcomeSelectorCustom()

            for columnName in clean_custom_df.columns:
                if columnName != outcome_column_name:
                    if not pd.api.types.is_numeric_dtype(clean_custom_df[columnName]):
                        clean_custom_df[columnName] = pd.factorize(clean_custom_df[columnName])[0]

    # CONVERTIR A NÚMERO SELECCIONADO LOS VALORES NULOS DE LA COLUMNA SELECCIONADA
    @reactive.Effect
    @reactive.event(input.convert_custom_numeric_selected_column_custom_clean)
    def _():
        if  len(custom_df) > 0:
            column_selected = input.dropIdSelectorCustom()
            new_value = input.numeric_selected_column_custom_clean()
            if new_value == None:
                return
            
            clean_custom_df[column_selected].fillna(new_value, inplace=True)
            
    # CONVERTIR A STRING SELECCIONADA LOS VALORES NULOS DE LA COLUMNA SELECCIONADA
    @reactive.Effect
    @reactive.event(input.convert_custom_text_selected_column_custom_clean)
    def _():
        if  len(custom_df) > 0:
            column_selected = input.dropIdSelectorCustom()
            new_value = input.text_selected_column_custom_clean()
            if new_value == None:
                return
            
            clean_custom_df[column_selected].fillna(new_value, inplace=True)

    # ACTUALIZAR EL SELECTOR DE COLUMNA
    @reactive.Effect
    def update_dropIdSelector_custom():
        # Variables a las que reaccionar:
        input.drop_selected_column_custom_clean()
        
        outcome_name = input.outcomeSelectorCustom()
        column_dict = {}
        for col in clean_custom_df.columns:
            if col != outcome_name:
                column_dict[col] = col
        ui.update_select("dropIdSelectorCustom", choices=column_dict, selected=None)

    # ELIMINAR FILA SELECCIONADA
    @reactive.Effect
    @reactive.event(input.drop_row_selected_custom_clean)
    def _():
        if  len(custom_df) > 0:
            row_index = input.selected_row_to_drop_custom_clean()
            if row_index >= 0 and row_index < len(clean_custom_df):
                clean_custom_df.drop(row_index, inplace=True)
                clean_custom_df.reset_index(drop=True, inplace=True)
                update_dropRowSelector_custom()

    # ELIMINAR TODAS LAS FILAS CON VALORES NULOS
    @reactive.Effect
    @reactive.event(input.drop_all_NA_rows_custom_clean)
    def _():
        if  len(custom_df) > 0:
            deleted_rows_counter = 0
            for idx, row in clean_custom_df.iterrows():
                if row.isnull().values.any():
                    clean_custom_df.drop(idx, inplace=True)
                    deleted_rows_counter += 1
                    #print("Fila eliminada: " + str(idx))
            clean_custom_df.reset_index(drop=True, inplace=True)
            update_dropRowSelector_custom()
            custom_rows_deleted.set(deleted_rows_counter)
            ui.remove_ui("#rows-deleted-custom")
            rows_deleted = ui.output_text("rows_deleted_txt"),
            ui.insert_ui(
                ui.div({"id": "rows-deleted-custom"}, rows_deleted, style="color:#006ee6; font-style:italic; margin-top:20px; padding: 10px; background: #f7f7f7; border-radius: 10px;"),
                selector="#custom-clean-row-buttons",
                where="beforeEnd",
            )

    # MOSTRAR LA TABLA DE DATOS LIMPIA
    @reactive.Effect
    def _():
        # Variables a las que reaccionar:
        custom_df_counter.get()

        custom_clean_table_switch = input.view_custom_clean_table()
        if custom_clean_table_switch == True:
            if len(custom_df) > 0:
                ui.remove_ui("#inserted-custom-clean-table")
                custom_clean_table = ui.output_table("customcleanTable", style = "overflow-x:scroll; height:260px; overflow-y:auto;"),
                ui.insert_ui(
                    ui.div({"id": "inserted-custom-clean-table"}, custom_clean_table),
                    selector="#custom-clean-table",
                    where="beforeEnd",
                )

            else:
                ui.remove_ui("#inserted-custom-clean-table")
                ui.insert_ui(
                    ui.div({"id": "inserted-custom-clean-table"}, custom_missing_data_clean_table_warning_ui("custom_tool_warnings_general")),
                    selector="#custom-clean-table",
                    where="beforeEnd",
                )
        else:
            ui.remove_ui("#inserted-custom-clean-table")

    # MOSTRAR LOS TIPOS DE DATOS DE LOS DATOS LIMPIOS
    @reactive.Effect
    def _():
        # Variables a las que reaccionar:
        custom_df_counter.get()

        custom_clean_table_types_switch = input.view_custom_clean_table_types()
        if custom_clean_table_types_switch == True:
            if len(custom_df) > 0:
                ui.remove_ui("#inserted-custom-clean-table-types")
                custom_table_types = ui.output_table("customCleanTableTypes", style = "overflow-x:auto;"),
                ui.insert_ui(
                    ui.div({"id": "inserted-custom-clean-table-types"}, custom_table_types),
                    selector="#custom-clean-table-types",
                    where="beforeEnd",
                )
            else:
                ui.remove_ui("#inserted-custom-clean-table-types")
                ui.insert_ui(
                    ui.div({"id": "inserted-custom-clean-table-types"}, custom_missing_data_clean_types_warning_ui("custom_tool_warnings_general")),
                    selector="#custom-clean-table-types",
                    where="beforeEnd"
                )
        else:
            ui.remove_ui("#inserted-custom-clean-table-types")

    # MOSTRAR EL HISTOGRAMA DE LOS DATOS LIMPIOS
    @reactive.Effect
    def _():
        # Variables a las que reaccionar:
        custom_df_counter.get()

        custom_clean_hist_switch = input.view_custom_clean_table_histogram()
        if custom_clean_hist_switch == True:
            if len(custom_df) > 0:
                ui.remove_ui("#custom-clean-hist-plot")
                custom_hist_plot = output_widget("widget_custom_clean_observation")
                ui.insert_ui(
                    ui.div({"id": "custom-clean-hist-plot"}, custom_hist_plot, style = "width:100%; overflow-x:auto;"),
                    selector="#custom-clean-table-histogram",
                    where="beforeEnd",
                )
            else:
                ui.remove_ui("#custom-clean-hist-plot")
                ui.insert_ui(
                    ui.div({"id": "custom-clean-hist-plot"}, custom_missing_data_clean_hist_warning_ui("custom_tool_warnings_general")),
                    selector="#custom-clean-table-histogram",
                    where="beforeEnd",
                )
        else:
            ui.remove_ui("#custom-clean-hist-plot")

    # MOSTRAR EL HISTOGRAMA DE LOS DATOS LIMPIOS DIVIDIDOS POR LA VARIABLE A PREDECIR
    @reactive.Effect
    def _():
        # Variables a las que reaccionar:
        custom_df_counter.get()

        outcome_name = input.outcomeSelectorCustom()
        custom_clean_hist_switch_div_outcome = input.view_custom_clean_table_histogram_div_outcome()
        if custom_clean_hist_switch_div_outcome == True:
            if outcome_name in clean_custom_df.columns and len(clean_custom_df[outcome_name].unique()) > 5:
                ui.remove_ui("#custom-clean-hist-plot-div-outcome")
                ui.insert_ui(
                    ui.div({"id": "custom-clean-hist-plot-div-outcome"}, custom_too_many_unique_clean_hist_div_outcome_warning_ui("custom_tool_warnings_general")),
                    selector="#custom-clean-table-histogram-div-outcome",
                    where="beforeEnd",
                )
            elif len(custom_df) > 0:
                ui.remove_ui("#custom-clean-hist-plot-div-outcome")
                custom_hist_plot = output_widget("widget_custom_clean_observation_div_outcome")
                ui.insert_ui(
                    ui.div({"id": "custom-clean-hist-plot-div-outcome"}, custom_hist_plot, style = "width:100%; overflow-x:auto;"),
                    selector="#custom-clean-table-histogram-div-outcome",
                    where="beforeEnd",
                )
            else:
                ui.remove_ui("#custom-clean-hist-plot-div-outcome")
                ui.insert_ui(
                    ui.div({"id": "custom-clean-hist-plot-div-outcome"}, custom_missing_data_clean_hist_div_outcome_warning_ui("custom_tool_warnings_general")),
                    selector="#custom-clean-table-histogram-div-outcome",
                    where="beforeEnd",
                )
        else:
            ui.remove_ui("#custom-clean-hist-plot-div-outcome")


#################################### WIDGETS ################################################# 

    # WIDGET HISTOGRAMA DE DATOS CUSTOM LIMPIOS
    @output
    @render_widget
    def widget_custom_clean_observation():
        # Variables a las que reaccionar:
        custom_df_counter.get()
        input.convert_custom_outcome()
        input.convert_custom_outcome_higher_0()
        input.drop_selected_column_custom_clean()
        input.fillNA_selected_column_custom_clean()
        input.convert_numeric_selected_column_custom_clean()
        input.fillNA_all_column_custom_clean()
        input.convert_numeric_all_column_custom_clean()
        input.convert_custom_text_selected_column_custom_clean()
        input.convert_custom_numeric_selected_column_custom_clean()
        input.drop_row_selected_custom_clean()
        input.drop_all_NA_rows_custom_clean()
        custom_correlation_execution_counter.get()

        aux_df = clean_custom_df.copy()
        selected_cols = aux_df.columns
        for columnName in aux_df.columns:
            if not pd.api.types.is_numeric_dtype(aux_df[columnName]):
                aux_df[columnName] = aux_df[columnName].astype(str)
        
        # Mantener el diseño de colores del resto de UI
        num_colors = len(selected_cols)
        color_array = []
        if num_colors == 2:
            color_array = ['#440154','#5ec962']
        else:
            color_array = px.colors.sample_colorscale("viridis_r", [n/(num_colors - 1) for n in range(num_colors)])

        # Dividir los datos en subplots
        subplot_cols_number = 4
        subplot_rows_number=math.ceil(len(selected_cols) / subplot_cols_number)
        
        fig = make_subplots(rows=subplot_rows_number, cols=subplot_cols_number, 
                            subplot_titles=selected_cols,
        )
        
        for idx,curr_col in enumerate(selected_cols):
            fig.add_trace(go.Histogram(x=aux_df[curr_col], opacity=0.7, name=curr_col, marker_color=color_array[idx]),
                          row=math.floor(idx/subplot_cols_number)+1, col=(idx%subplot_cols_number)+1)
        
        fig.update_layout(autosize=True,
                          barmode='overlay',
                          showlegend=True,
                          height=subplot_rows_number*180,
                          margin=dict(l=20, r=20, t=40, b=20))

        fig.update_traces(hovertemplate='%{y}<br>Rango: %{x}')  
        
        return fig
    
    # WIDGET HISTOGRAMA DE DATOS CUSTOM LIMPIOS SEPARADOS POR LA VARIABLE A PREDECIR
    @output
    @render_widget
    def widget_custom_clean_observation_div_outcome():
        # Variables a las que reaccionar:
        custom_df_counter.get()
        input.convert_custom_outcome()
        input.convert_custom_outcome_higher_0()
        input.drop_selected_column_custom_clean()
        input.fillNA_selected_column_custom_clean()
        input.convert_numeric_selected_column_custom_clean()
        input.fillNA_all_column_custom_clean()
        input.convert_numeric_all_column_custom_clean()
        input.convert_custom_text_selected_column_custom_clean()
        input.convert_custom_numeric_selected_column_custom_clean()
        input.drop_row_selected_custom_clean()
        input.drop_all_NA_rows_custom_clean()
        custom_correlation_execution_counter.get()

        outcome_name = input.outcomeSelectorCustom()

        if outcome_name not in clean_custom_df.columns:
            return go.Figure()

        selected_cols = list()
        for columnName in clean_custom_df.columns:
            if pd.api.types.is_numeric_dtype(clean_custom_df[columnName]) and columnName != outcome_name:
                selected_cols.append(columnName)
        
        # Crear las diferentes dataframes para los diferentes valores de la variable a predecir:
        unique_outcome_values = clean_custom_df[outcome_name].unique()
        unique_outcome_values_lenght = len(unique_outcome_values)
        
        divided_dataframes_list = list()
        for idx, value in enumerate(unique_outcome_values):
            divided_dataframes_list.append(pd.DataFrame())
            divided_dataframes_list[idx]=clean_custom_df[clean_custom_df[outcome_name] == value]

        # Mantener el diseño de colores del resto de la UI
        num_colors = unique_outcome_values_lenght
        color_array = []
        if num_colors == 2:
            color_array = ['#440154','#5ec962']
        else:
            color_array = px.colors.sample_colorscale("viridis_r", [n/(num_colors - 1) for n in range(num_colors)])

        # Dividir los subplots en 4 columnas
        subplot_cols_number = 4
        subplot_rows_number=math.ceil(len(selected_cols) / subplot_cols_number)
        
        fig = make_subplots(rows=subplot_rows_number, cols=subplot_cols_number, 
                            subplot_titles=selected_cols,
        )
        
        for idx, curr_col in enumerate(selected_cols):
            for idx2, value in enumerate(unique_outcome_values):
                fig.add_trace(go.Histogram(x=divided_dataframes_list[idx2][curr_col], 
                                           name = str(value), marker_color=color_array[idx2], 
                                           opacity=0.7, legendgroup=str(value), showlegend=idx==0),
                                row=math.floor(idx/subplot_cols_number)+1, col=(idx%subplot_cols_number)+1)
        
        fig.update_layout(autosize=True,
                          barmode='overlay',
                          showlegend=True,
                          height=subplot_rows_number*180,
                          margin=dict(l=20, r=20, t=40, b=20))

        fig.update_traces(hovertemplate='%{y}<br>Rango: %{x}') 
        return fig
    

#################################### TEXTOS ##################################################
    
    # INDICADOR FILAS ELIMINADAS
    @output
    @render.text
    def rows_deleted_txt():
        return "Filas eliminadas: " + str(custom_rows_deleted.get())
    

#################################### UPDATES Y OTROS #########################################

    def update_dropRowSelector_custom():
        ui.update_numeric("selected_row_to_drop_custom_clean", max=len(clean_custom_df)-1)

    
##############################################################################################
############################## CUSTOM: CORRELACIÓN DE LOS DATOS ##############################
##############################################################################################

#################################### EFECTOS REACTIVOS #######################################

    # ELIMINAR COLUMNAS CON CORRELACIÓN ALTA
    @reactive.Effect
    @reactive.event(input.custom_drop_correlation)
    def _():
        outcome_column_name = input.outcomeSelectorCustom()
        aux_df = pd.DataFrame()
        
        for columnName in clean_custom_df.columns:
            if columnName != outcome_column_name:
                if pd.api.types.is_numeric_dtype(clean_custom_df[columnName]):
                    aux_df[columnName] = clean_custom_df[columnName]

        custom_correlation_map = aux_df.corr().abs()
        upper_tri = custom_correlation_map.where(np.triu(np.ones(custom_correlation_map.shape),k=1).astype(bool))
        columns_to_drop = [column for column in upper_tri.columns if any(upper_tri[column] >= input.custom_maximum_correlation())]
        clean_custom_df.drop(columns_to_drop, axis=1, inplace=True)
        custom_correlation_execution_counter.set(custom_correlation_execution_counter.get() + 1)
        update_all_selectors_custom()
        ui.update_select("outcomeSelectorCustom", selected=outcome_column_name)
        custom_correlation_switch = input.custom_view_correlation()
        if custom_correlation_switch == True:
            ui.remove_ui("#custom-correlation-plot")
            custom_correlation_plot = output_widget("custom_widget_correlation")
            ui.insert_ui(
                ui.div({"id": "custom-correlation-plot"}, custom_correlation_plot, style = "width:100%; height:1000px; overflow-x:auto; overflow-y:auto;"),
                selector="#custom-correlation",
                where="beforeEnd",
            )

    # MOSTRAR EL WIDGET DE CORRELACIÓN
    @reactive.Effect
    def _():
        # Variables a las que reaccionar:
        custom_df_counter.get()

        custom_correlation_switch = input.custom_view_correlation()
        if custom_correlation_switch == True:
            if len(custom_df) > 0:
                if input.outcomeSelectorCustom() != None and input.outcomeSelectorCustom() in clean_custom_df.columns and pd.api.types.is_numeric_dtype(clean_custom_df[input.outcomeSelectorCustom()]):
                    ui.remove_ui("#custom-correlation-plot")
                    custom_correlation_plot = output_widget("custom_widget_correlation")
                    ui.insert_ui(
                        ui.div({"id": "custom-correlation-plot"}, custom_correlation_plot, style = "width:100%; overflow-x:auto; overflow-y:auto;"),
                        selector="#custom-correlation",
                        where="beforeEnd",
                    )
                else:
                    ui.remove_ui("#custom-correlation-plot")
                    custom_correlation_warning = ui.output_text("custom_correlation_warning_txt"),
                    ui.insert_ui(
                        ui.div({"id": "custom-correlation-plot"}, custom_correlation_warning_ui("custom_tool_warnings_general")),
                        selector="#custom-correlation",
                        where="beforeEnd",
                    )
            else:
                ui.remove_ui("#custom-correlation-plot")
                ui.insert_ui(
                    ui.div({"id": "custom-correlation-plot"}, custom_correlation_no_data_warning_ui("custom_tool_warnings_general")),
                    selector="#custom-correlation",
                    where="beforeEnd",
                )
        else:
            ui.remove_ui("#custom-correlation-plot")


#################################### WIDGETS ################################################# 

    # WIDGET DE CORRELACION CUSTOM
    @output
    @render_widget
    def custom_widget_correlation():
        # Variables a las que reaccionar:
        input.outcomeSelectorCustom()
        custom_df_counter.get()
        input.convert_custom_outcome()
        input.convert_custom_outcome_higher_0()
        input.drop_selected_column_custom_clean()
        input.fillNA_selected_column_custom_clean()
        input.convert_numeric_selected_column_custom_clean()
        input.fillNA_all_column_custom_clean()
        input.convert_numeric_all_column_custom_clean()
        input.convert_custom_text_selected_column_custom_clean()
        input.convert_custom_numeric_selected_column_custom_clean()
        input.drop_row_selected_custom_clean()
        input.drop_all_NA_rows_custom_clean()
        custom_correlation_execution_counter.get()

        outcome_column_name = input.outcomeSelectorCustom()
        if outcome_column_name == None or outcome_column_name not in clean_custom_df.columns:
            return go.Figure()

        if not pd.api.types.is_numeric_dtype(clean_custom_df[outcome_column_name]):
            return go.Figure()

        aux_df = pd.DataFrame()
        aux_df[outcome_column_name] = clean_custom_df[outcome_column_name]
        
        for columnName in clean_custom_df.columns:
            if columnName != outcome_column_name:
                if pd.api.types.is_numeric_dtype(clean_custom_df[columnName]):
                    aux_df[columnName] = clean_custom_df[columnName]

        correlation_map = aux_df.corr().round(decimals=3)
        fig = go.Figure(data=[go.Heatmap(z=correlation_map,
                                        x = correlation_map.columns.values,
                                        y = correlation_map.columns.values,
                                        xgap = 1,
                                        ygap = 1,
                                        colorscale=px.colors.sequential.Viridis_r,
                                        name="") 
                                        #zauto=False, zmax=1, zmin=-0.5
        ])

        fig.update_layout(autosize=True,
                          height=min(80*len(aux_df.columns), 1000),
                          yaxis=dict(scaleanchor = 'x'),
                          margin=dict(l=20, r=20, t=40, b=20),)
        
        fig = fig.update_traces(text=correlation_map,
                                texttemplate="%{text}",
                                hovertemplate='%{x} - %{y}<br>Correlación: %{z}')
        
        fig.update_yaxes(autorange="reversed")
        
        return fig



##############################################################################################
############################## CUSTOM: ÁLGORITMOS DE PREDICCIÓN ##############################
##############################################################################################

    # REALIZAR LA DIVISIÓN DE ENTRENAMIENTO Y TESTEO (DUMB STEP :P)
    @reactive.Effect
    @reactive.event(input.custom_make_test_split)
    def _():
        custom_test_split_done.set(True)


##############################################################################################
############################## CUSTOM: ÁRBOL DE DECISIÓN #####################################
##############################################################################################

#################################### IMPORTANTES #############################################

    # COMPROBACIONES PREVIAS ÁRBOL DE DECISIÓN
    def custom_dec_tree_previous_checks(custom_test_size_split, df_len, outcome_column_name):
        if len(custom_df) <= 0:
            ui.insert_ui(
                ui.div({"id": "custom-dec-tree-warning"}, custom_no_data_warning_ui("custom_tool_warnings_dec_tree")),
                selector="#custom_dec_tree_generator",
                where="beforeEnd",
            )
            return True

        if not pd.api.types.is_numeric_dtype(clean_custom_df[outcome_column_name]):
            ui.insert_ui(
                ui.div({"id": "custom-dec-tree-warning"}, custom_outcome_warning_ui("custom_tool_warnings_dec_tree")),
                selector="#custom_dec_tree_generator",
                where="beforeEnd",
            )
            return True

        if custom_test_split_done.get() == False:
            ui.insert_ui(
                ui.div({"id": "custom-dec-tree-warning"}, custom_test_split_warning_ui("custom_tool_warnings_dec_tree")),
                selector="#custom_dec_tree_generator",
                where="beforeEnd",
            )
            return True
        
        if len(list(input.custom_dec_tree_features_sel())) == 0:
            ui.insert_ui(
                ui.div({"id": "custom-dec-tree-warning"}, custom_features_warning_ui("custom_tool_warnings_dec_tree")),
                selector="#custom_dec_tree_generator",
                where="beforeEnd",
            )
            return True
        
        if df_len * custom_test_size_split < 1.0:
            ui.insert_ui(
                ui.div({"id": "custom-dec-tree-warning"}, custom_test_split_low_warning_ui("custom_tool_warnings_dec_tree")),
                selector="#custom_dec_tree_generator",
                where="beforeEnd",
            )
            return True
        
        if df_len * ( 1 - custom_test_size_split ) < 1.0:
            ui.insert_ui(
                ui.div({"id": "custom-dec-tree-warning"}, custom_test_split_high_warning_ui("custom_tool_warnings_dec_tree")),
                selector="#custom_dec_tree_generator",
                where="beforeEnd",
            )
            return True
        
        feature_column_is_not_numeric = False
        feature_column_has_nan = False
        for columnName in list(input.custom_dec_tree_features_sel()):
            if not pd.api.types.is_numeric_dtype(clean_custom_df[columnName]):
                feature_column_is_not_numeric = True
                break
            if clean_custom_df[columnName].isnull().values.any():
                feature_column_has_nan = True
                break

        if feature_column_is_not_numeric == True:
            ui.insert_ui(
                ui.div({"id": "custom-dec-tree-warning"}, custom_features_non_numeric_warning_ui("custom_tool_warnings_dec_tree")),
                selector="#custom_dec_tree_generator",
                where="beforeEnd",
            )
            return True
        
        if feature_column_has_nan == True:
            ui.insert_ui(
                ui.div({"id": "custom-dec-tree-warning"}, custom_features_nan_warning_ui("custom_tool_warnings_dec_tree")),
                selector="#custom_dec_tree_generator",
                where="beforeEnd",
            )
            return True
        
        return False
    
    # FIT, PREDICCIÓN Y GUARDADO DE TODOS LOS DATOS DEL MODELO DE ÁRBOL DE DECISIÓN CUSTOM
    def custom_classification_model_dec_tree(model, data, size_test, predictors, outcome):
        # Crear la división de entrenamiento y testeo
        data_train, data_test = train_test_split(data, test_size = size_test)
        
        # Fit del modelo:
        model.fit(data_train[predictors],data_train[outcome])

        # Realizar predicciones en el set de entrenamiento:
        predictions = model.predict(data_train[predictors])

        # Guardar los valores de los resultados con el set de entrenamiento:
        custom_accuracy_decTree.set((metrics.accuracy_score(predictions,data_train[outcome]) * 100).round(decimals=3))
        custom_recall_decTree.set((metrics.recall_score(predictions,data_train[outcome],average='micro') * 100).round(decimals=3))
        custom_precision_decTree.set((metrics.precision_score(predictions,data_train[outcome],average='micro') * 100).round(decimals=3))
        custom_f1_decTree.set((metrics.f1_score(predictions,data_train[outcome],average='micro') * 100).round(decimals=3))

        # Realizar predicciones en el set de testeo:
        predictions_test = model.predict(data_test[predictors])

        # Guardar los valores de los resultados con el set de testeo:
        custom_accuracy_decTree_test.set((metrics.accuracy_score(predictions_test,data_test[outcome]) * 100).round(decimals=3))
        custom_recall_decTree_test.set((metrics.recall_score(predictions_test,data_test[outcome],average='micro') * 100).round(decimals=3))
        custom_precision_decTree_test.set((metrics.precision_score(predictions_test,data_test[outcome],average='micro') * 100).round(decimals=3))
        custom_f1_decTree_test.set((metrics.f1_score(predictions_test,data_test[outcome],average='micro') * 100).round(decimals=3))

        # Crear y guardar la matriz de confusión
        cm_train = metrics.confusion_matrix(predictions,data_train[outcome])
        cm_test = metrics.confusion_matrix(predictions_test,data_test[outcome])
        custom_tree_conf_mat_train.set(cm_train)
        custom_tree_conf_mat_test.set(cm_test)

        # Guardar la figura / imagen del árbol de decisión
        plt.figure(figsize=(12,12))
        m_tree = plot_tree(model, filled=True, feature_names=predictors, class_names=list(map(str, list(clean_custom_df[outcome].unique()))), rounded=True, fontsize=5)
        plt.savefig( str(decTree_image_folder) + "\\" + str(session.id) + 'custom_dec_tree.jpg',format='jpg',bbox_inches = "tight", dpi=600)
        # Cerrar todas las figuras para evitar llenar la memoria de información innecesaria
        plt.close('all')

        # Guardar los valores de la figura del árbol de decisión
        coords = list()
        coords_x = list()
        coords_y = list()
        texts = list()

        for node in m_tree:
            coords.append(list(node.get_position()))
            texts.append(node.get_text().replace("\n", "<br>"))

        for x, y in coords:
            coords_x.append(x)
            coords_y.append(y)
        
        custom_tree_plot_x_coords.set(coords_x)
        custom_tree_plot_y_coords.set(coords_y)
        custom_tree_plot_texts.set(texts)

#################################### EFECTOS REACTIVOS #######################################

    # GENERAR EL ÁRBOL DE DECISIÓN CUSTOM
    @reactive.Effect
    @reactive.event(input.custom_generate_decission_tree)
    def _():
        ui.remove_ui("#custom-dec-tree-warning")

        # Obtener el tamaño del set de testeo y la longitud de la dataframe para las comprobaciones:
        custom_test_size_split = input.custom_test_split_value()
        df_len = len(clean_custom_df)

        outcome_column_name = input.outcomeSelectorCustom()

        # Comprobaciones previas a realizar el algoritmo de árbol de decisión
        if custom_dec_tree_previous_checks(custom_test_size_split, df_len, outcome_column_name) == True:
            # Cerrar todas las visualizaciones
            ui.update_switch("custom_view_variable_importance_dec_tree", value=False)
            ui.update_switch("custom_conf_mat_dec_tree_switch", value=False)
            ui.update_switch("custom_view_tree_dec_tree_switch", value=False)
            # Resetear todos los resultados
            reset_results_custom_dec_tree()
            custom_empty_dec_tree_feature_importance_df()
            custom_decision_tree_execution_counter.set(0)
            return

        # Modificar valores None para poder ser aceptados:
        max_depth_val = input.custom_dec_tree_max_depth()
        if max_depth_val == 0:
            max_depth_val = None
        
        max_features_value = input.custom_dec_tree_max_features()
        if max_features_value == 'None':
            max_features_value = None

        # Crear el modelo
        custom_dec_tree_model = DecisionTreeClassifier(criterion=input.custom_dec_tree_criterion(),
                                                       splitter=input.custom_dec_tree_splitter(),
                                                       max_depth=max_depth_val,
                                                       min_samples_split=input.custom_dec_tree_min_samples_split(),
                                                       min_samples_leaf=input.custom_dec_tree_min_samples_leaf(),
                                                       max_features=max_features_value)

        # Realizar la lista de las características a utilizar:
        custom_features_list = list(input.custom_dec_tree_features_sel())

        # Fit, predecir y guardar todos los datos del árbol de decisión
        custom_classification_model_dec_tree(custom_dec_tree_model,clean_custom_df,custom_test_size_split,custom_features_list,outcome_column_name)
        
        # Variables importantes y guardado de sus datos
        custom_empty_dec_tree_feature_importance_df()
        custom_dec_tree_feat_imp = pd.Series(custom_dec_tree_model.feature_importances_, index=custom_features_list).sort_values(ascending=False)
        custom_dec_tree_feat_imp_df.insert(0, "Característica", custom_dec_tree_feat_imp.index)
        custom_dec_tree_feat_imp_df.insert(1, "Valor", custom_dec_tree_feat_imp.values.round(decimals=3) * 100)

        custom_decision_tree_execution_counter.set(custom_decision_tree_execution_counter.get()+1)

    # MOSTRAR EL WIDGET DE IMPORTANCIA DE VARIABLES DEL ÁRBOL DE DECISIÓN CUSTOM
    @reactive.Effect
    def _():
        custom_var_imp_dec_tree_switch = input.custom_view_variable_importance_dec_tree()
        if custom_var_imp_dec_tree_switch == True:
            ui.remove_ui("#custom-var-imp-dec-tree-plot")
            if custom_decision_tree_execution_counter.get() > 0:
                custom_var_imp_dec_tree_plot = output_widget("custom_widget_dec_tree_var_imp")
                ui.insert_ui(
                    ui.div({"id": "custom-var-imp-dec-tree-plot"}, custom_var_imp_dec_tree_plot, style = "width:100%; overflow-x:auto; overflow-y:auto;"),
                    selector="#custom_var_imp_dec_tree",
                    where="beforeEnd",
                )
            else:
                custom_var_imp_dec_tree_warning = ui.output_text("decision_tree_warning_feat_imp_txt"),
                ui.insert_ui(
                    ui.div({"id": "custom-var-imp-dec-tree-plot"}, custom_var_imp_dec_tree_warning, style="color:red; font-style:italic; margin-top:20px; padding: 10px; background: #f7f7f7; border-radius: 10px;"),
                    selector="#custom_var_imp_dec_tree",
                    where="beforeEnd",
                )
        else:
            ui.remove_ui("#custom-var-imp-dec-tree-plot")
    
    # DESELECCIONAR LAS VARIABLES POCO IMPORTANTES ÁRBOL DE DECISIÓN CUSTOM
    @reactive.Effect
    @reactive.event(input.custom_deselect_not_imp_vars_dec_tree)
    def _():
        custom_minimum_importance = input.custom_minimum_importance_dec_tree()
        custom_important_columns_auto = [feature["Característica"] for idx, feature in custom_dec_tree_feat_imp_df.iterrows() if (feature["Valor"] >= custom_minimum_importance)]
        ui.update_checkbox_group("custom_dec_tree_features_sel", selected=custom_important_columns_auto)

    # MOSTRAR LA MATRIZ DE CONFUSIÓN DEL ÁRBOL DE DECISIÓN CUSTOM
    @reactive.Effect
    def _():
        custom_conf_mat_dec_tree_switch = input.custom_conf_mat_dec_tree_switch()
        if custom_conf_mat_dec_tree_switch == True:
            ui.remove_ui("#custom-dec-tree-conf-mat-train")
            ui.remove_ui("#custom-dec-tree-conf-mat-test")
            if custom_decision_tree_execution_counter.get() > 0:
                custom_dec_tree_conf_mat_train = output_widget("custom_widget_dec_tree_conf_mat_train")
                ui.insert_ui(
                    ui.div({"id": "custom-dec-tree-conf-mat-train"}, custom_dec_tree_conf_mat_train, style = "width:100%; height:300px; overflow-x:auto; overflow-y:auto;"),
                    selector="#custom_dec_tree_conf_matrix_train",
                    where="beforeEnd",
                )
                custom_dec_tree_conf_mat_test = output_widget("custom_widget_dec_tree_conf_mat_test")
                ui.insert_ui(
                    ui.div({"id": "custom-dec-tree-conf-mat-test"}, custom_dec_tree_conf_mat_test, style = "width:100%; height:300px; overflow-x:auto; overflow-y:auto;"),
                    selector="#custom_dec_tree_conf_matrix_test",
                    where="beforeEnd",
                )
            else:
                custom_conf_mat_dec_tree_warning = ui.output_text("custom_decision_tree_warning_conf_matrix_txt"),
                ui.insert_ui(
                    ui.div({"id": "custom-dec-tree-conf-mat-train"}, custom_conf_mat_dec_tree_warning, style="color:red; font-style:italic; margin-top:20px; padding: 10px; background: #f7f7f7; border-radius: 10px;"),
                    selector="#custom_dec_tree_conf_matrix",
                    where="beforeEnd",
                )
        else:
            ui.remove_ui("#custom-dec-tree-conf-mat-train")
            ui.remove_ui("#custom-dec-tree-conf-mat-test")

    # MOSTRAR LA REPRESENTACIÓN DEL ÁRBOL DE DECISIÓN CUSTOM
    @reactive.Effect
    def _():
        custom_view_tree_dec_tree_switch = input.custom_view_tree_dec_tree_switch()
        if custom_view_tree_dec_tree_switch == True:
            ui.remove_ui("#custom-dec-tree-view-img")
            if custom_decision_tree_execution_counter.get() > 0:
                custom_dec_tree_view = output_widget("custom_widget_dec_tree_view")
                ui.insert_ui(
                    ui.div({"id": "custom-dec-tree-view-img"}, custom_dec_tree_view, style = "width:100%; height:1000px; overflow-x:auto; overflow-y:auto;"),
                    selector="#custom_dec_tree_view",
                    where="beforeEnd",
                )
            else:
                custom_view_tree_dec_tree_warning = ui.output_text("custom_decision_tree_warning_view_txt"),
                ui.insert_ui(
                    ui.div({"id": "custom-dec-tree-view-img"}, custom_view_tree_dec_tree_warning, style="color:red; font-style:italic; margin-top:20px; padding: 10px; background: #f7f7f7; border-radius: 10px;"),
                    selector="#custom_dec_tree_view",
                    where="beforeEnd",
                )
        else:
            ui.remove_ui("#custom-dec-tree-view-img")

    # UPDATEAR EL CHECKBOX AL CAMBIAR VALORES EN EL SELECTOR DE LA VARIABLE A PREDECIR
    @reactive.Effect
    def _():
        input.outcomeSelectorCustom()
        custom_update_decTree_checkbox_group()  
    

#################################### WIDGETS ################################################# 

    # WIDGET IMPORTANCIA VARIABLES DEL ÁRBOL DE DECISIÓN CUSTOM
    @output
    @render_widget
    def custom_widget_dec_tree_var_imp():
        # Variables a las que reaccionar:
        custom_decision_tree_execution_counter.get()

        if len(custom_dec_tree_feat_imp_df) == 0:
            return go.Figure()
        
        fig = go.Figure(data=[go.Bar(x = custom_dec_tree_feat_imp_df["Valor"],
                                     y = custom_dec_tree_feat_imp_df["Característica"],
                                     orientation='h',
                                     name="",
                                     marker=dict(color = custom_dec_tree_feat_imp_df["Valor"],
                                                 colorscale=px.colors.sequential.Viridis_r))
        ])

        fig.update_layout(autosize=True,
                          height=max(280, 40*len(custom_dec_tree_feat_imp_df)),
                          margin=dict(l=20, r=20, t=40, b=20),)
        
        fig = fig.update_traces(hovertemplate='%{y} : %{x}%')

        fig.update_yaxes(autorange="reversed")
        
        return fig
    
    # WIDGET MOSTRAR MATRIZ DE CONFUSIÓN DE ENTRENAMIENTO DEL ÁRBOL DE DECISIÓN CUSTOM
    @output
    @render_widget
    def custom_widget_dec_tree_conf_mat_train():
        cm_map = custom_tree_conf_mat_train.get()
        outcome_column_name = input.outcomeSelectorCustom()
        tick_vals_list = list(clean_custom_df[outcome_column_name].unique())
        tick_text_list = list(map(str, list(clean_custom_df[outcome_column_name].unique())))

        fig = go.Figure(data=[go.Heatmap(z=cm_map,
                                        xgap = 1,
                                        ygap = 1,
                                        colorscale=px.colors.sequential.Teal,
                                        name="")
        ])

        fig.update_xaxes(
            autorange="reversed",
        )

        fig.update_layout(title="Matriz de confusión: datos entrenamiento",
                          xaxis_title="Valores reales",
                          yaxis_title="Valores predichos",
                          xaxis = dict(
                                tickmode = 'array',
                                tickvals = tick_vals_list,
                                ticktext = tick_text_list,
                            ),
                          yaxis = dict(
                                scaleanchor = 'x',
                                tickmode = 'array',
                                tickvals = tick_vals_list,
                                ticktext = tick_text_list,
                            ),
                          autosize=True,
                          height=300,
                          width=400,
                          margin=dict(l=20, r=20, t=40, b=20),)
        
        fig = fig.update_traces(text=cm_map,
                                texttemplate="%{text}",
                                hovertemplate='Valor real: %{x}<br>Valor predicho: %{y}<br>Cantidad: %{z}')
                
        return fig
    
    # WIDGET MOSTRAR MATRIZ DE CONFUSIÓN DE TESTEO DEL ÁRBOL DE DECISIÓN CUSTOM
    @output
    @render_widget
    def custom_widget_dec_tree_conf_mat_test():
        cm_map = custom_tree_conf_mat_test.get()
        outcome_column_name = input.outcomeSelectorCustom()
        tick_vals_list = list(clean_custom_df[outcome_column_name].unique())
        tick_text_list = list(map(str, list(clean_custom_df[outcome_column_name].unique())))

        fig = go.Figure(data=[go.Heatmap(z=cm_map,
                                        xgap = 1,
                                        ygap = 1,
                                        colorscale=px.colors.sequential.Teal,
                                        name="")
        ])

        fig.update_xaxes(
            autorange="reversed",
        )

        fig.update_layout(title="Matriz de confusión: datos test",
                          xaxis_title="Valores reales",
                          yaxis_title="Valores predichos",
                          xaxis = dict(
                                tickmode = 'array',
                                tickvals = tick_vals_list,
                                ticktext = tick_text_list,
                            ),
                          yaxis = dict(
                                scaleanchor = 'x',
                                tickmode = 'array',
                                tickvals = tick_vals_list,
                                ticktext = tick_text_list,
                            ),
                          autosize=True,
                          height=300,
                          width=400,
                          margin=dict(l=20, r=20, t=40, b=20),)
        
        fig = fig.update_traces(text=cm_map,
                                texttemplate="%{text}",
                                hovertemplate='Valor real: %{x}<br>Valor predicho: %{y}<br>Cantidad: %{z}')
                
        return fig
    
    # WIDGET MOSTRAR REPRESENTACIÓN DEL ÁRBOL DE DECISIÓN CUSTOM
    @output
    @render_widget
    def custom_widget_dec_tree_view():
        # Variables a las que reaccionar:
        custom_decision_tree_execution_counter.get()

        img_path = str(Path(__file__).parent / "DecTrees") + "\\" + str(session.id) + "custom_dec_tree.jpg"
        img_src = Image.open( img_path )
        
        fig = go.Figure()

        fig.add_trace(
            go.Scatter(
                x=custom_tree_plot_x_coords.get(),
                y=custom_tree_plot_y_coords.get(),
                text=custom_tree_plot_texts.get(),
                mode="markers",
                marker=dict(
                    color="white",
                    size=60,
                    opacity=0.1,
                ),
                name="",
            )
        )

        # Configurar los ejes
        fig.update_xaxes(
            visible=False,
            range=[0,1],
        )

        fig.update_yaxes(
            visible=False,
            range=[0,1],
            # el atributo de scaleanchor asegura que la relación de aspecto no se modifique
            scaleanchor="x"
        )

        fig.add_layout_image(
            dict(
                x=-0.02,
                sizex=1.04,
                y=1.01,
                sizey=1.02,
                xref="x",
                yref="y",
                opacity=1.0,
                layer="above",
                sizing="stretch",
                source=img_src)
        )

        fig = fig.update_traces(hovertemplate='%{text}')

        fig.update_layout(autosize=True,
                          height=1000,
                          margin=dict(l=20, r=20, t=40, b=20),)
        
        return fig
    

#################################### TEXTOS ##################################################

    # RESULTADOS DE SET DE ENTRENAMIENTO CON EL ÁRBOL DE DECISIÓN CUSTOM
    @output
    @render.text
    def custom_decision_tree_accuracy():
        if custom_accuracy_decTree.get() == -1:
            return "Exactitud: "
        return "Exactitud: " + str(custom_accuracy_decTree.get()) + "%"

    @output
    @render.text
    def custom_decision_tree_recall():
        if custom_recall_decTree.get() == -1:
            return "Sensibilidad o TVP: "
        return "Sensibilidad o TVP: " + str(custom_recall_decTree.get()) + "%"
    
    @output
    @render.text
    def custom_decision_tree_precision():
        if custom_precision_decTree.get() == -1:
            return "Precisión: "
        return "Precisión: " + str(custom_precision_decTree.get()) + "%"
    
    @output
    @render.text
    def custom_decision_tree_f1():
        if custom_f1_decTree.get() == -1:
            return "F1 Score: "
        return "F1 Score: " + str(custom_f1_decTree.get()) + "%"
    
    # RESULTADOS DE SET DE TESTEO CON EL ÁRBOL DE DECISIÓN CUSTOM
    @output
    @render.text
    def custom_decision_tree_accuracy_test():
        if custom_accuracy_decTree_test.get() == -1:
            return "Exactitud: "
        return "Exactitud: " + str(custom_accuracy_decTree_test.get()) + "%"

    @output
    @render.text
    def custom_decision_tree_recall_test():
        if custom_recall_decTree_test.get() == -1:
            return "Sensibilidad o TVP: "
        return "Sensibilidad o TVP: " + str(custom_recall_decTree_test.get()) + "%"
    
    @output
    @render.text
    def custom_decision_tree_precision_test():
        if custom_precision_decTree_test.get() == -1:
            return "Precisión: "
        return "Precisión: " + str(custom_precision_decTree_test.get()) + "%"
    
    @output
    @render.text
    def custom_decision_tree_f1_test():
        if custom_f1_decTree_test.get() == -1:
            return "F1 Score: "
        return "F1 Score: " + str(custom_f1_decTree_test.get()) + "%"
    
    # WARNING MATRIZ DE CONFUSIÓN
    @output
    @render.text
    def custom_decision_tree_warning_conf_matrix_txt():
        return "¡No se puede mostrar la matriz de confusión del árbol de decisión sin haber creado el modelo!"
    
    # WARNING VISUALIZACIÓN ÁRBOL
    @output
    @render.text
    def custom_decision_tree_warning_view_txt():
        return "¡No se puede mostrar el árbol de decisión sin haber creado el modelo!"

#################################### UPDATES Y OTROS #########################################

    # ACTUALIZAR OPCIONES DEL CHECKBOX DE CARACTERÍSTICAS
    def custom_update_decTree_checkbox_group():
        column_dict = {}
        for col in clean_custom_df.columns:
            if col != input.outcomeSelectorCustom():
                column_dict[col] = col
        ui.update_checkbox_group("custom_dec_tree_features_sel", choices=column_dict, selected=list(column_dict))




##############################################################################################
#################################### RANDOM FOREST ###########################################
##############################################################################################

#################################### IMPORTANTES #############################################

    # COMPROBACIONES PREVIAS RANDOM FOREST
    def custom_ran_forest_previous_checks(test_size_split, df_len, outcome_column_name):
        if len(custom_df) <= 0:
            ui.insert_ui(
                ui.div({"id": "custom-ran-forest-warning"}, custom_no_data_warning_ui("custom_tool_warnings_ran_forest")),
                selector="#custom_ran_forest_generator",
                where="beforeEnd",
            )
            return True

        if not pd.api.types.is_numeric_dtype(clean_custom_df[outcome_column_name]):
            ui.insert_ui(
                ui.div({"id": "custom-ran-forest-warning"}, custom_outcome_warning_ui("custom_tool_warnings_ran_forest")),
                selector="#custom_ran_forest_generator",
                where="beforeEnd",
            )
            return True

        if custom_test_split_done.get() == False:
            ui.insert_ui(
                ui.div({"id": "custom-ran-forest-warning"}, custom_test_split_warning_ui("custom_tool_warnings_ran_forest")),
                selector="#custom_ran_forest_generator",
                where="beforeEnd",
            )
            return True
        
        if len(list(input.custom_ran_forest_features_sel())) == 0:
            ui.insert_ui(
                ui.div({"id": "custom-ran-forest-warning"}, custom_features_warning_ui("custom_tool_warnings_ran_forest")),
                selector="#custom_ran_forest_generator",
                where="beforeEnd",
            )
            return True
        
        if df_len * test_size_split < 1.0:
            ui.insert_ui(
                ui.div({"id": "custom-ran-forest-warning"}, custom_test_split_low_warning_ui("custom_tool_warnings_ran_forest")),
                selector="#custom_ran_forest_generator",
                where="beforeEnd",
            )
            return True
        
        if df_len * ( 1 - test_size_split ) < 1.0:
            ui.insert_ui(
                ui.div({"id": "custom-ran-forest-warning"}, custom_test_split_high_warning_ui("custom_tool_warnings_ran_forest")),
                selector="#custom_ran_forest_generator",
                where="beforeEnd",
            )
            return True
        
        feature_column_is_not_numeric = False
        feature_column_has_nan = False
        for columnName in list(input.custom_ran_forest_features_sel()):
            if not pd.api.types.is_numeric_dtype(clean_custom_df[columnName]):
                feature_column_is_not_numeric = True
                break
            if clean_custom_df[columnName].isnull().values.any():
                feature_column_has_nan = True
                break

        if feature_column_is_not_numeric == True:
            ui.insert_ui(
                ui.div({"id": "custom-ran-forest-warning"}, custom_features_non_numeric_warning_ui("custom_tool_warnings_ran_forest")),
                selector="#custom_ran_forest_generator",
                where="beforeEnd",
            )
            return True
        
        if feature_column_has_nan == True:
            ui.insert_ui(
                ui.div({"id": "custom-ran-forest-warning"}, custom_features_nan_warning_ui("custom_tool_warnings_ran_forest")),
                selector="#custom_ran_forest_generator",
                where="beforeEnd",
            )
            return True
        
        return False

    # FIT, PREDICCIÓN Y GUARDADO DE DATOS DEL RANDOM FOREST
    def custom_classification_model_random_forest(model, data, size_test, predictors, outcome, n_estimators):
        # Crear la división de test y entrenamiento!
        data_train, data_test = train_test_split(data, test_size = size_test)
        
        # Fit del modelo:
        model.fit(data_train[predictors],data_train[outcome])

        # Hacer predicciones del set de entrenamiento:
        predictions = model.predict(data_train[predictors])
        
        # Setear los resultados del set de entrenamiento:
        custom_accuracy_ranForest.set((metrics.accuracy_score(predictions,data_train[outcome]) * 100).round(decimals=3))
        custom_recall_ranForest.set((metrics.recall_score(predictions,data_train[outcome]) * 100).round(decimals=3))
        custom_precision_ranForest.set((metrics.precision_score(predictions,data_train[outcome]) * 100).round(decimals=3))
        custom_f1_ranForest.set((metrics.f1_score(predictions,data_train[outcome]) * 100).round(decimals=3))

        # Hacer predicciones del set de test:
        predictions_test = model.predict(data_test[predictors])

        # Setear los resultados del set de test:
        custom_accuracy_ranForest_test.set((metrics.accuracy_score(predictions_test,data_test[outcome]) * 100).round(decimals=3))
        custom_recall_ranForest_test.set((metrics.recall_score(predictions_test,data_test[outcome]) * 100).round(decimals=3))
        custom_precision_ranForest_test.set((metrics.precision_score(predictions_test,data_test[outcome]) * 100).round(decimals=3))
        custom_f1_ranForest_test.set((metrics.f1_score(predictions_test,data_test[outcome]) * 100).round(decimals=3))

        # Creación y guardado de la matriz de confusión
        cm_train = metrics.confusion_matrix(predictions,data_train[outcome])
        cm_test = metrics.confusion_matrix(predictions_test,data_test[outcome])
        custom_ranForest_tree_conf_mat_train.set(cm_train)
        custom_ranForest_tree_conf_mat_test.set(cm_test)

        coords_x_list = list()
        coords_y_list = list()
        texts_list = list()

        # Creación de las figuras de árboles de decisión (máximo 5 para ahorrar espacio)
        for index in range(0, min(5, n_estimators)):
            plt.figure(figsize=(12,12))
            m_tree = plot_tree(model.estimators_[index], filled=True, feature_names=predictors, class_names=list(map(str, list(clean_custom_df[outcome].unique()))), rounded=True, fontsize=5)
            plt.savefig( str(ranForest_image_folder) + "\\" + str(session.id) + 'custom_ran_forest' + str(index) + '.jpg',format='jpg',bbox_inches = "tight", dpi=600)
            # Cerrar todas las figuras para evitar llenar la memoria de información innecesaria
            plt.close('all')

            # Guardado de datos de la figura del árbol de decisión
            coords = list()
            coords_x = list()
            coords_y = list()
            texts = list()

            for node in m_tree:
                coords.append(list(node.get_position()))
                # Arreglo del problema generado por boostrap sampling en los random forest:
                new_texts = node.get_text().split("\n")
                first_value = 0
                second_value = 0
                value_index = 0
                for idx, string in enumerate(new_texts):
                    values_split = re.split('(\d+)', string)
                    if len(values_split) > 0 and values_split[0] == 'value = [':
                        first_value = int(values_split[1])
                        second_value = int(values_split[3])
                        value_index = idx

                if value_index != 0:
                    new_texts[value_index - 1] = 'samples = ' + str(first_value + second_value)

                final_string = '<br>'.join(new_texts)

                texts.append(final_string)

            for x, y in coords:
                coords_x.append(x)
                coords_y.append(y)

            coords_x_list.append(coords_x)
            coords_y_list.append(coords_y)
            texts_list.append(texts)
        
        custom_ranForest_tree_plot_x_coords.set(coords_x_list)
        custom_ranForest_tree_plot_y_coords.set(coords_y_list)
        custom_ranForest_tree_plot_texts.set(texts_list)

        custom_random_forest_last_estimators_num.set(n_estimators)

#################################### EFECTOS REACTIVOS #######################################
    
    # GENERAR EL MODELO DE RANDOM FOREST Y REALIZAR TODOS LOS CÁLCULOS
    @reactive.Effect
    @reactive.event(input.generate_custom_random_forest)
    def _():
        ui.remove_ui("#custom-ran-forest-warning")

        # Obtener el tamaño de la separación de entrenamiento y la longitud de la base de datos para comprobaciones:
        test_size_split = input.custom_test_split_value()
        df_len = len(clean_custom_df)

        outcome_column_name = input.outcomeSelectorCustom()

        # Comprobaciones previas. Si algo falla, el modelo no se calcula:
        if custom_ran_forest_previous_checks(test_size_split, df_len, outcome_column_name) == True:
            # Cerrar todas las visualizaciones
            ui.update_switch("view_variable_importance_custom_ran_forest", value=False)
            ui.update_switch("conf_mat_custom_ran_forest_switch", value=False)
            ui.update_switch("view_tree_custom_ran_forest_switch", value=False)
            # Resetear todos los resultados
            reset_results_custom_ran_forest()
            custom_empty_ran_forest_feature_importance_df()
            custom_random_forest_execution_counter.set(0)
            return

        # Arreglar valores None para poder ser aceptados por el modelo:
        max_depth_val = input.custom_ran_forest_max_depth()
        if max_depth_val == 0:
            max_depth_val = None
        
        max_features_value = input.custom_ran_forest_max_features()
        if max_features_value == 'None':
            max_features_value = None

        n_estimators_ran_forest = input.custom_ran_forest_n_estimators()

        # Crear el modelo de random forest
        custom_ran_forest_model = RandomForestClassifier(n_estimators=n_estimators_ran_forest,
                                                         criterion=input.custom_ran_forest_criterion(),
                                                         max_depth=max_depth_val,
                                                         min_samples_split=input.custom_ran_forest_min_samples_split(),
                                                         min_samples_leaf=input.custom_ran_forest_min_samples_leaf(),
                                                         max_features=max_features_value)
        # bootstrap=False # Boostrap sampling causa problemas al representar los árboles, su número de samples no
        # corresponde a la suma de los valores de cada tipo. Sin embargo, si se desactiva, todos los árboles generados
        # son exactamente iguales.
        
        # Lista de las características que usamos:
        features_list = list(input.custom_ran_forest_features_sel())

        #Fit y predicciónes del modelo. Guardado de todos los datos
        custom_classification_model_random_forest(custom_ran_forest_model,clean_custom_df,test_size_split,features_list,outcome_column_name,n_estimators_ran_forest)

        # Variables importantes y guardado de sus resultados
        custom_empty_ran_forest_feature_importance_df()
        custom_ran_forest_feat_imp = pd.Series(custom_ran_forest_model.feature_importances_, index=features_list).sort_values(ascending=False)
        custom_ran_forest_feat_imp_df.insert(0, "Característica", custom_ran_forest_feat_imp.index)
        custom_ran_forest_feat_imp_df.insert(1, "Valor", custom_ran_forest_feat_imp.values.round(decimals=3) * 100)

        custom_random_forest_execution_counter.set(custom_random_forest_execution_counter.get()+1)

    # MOSTRAR EL WIDGET DE IMPORTANCIA DE VARIABLES DEL RANDOM FOREST
    @reactive.Effect
    def _():
        custom_var_imp_ran_forest_switch = input.view_variable_importance_custom_ran_forest()
        if custom_var_imp_ran_forest_switch == True:
            ui.remove_ui("#custom-var-imp-ran-forest-plot")
            if custom_random_forest_execution_counter.get() > 0:
                custom_var_imp_ran_forest_plot = output_widget("custom_widget_ran_forest_var_imp")
                ui.insert_ui(
                    ui.div({"id": "custom-var-imp-ran-forest-plot"}, custom_var_imp_ran_forest_plot, style = "width:100%; overflow-x:auto; overflow-y:auto;"),
                    selector="#var_imp_custom_ran_forest",
                    where="beforeEnd",
                )
            else:
                custom_var_imp_ran_forest_warning = ui.output_text("custom_random_forest_warning_feat_imp_txt"),
                ui.insert_ui(
                    ui.div({"id": "custom-var-imp-ran-forest-plot"}, custom_var_imp_ran_forest_warning, style="color:red; font-style:italic; margin-top:20px; padding: 10px; background: #f7f7f7; border-radius: 10px;"),
                    selector="#var_imp_custom_ran_forest",
                    where="beforeEnd",
                )
        else:
            ui.remove_ui("#custom-var-imp-ran-forest-plot")
    
    # DESELECCIONAR VARIABLES POCO IMPORTANTES DEL RANDOM FOREST
    @reactive.Effect
    @reactive.event(input.deselect_not_imp_vars_custom_ran_forest)
    def _():
        minimum_importance = input.minimum_importance_custom_ran_forest()
        important_columns_auto = [feature["Característica"] for idx, feature in custom_ran_forest_feat_imp_df.iterrows() if (feature["Valor"] >= minimum_importance)]
        ui.update_checkbox_group("custom_ran_forest_features_sel", selected=important_columns_auto)

    # MOSTRAR LA MATRIZ DE CONFUSIÓN DEL RANDOM FOREST
    @reactive.Effect
    def _():
        custom_conf_mat_ran_forest_switch = input.conf_mat_custom_ran_forest_switch()
        if custom_conf_mat_ran_forest_switch == True:
            ui.remove_ui("#custom-ran-forest-conf-mat-train")
            ui.remove_ui("#custom-ran-forest-conf-mat-test")
            if custom_random_forest_execution_counter.get() > 0:
                custom_ran_forest_conf_mat_train = output_widget("custom_widget_ran_forest_conf_mat_train")
                ui.insert_ui(
                    ui.div({"id": "custom-ran-forest-conf-mat-train"}, custom_ran_forest_conf_mat_train, style = "width:100%; height:300px; overflow-x:auto; overflow-y:auto;"),
                    selector="#custom_ran_forest_conf_matrix_train",
                    where="beforeEnd",
                )
                custom_ran_forest_conf_mat_test = output_widget("custom_widget_ran_forest_conf_mat_test")
                ui.insert_ui(
                    ui.div({"id": "custom-ran-forest-conf-mat-test"}, custom_ran_forest_conf_mat_test, style = "width:100%; height:300px; overflow-x:auto; overflow-y:auto;"),
                    selector="#custom_ran_forest_conf_matrix_test",
                    where="beforeEnd",
                )
            else:
                custom_conf_mat_ran_forest_warning = ui.output_text("custom_random_forest_warning_conf_matrix_txt"),
                ui.insert_ui(
                    ui.div({"id": "custom-ran-forest-conf-mat-train"}, custom_conf_mat_ran_forest_warning, style="color:red; font-style:italic; margin-top:20px; padding: 10px; background: #f7f7f7; border-radius: 10px;"),
                    selector="#custom_ran_forest_conf_matrix",
                    where="beforeEnd",
                )
        else:
            ui.remove_ui("#custom-ran-forest-conf-mat-train")
            ui.remove_ui("#custom-ran-forest-conf-mat-test")

    # MOSTRAR EL WIDGET DEL RANDOM FOREST
    @reactive.Effect
    def _():
        custom_view_tree_ran_forest_switch = input.view_tree_custom_ran_forest_switch()
        if custom_view_tree_ran_forest_switch == True:
            ui.remove_ui("#custom-ran-forest-view-img")
            ui.remove_ui("#custom-ran-forest-view-img-foot")
            if custom_random_forest_execution_counter.get() > 0:
                custom_ran_forest_view = output_widget("custom_widget_ran_forest_view")
                ui.insert_ui(
                    ui.div({"id": "custom-ran-forest-view-img"}, custom_ran_forest_view, style = "width:100%; height:1000px; overflow-x:auto; overflow-y:auto;"),
                    selector="#custom_ran_forest_view",
                    where="beforeEnd",
                )
                custom_ran_forest_view_foot = ui.output_text("custom_random_forest__view_foot_txt")
                ui.insert_ui(
                    ui.div({"id": "custom-ran-forest-view-img-foot"}, custom_ran_forest_view_foot, style="color:grey; font-style:italic; text-align:center; font-size: 0.7em;"),
                    selector="#custom_ran_forest_view",
                    where="beforeEnd",
                )
            else:
                custom_view_tree_ran_forest_warning = ui.output_text("custom_random_forest_warning_view_txt"),
                ui.insert_ui(
                    ui.div({"id": "custom-ran-forest-view-img"}, custom_view_tree_ran_forest_warning, style="color:red; font-style:italic; margin-top:20px; padding: 10px; background: #f7f7f7; border-radius: 10px;"),
                    selector="#custom_ran_forest_view",
                    where="beforeEnd",
                )
        else:
            ui.remove_ui("#custom-ran-forest-view-img")
            ui.remove_ui("#custom-ran-forest-view-img-foot")

    # ACTUALIZAR EL SELECTOR DE ÁRBOL DE DECISIÓN PARA MOSTRAR
    @reactive.Effect
    def _():
        n_estimators = custom_random_forest_last_estimators_num.get()
        new_list = list()
        for index in range(0, min(5, n_estimators)):
            new_list.append(index)
        ui.update_select("view_tree_custom_ran_forest_number", choices=new_list)

    # UPDATEAR EL CHECKBOX AL CAMBIAR VALORES EN EL SELECTOR DE LA VARIABLE A PREDECIR
    @reactive.Effect
    def _():
        input.outcomeSelectorCustom()
        custom_update_ranForest_checkbox_group()  

#################################### WIDGETS #################################################    

    # WIDGET DE LA IMPORTANCIA DE LAS VARIABLES DEL RANDOM FOREST
    @output
    @render_widget
    def custom_widget_ran_forest_var_imp():
        # Variables a las que reaccionar:
        custom_random_forest_execution_counter.get()

        if len(custom_ran_forest_feat_imp_df) == 0:
            return go.Figure()
        
        fig = go.Figure(data=[go.Bar(x = custom_ran_forest_feat_imp_df["Valor"],
                                     y = custom_ran_forest_feat_imp_df["Característica"],
                                     orientation='h',
                                     name="",
                                     marker=dict(color = custom_ran_forest_feat_imp_df["Valor"],
                                                 colorscale=px.colors.sequential.Viridis_r))
        ])

        fig.update_layout(autosize=True,
                          height=max(280, 40*len(custom_ran_forest_feat_imp_df)),
                          margin=dict(l=20, r=20, t=40, b=20),)
        
        fig = fig.update_traces(hovertemplate='%{y} : %{x}%')

        fig.update_yaxes(autorange="reversed")
        
        return fig
    
    # WIDGET MATRIZ DE CONFUSIÓN ENTRENAMIENTO DEL RANDOM FOREST
    @output
    @render_widget
    def custom_widget_ran_forest_conf_mat_train():
        outcome_column_name = input.outcomeSelectorCustom()
        tick_vals_list = list(clean_custom_df[outcome_column_name].unique())
        tick_text_list = list(map(str, list(clean_custom_df[outcome_column_name].unique())))

        cm_map = custom_ranForest_tree_conf_mat_train.get()
        fig = go.Figure(data=[go.Heatmap(z=cm_map,
                                        xgap = 1,
                                        ygap = 1,
                                        colorscale=px.colors.sequential.Teal,
                                        name="")
        ])

        fig.update_xaxes(
            autorange="reversed",
        )

        fig.update_layout(title="Matriz de confusión: datos entrenamiento",
                          xaxis_title="Valores reales",
                          yaxis_title="Valores predichos",
                          xaxis = dict(
                                tickmode = 'array',
                                tickvals = tick_vals_list,
                                ticktext = tick_text_list,
                            ),
                          yaxis = dict(
                                scaleanchor = 'x',
                                tickmode = 'array',
                                tickvals = tick_vals_list,
                                ticktext = tick_text_list,
                            ),
                          autosize=True,
                          height=300,
                          width=400,
                          margin=dict(l=20, r=20, t=40, b=20),)
        
        fig = fig.update_traces(text=cm_map,
                                texttemplate="%{text}",
                                hovertemplate='Valor real: %{x}<br>Valor predicho: %{y}<br>Cantidad: %{z}')
                
        return fig
    
    # WIDGET MATRIZ DE CONFUSIÓN TESTING DEL RANDOM FOREST
    @output
    @render_widget
    def custom_widget_ran_forest_conf_mat_test():
        outcome_column_name = input.outcomeSelectorCustom()
        tick_vals_list = list(clean_custom_df[outcome_column_name].unique())
        tick_text_list = list(map(str, list(clean_custom_df[outcome_column_name].unique())))

        cm_map = custom_ranForest_tree_conf_mat_test.get()
        fig = go.Figure(data=[go.Heatmap(z=cm_map,
                                        xgap = 1,
                                        ygap = 1,
                                        colorscale=px.colors.sequential.Teal,
                                        name="")
        ])

        fig.update_xaxes(
            autorange="reversed",
        )

        fig.update_layout(title="Matriz de confusión: datos test",
                          xaxis_title="Valores reales",
                          yaxis_title="Valores predichos",
                          xaxis = dict(
                                tickmode = 'array',
                                tickvals = tick_vals_list,
                                ticktext = tick_text_list,
                            ),
                          yaxis = dict(
                                scaleanchor = 'x',
                                tickmode = 'array',
                                tickvals = tick_vals_list,
                                ticktext = tick_text_list,
                            ),
                          autosize=True,
                          height=300,
                          width=400,
                          margin=dict(l=20, r=20, t=40, b=20),)
        
        fig = fig.update_traces(text=cm_map,
                                texttemplate="%{text}",
                                hovertemplate='Valor real: %{x}<br>Valor predicho: %{y}<br>Cantidad: %{z}')
                
        return fig
    
    # WIDGET VISUALIZACIÓN DEL RANDOM FOREST
    @output
    @render_widget
    def custom_widget_ran_forest_view():
        # Variables a las que reaccionar:
        custom_random_forest_execution_counter.get()

        num_tree = int(input.view_tree_custom_ran_forest_number())

        img_path = str(Path(__file__).parent / "RanForests") + "\\" + str(session.id) + 'custom_ran_forest' + str(num_tree) + '.jpg'
        img_src = Image.open( img_path )
        
        fig = go.Figure()

        fig.add_trace(
            go.Scatter(
                x=custom_ranForest_tree_plot_x_coords.get()[num_tree],
                y=custom_ranForest_tree_plot_y_coords.get()[num_tree],
                text=custom_ranForest_tree_plot_texts.get()[num_tree],
                mode="markers",
                marker=dict(
                    color="white",
                    size=60,
                    opacity=0.1,
                ),
                name="",
            )
        )

        # Configurar ejes
        fig.update_xaxes(
            visible=False,
            range=[0,1],
        )

        fig.update_yaxes(
            visible=False,
            range=[0,1],
            # el atributo de scaleanchor asegura que la relación de aspecto no se modifique
            scaleanchor="x"
        )

        fig.add_layout_image(
            dict(
                x=-0.02,
                sizex=1.04,
                y=1.01,
                sizey=1.02,
                xref="x",
                yref="y",
                opacity=1.0,
                layer="above",
                sizing="stretch",
                source=img_src)
        )

        fig = fig.update_traces(hovertemplate='%{text}')

        fig.update_layout(autosize=True,
                          height=1000,
                          margin=dict(l=20, r=20, t=40, b=20),)
        
        return fig

#################################### TEXTOS ##################################################

    # RESULTADOS
    @output
    @render.text
    def custom_random_forest_accuracy():
        if custom_accuracy_ranForest.get() == -1:
            return "Exactitud: "
        return "Exactitud: " + str(custom_accuracy_ranForest.get()) + "%"

    @output
    @render.text
    def custom_random_forest_recall():
        if custom_recall_ranForest.get() == -1:
            return "Sensibilidad o TVP: "
        return "Sensibilidad o TVP: " + str(custom_recall_ranForest.get()) + "%"
    
    @output
    @render.text
    def custom_random_forest_precision():
        if custom_precision_ranForest.get() == -1:
            return "Precisión: "
        return "Precisión: " + str(custom_precision_ranForest.get()) + "%"
    
    @output
    @render.text
    def custom_random_forest_f1():
        if custom_f1_ranForest.get() == -1:
            return "F1 Score: "
        return "F1 Score: " + str(custom_f1_ranForest.get()) + "%"
    
    @output
    @render.text
    def custom_random_forest_accuracy_test():
        if custom_accuracy_ranForest_test.get() == -1:
            return "Exactitud: "
        return "Exactitud: " + str(custom_accuracy_ranForest_test.get()) + "%"

    @output
    @render.text
    def custom_random_forest_recall_test():
        if custom_recall_ranForest_test.get() == -1:
            return "Sensibilidad o TVP: "
        return "Sensibilidad o TVP: " + str(custom_recall_ranForest_test.get()) + "%"
    
    @output
    @render.text
    def custom_random_forest_precision_test():
        if custom_precision_ranForest_test.get() == -1:
            return "Precisión: "
        return "Precisión: " + str(custom_precision_ranForest_test.get()) + "%"
    
    @output
    @render.text
    def custom_random_forest_f1_test():
        if custom_f1_ranForest_test.get() == -1:
            return "F1 Score: "
        return "F1 Score: " + str(custom_f1_ranForest_test.get()) + "%"
    
    # WARNING MATRIZ DE CONFUSIÓN
    @output
    @render.text
    def custom_random_forest_warning_conf_matrix_txt():
        return "¡No se puede mostrar la matriz de confusión del random forest sin haber creado el modelo!"
    
    # WARNING VISUALIZACIÓN ÁRBOL
    @output
    @render.text
    def custom_random_forest_warning_view_txt():
        return "¡No se puede mostrar uno de los árboles de decisión sin haber creado el modelo!"
    
    @output
    @render.text
    def custom_random_forest_view_foot_txt():
        return "Nota: Los valores de samples mostrados en la imagen son erroneos. En los bocadillos de información son correctos, son la suma de samples."

#################################### UPDATES Y OTROS #########################################   

    # ACTUALIZAR CHECKBOX ÁRBOL DE DECISIÓN
    def custom_update_ranForest_checkbox_group():
        column_dict = {}
        for col in clean_custom_df.columns:
            if col != input.outcomeSelectorCustom():
                column_dict[col] = col
        ui.update_checkbox_group("custom_ran_forest_features_sel", choices=column_dict, selected=list(column_dict))


##############################################################################################
################################### REGRESIÓN LOGÍSTICA ######################################
##############################################################################################

#################################### IMPORTANTES #############################################

    # COMPROBACIONES PREVIAS DE LA REGRESIÓN LOGÍSTICA
    def custom_log_reg_previous_checks(test_size_split, df_len, outcome_column_name):
        if len(custom_df) <= 0:
            ui.insert_ui(
                ui.div({"id": "custom-log-reg-warning"}, custom_no_data_warning_ui("custom_tool_warnings_log_reg")),
                selector="#custom_log_reg_generator",
                where="beforeEnd",
            )
            return True
        
        if not pd.api.types.is_numeric_dtype(clean_custom_df[outcome_column_name]):
            ui.insert_ui(
                ui.div({"id": "custom-log-reg-warning"}, custom_outcome_warning_ui("custom_tool_warnings_log_reg")),
                selector="#custom_log_reg_generator",
                where="beforeEnd",
            )
            return True

        if custom_test_split_done.get() == False:
            ui.insert_ui(
                ui.div({"id": "custom-log-reg-warning"}, custom_test_split_warning_ui("custom_tool_warnings_log_reg")),
                selector="#custom_log_reg_generator",
                where="beforeEnd",
            )
            return True
        
        if len(list(input.custom_log_reg_features_sel())) == 0:
            ui.insert_ui(
                ui.div({"id": "custom-log-reg-warning"}, custom_features_warning_ui("custom_tool_warnings_log_reg")),
                selector="#custom_log_reg_generator",
                where="beforeEnd",
            )
            return True
        
        if df_len * test_size_split < 1.0:
            ui.insert_ui(
                ui.div({"id": "custom-log-reg-warning"}, custom_test_split_low_warning_ui("custom_tool_warnings_log_reg")),
                selector="#custom_log_reg_generator",
                where="beforeEnd",
            )
            return True
        
        if df_len * ( 1 - test_size_split ) < 1.0:
            ui.insert_ui(
                ui.div({"id": "custom-log-reg-warning"}, custom_test_split_high_warning_ui("custom_tool_warnings_log_reg")),
                selector="#custom_log_reg_generator",
                where="beforeEnd",
            )
            return True
        
        feature_column_is_not_numeric = False
        feature_column_has_nan = False
        for columnName in list(input.custom_log_reg_features_sel()):
            if not pd.api.types.is_numeric_dtype(clean_custom_df[columnName]):
                feature_column_is_not_numeric = True
                break
            if clean_custom_df[columnName].isnull().values.any():
                feature_column_has_nan = True
                break
        
        if feature_column_is_not_numeric == True:
            ui.insert_ui(
                ui.div({"id": "custom-log-reg-warning"}, custom_features_non_numeric_warning_ui("custom_tool_warnings_log_reg")),
                selector="#custom_log_reg_generator",
                where="beforeEnd",
            )
            return True
        
        if feature_column_has_nan == True:
            ui.insert_ui(
                ui.div({"id": "custom-log-reg-warning"}, custom_features_nan_warning_ui("custom_tool_warnings_log_reg")),
                selector="#custom_log_reg_generator",
                where="beforeEnd",
            )
            return True
        
        return False

    # FIT, PREDICCIÓN Y GUARDADO DE DATOS DE LA REGRESIÓN LOGÍSTICA
    def custom_classification_model_log_reg(model, data, size_test, predictors, outcome, log_reg_max_iter):
        # Crear la división de test y entrenamiento!
        data_train, data_test = train_test_split(data, test_size = size_test)
        
        # Fit del modelo:
        model.fit(data_train[predictors],data_train[outcome])

        if log_reg_max_iter == model.n_iter_[0]:
            custom_logistic_regression_warning = ui.output_text("custom_logistic_regression_warning_iters_txt"),
            ui.insert_ui(
                ui.div({"id": "custom-log-reg-warning"}, custom_logistic_regression_warning, style="color:orange; font-style:italic; margin-top:20px; padding: 10px; background: #f7f7f7; border-radius: 10px;"),
                selector="#custom_log_reg_generator",
                where="beforeEnd",
            )

        # Hacer predicciones del set de entrenamiento:
        predictions = model.predict(data_train[predictors])

        # Setear los resultados del set de entrenamiento:
        custom_accuracy_logReg.set((metrics.accuracy_score(predictions,data_train[outcome]) * 100).round(decimals=3))
        custom_recall_logReg.set((metrics.recall_score(predictions,data_train[outcome]) * 100).round(decimals=3))
        custom_precision_logReg.set((metrics.precision_score(predictions,data_train[outcome]) * 100).round(decimals=3))
        custom_f1_logReg.set((metrics.f1_score(predictions,data_train[outcome]) * 100).round(decimals=3))

        # Hacer predicciones del set de test:
        predictions_test = model.predict(data_test[predictors])

        # Setear los resultados del set des test:
        custom_accuracy_logReg_test.set((metrics.accuracy_score(predictions_test,data_test[outcome]) * 100).round(decimals=3))
        custom_recall_logReg_test.set((metrics.recall_score(predictions_test,data_test[outcome]) * 100).round(decimals=3))
        custom_precision_logReg_test.set((metrics.precision_score(predictions_test,data_test[outcome]) * 100).round(decimals=3))
        custom_f1_logReg_test.set((metrics.f1_score(predictions_test,data_test[outcome]) * 100).round(decimals=3))

        # Creación y guardado de la matriz de confusión
        cm_train = metrics.confusion_matrix(predictions,data_train[outcome])
        cm_test = metrics.confusion_matrix(predictions_test,data_test[outcome])
        custom_logReg_conf_mat_train.set(cm_train)
        custom_logReg_conf_mat_test.set(cm_test)

#################################### EFECTOS REACTIVOS #######################################

    # GENERAR EL MODELO DE LA REGRESIÓN LOGÍSTICA Y REALIZAR TODOS LOS CÁLCULOS
    @reactive.Effect
    @reactive.event(input.custom_generate_logistic_regression)
    def _():
        ui.remove_ui("#custom-log-reg-warning")

        # Obtener el tamaño de la separación de entrenamiento y la longitud de la base de datos para comprobaciones:
        test_size_split = input.custom_test_split_value()
        df_len = len(clean_custom_df)

        outcome_column_name = input.outcomeSelectorCustom()

        # Comprobaciones previas. Si algo falla, el modelo no se calcula:
        if custom_log_reg_previous_checks(test_size_split, df_len, outcome_column_name) == True:
            # Cerrar todas las visualizaciones
            ui.update_switch("custom_view_variable_importance_log_reg", value=False)
            ui.update_switch("custom_conf_mat_log_reg_switch", value=False)
            ui.update_switch("custom_view_tree_log_reg_switch", value=False)
            # Resetear todos los resultados
            reset_results_custom_log_reg()
            custom_empty_log_reg_feature_importance_df()
            custom_logistic_regression_execution_counter.set(0)
            return

        # Arreglar valores None para poder ser aceptados por el modelo:
        log_reg_penalty = input.custom_log_reg_penalty()
        if log_reg_penalty == 'None':
            log_reg_penalty = None

        log_reg_tolerance = 1 * pow(10, input.custom_log_reg_tol())

        log_reg_max_iter = input.custom_log_reg_max_iter()

        log_reg_l1_rat = None
        if log_reg_penalty == "elasticnet":
            log_reg_l1_rat = 0.5

        # Crear el modelo de regresión logística
        custom_log_reg_model = LogisticRegression(penalty=log_reg_penalty,
                                                  tol=log_reg_tolerance,
                                                  C=input.custom_log_reg_c(),
                                                  solver=input.custom_log_reg_solver(),
                                                  max_iter=log_reg_max_iter,
                                                  l1_ratio=log_reg_l1_rat)
        
        # Lista de las características que usamos:
        features_list = list(input.custom_log_reg_features_sel())

        # Fit y predicciónes del modelo. Guardado de todos los datos
        custom_classification_model_log_reg(custom_log_reg_model,clean_custom_df,test_size_split,features_list,outcome_column_name,log_reg_max_iter)

        # Variables importantes y guardado de sus resultados
        custom_empty_log_reg_feature_importance_df()
        custom_log_reg_feat_imp = pd.Series(np.abs(custom_log_reg_model.coef_[0]), index=features_list).sort_values(ascending=False)
        # La importancia de las variables en regresión logística no suman 1, lo cambiamos a porcentaje
        sum_all_imp_values = custom_log_reg_feat_imp.sum()
        custom_log_reg_feat_imp_df.insert(0, "Característica", custom_log_reg_feat_imp.index)
        custom_log_reg_feat_imp_df.insert(1, "Valor", (custom_log_reg_feat_imp.values / sum_all_imp_values).round(decimals=3) * 100)

        custom_logistic_regression_execution_counter.set(custom_logistic_regression_execution_counter.get()+1)

    # MOSTRAR EL WIDGET DE IMPORTANCIA DE VARIABLES DE LA REGRESIÓN LOGÍSTICA
    @reactive.Effect
    def _():
        custom_var_imp_log_reg_switch = input.custom_view_variable_importance_log_reg()
        if custom_var_imp_log_reg_switch == True:
            ui.remove_ui("#custom-var-imp-log-reg-plot")
            if custom_logistic_regression_execution_counter.get() > 0:
                custom_var_imp_log_reg_plot = output_widget("custom_widget_log_reg_var_imp")
                ui.insert_ui(
                    ui.div({"id": "custom-var-imp-log-reg-plot"}, custom_var_imp_log_reg_plot, style = "width:100%; overflow-x:auto; overflow-y:auto;"),
                    selector="#custom_var_imp_log_reg",
                    where="beforeEnd",
                )
            else:
                custom_var_imp_log_reg_warning = ui.output_text("custom_logistic_regression_warning_feat_imp_txt"),
                ui.insert_ui(
                    ui.div({"id": "custom-var-imp-log-reg-plot"}, custom_var_imp_log_reg_warning, style="color:red; font-style:italic; margin-top:20px; padding: 10px; background: #f7f7f7; border-radius: 10px;"),
                    selector="#custom_var_imp_log_reg",
                    where="beforeEnd",
                )
        else:
            ui.remove_ui("#custom-var-imp-log-reg-plot")
    
    # DESELECCIONAR VARIABLES POCO IMPORTANTES DE LA REGRESIÓN LOGÍSTICA
    @reactive.Effect
    @reactive.event(input.custom_deselect_not_imp_vars_log_reg)
    def _():
        minimum_importance = input.custom_minimum_importance_log_reg()
        important_columns_auto = [feature["Característica"] for idx, feature in custom_log_reg_feat_imp_df.iterrows() if (feature["Valor"] >= minimum_importance)]
        ui.update_checkbox_group("custom_log_reg_features_sel", selected=important_columns_auto)

    # MOSTRAR LA MATRIZ DE CONFUSIÓN DE LA REGRESIÓN LOGÍSTICA
    @reactive.Effect
    def _():
        custom_conf_mat_log_reg_switch = input.custom_conf_mat_log_reg_switch()
        if custom_conf_mat_log_reg_switch == True:
            ui.remove_ui("#custom-log-reg-conf-mat-train")
            ui.remove_ui("#custom-log-reg-conf-mat-test")
            if custom_logistic_regression_execution_counter.get() > 0:
                custom_log_reg_conf_mat_train = output_widget("custom_widget_log_reg_conf_mat_train")
                ui.insert_ui(
                    ui.div({"id": "custom-log-reg-conf-mat-train"}, custom_log_reg_conf_mat_train, style = "width:100%; height:300px; overflow-x:auto; overflow-y:auto;"),
                    selector="#custom_log_reg_conf_matrix_train",
                    where="beforeEnd",
                )
                custom_log_reg_conf_mat_test = output_widget("custom_widget_log_reg_conf_mat_test")
                ui.insert_ui(
                    ui.div({"id": "custom-log-reg-conf-mat-test"}, custom_log_reg_conf_mat_test, style = "width:100%; height:300px; overflow-x:auto; overflow-y:auto;"),
                    selector="#custom_log_reg_conf_matrix_test",
                    where="beforeEnd",
                )
            else:
                custom_conf_mat_log_reg_warning = ui.output_text("custom_logistic_regression_warning_conf_matrix_txt"),
                ui.insert_ui(
                    ui.div({"id": "custom-log-reg-conf-mat-train"}, custom_conf_mat_log_reg_warning, style="color:red; font-style:italic; margin-top:20px; padding: 10px; background: #f7f7f7; border-radius: 10px;"),
                    selector="#custom_log_reg_conf_matrix",
                    where="beforeEnd",
                )
        else:
            ui.remove_ui("#custom-log-reg-conf-mat-train")
            ui.remove_ui("#custom-log-reg-conf-mat-test")

    # UPDATEAR EL CHECKBOX AL CAMBIAR VALORES EN EL SELECTOR DE LA VARIABLE A PREDECIR
    @reactive.Effect
    def _():
        input.outcomeSelectorCustom()
        custom_update_logReg_checkbox_group()

    # ACTUALIZAR PENALTY SEGÚN SOLVER DE LA REGRESIÓN LOGÍSTICA
    @reactive.Effect
    def _():
        custom_solver = input.custom_log_reg_solver()
        if custom_solver == "saga":
            ui.update_select("custom_log_reg_penalty", choices={"elasticnet": "Elasticnet (L1 + L2)", "l1": "L1", "l2": "L2 (default)", "None": "None"})
        elif custom_solver == "liblinear":
            ui.update_select("custom_log_reg_penalty", choices={"l1": "L1", "l2": "L2 (default)"})
        else:
            ui.update_select("custom_log_reg_penalty", choices={"l2": "L2 (default)", "None": "None"})


#################################### WIDGETS #################################################    

    # WIDGET DE LA IMPORTANCIA DE LAS VARIABLES DE LA REGRESIÓN LOGÍSTICA
    @output
    @render_widget
    def custom_widget_log_reg_var_imp():
        # Variables a las que reaccionar:
        custom_logistic_regression_execution_counter.get()

        if len(custom_log_reg_feat_imp_df) == 0:
            return go.Figure()
        
        fig = go.Figure(data=[go.Bar(x = custom_log_reg_feat_imp_df["Valor"],
                                     y = custom_log_reg_feat_imp_df["Característica"],
                                     orientation='h',
                                     name="",
                                     marker=dict(color = custom_log_reg_feat_imp_df["Valor"],
                                                 colorscale=px.colors.sequential.Viridis_r))
        ])

        fig.update_layout(autosize=True,
                          height=max(280, 40*len(custom_log_reg_feat_imp_df)),
                          margin=dict(l=20, r=20, t=40, b=20),)
        
        fig = fig.update_traces(hovertemplate='%{y} : %{x}%')

        fig.update_yaxes(autorange="reversed")
        
        return fig
    
    # WIDGET MATRIZ DE CONFUSIÓN ENTRENAMIENTO DE LA REGRESIÓN LOGÍSTICA
    @output
    @render_widget
    def custom_widget_log_reg_conf_mat_train():
        cm_map = custom_logReg_conf_mat_train.get()
        outcome_column_name = input.outcomeSelectorCustom()
        tick_vals_list = list(clean_custom_df[outcome_column_name].unique())
        tick_text_list = list(map(str, list(clean_custom_df[outcome_column_name].unique())))

        fig = go.Figure(data=[go.Heatmap(z=cm_map,
                                        xgap = 1,
                                        ygap = 1,
                                        colorscale=px.colors.sequential.Teal,
                                        name="")
        ])

        fig.update_xaxes(
            autorange="reversed",
        )

        fig.update_layout(title="Matriz de confusión: datos entrenamiento",
                          xaxis_title="Valores reales",
                          yaxis_title="Valores predichos",
                          xaxis = dict(
                                tickmode = 'array',
                                tickvals = tick_vals_list,
                                ticktext = tick_text_list,
                            ),
                          yaxis = dict(
                                scaleanchor = 'x',
                                tickmode = 'array',
                                tickvals = tick_vals_list,
                                ticktext = tick_text_list,
                            ),
                          autosize=True,
                          height=300,
                          width=400,
                          margin=dict(l=20, r=20, t=40, b=20),)
        
        fig = fig.update_traces(text=cm_map,
                                texttemplate="%{text}",
                                hovertemplate='Valor real: %{x}<br>Valor predicho: %{y}<br>Cantidad: %{z}')
                
        return fig
    
    # WIDGET MATRIZ DE CONFUSIÓN TESTING DE LA REGRESIÓN LOGÍSTICA
    @output
    @render_widget
    def custom_widget_log_reg_conf_mat_test():
        cm_map = custom_logReg_conf_mat_test.get()
        outcome_column_name = input.outcomeSelectorCustom()
        tick_vals_list = list(clean_custom_df[outcome_column_name].unique())
        tick_text_list = list(map(str, list(clean_custom_df[outcome_column_name].unique())))

        fig = go.Figure(data=[go.Heatmap(z=cm_map,
                                        xgap = 1,
                                        ygap = 1,
                                        colorscale=px.colors.sequential.Teal,
                                        name="")
        ])

        fig.update_xaxes(
            autorange="reversed",
        )

        fig.update_layout(title="Matriz de confusión: datos test",
                          xaxis_title="Valores reales",
                          yaxis_title="Valores predichos",
                          xaxis = dict(
                                tickmode = 'array',
                                tickvals = tick_vals_list,
                                ticktext = tick_text_list,
                            ),
                          yaxis = dict(
                                scaleanchor = 'x',
                                tickmode = 'array',
                                tickvals = tick_vals_list,
                                ticktext = tick_text_list,
                            ),
                          autosize=True,
                          height=300,
                          width=400,
                          margin=dict(l=20, r=20, t=40, b=20),)
        
        fig = fig.update_traces(text=cm_map,
                                texttemplate="%{text}",
                                hovertemplate='Valor real: %{x}<br>Valor predicho: %{y}<br>Cantidad: %{z}')
                
        return fig
    

#################################### TEXTOS ##################################################

    # WARNINGS DE LA REGRESIÓN LOGÍSTICA    
    @output
    @render.text
    def custom_logistic_regression_warning_iters_txt():
        return "¡El modelo ha parado porque ha llegado al máximo de iteraciones! Modifica los datos de entrada o aumenta el número máximo de iteraciones."    

    # RESULTADOS DE LA REGRESIÓN LOGÍSTICA
    @output
    @render.text
    def custom_logistic_regression_accuracy():
        if custom_accuracy_logReg.get() == -1:
            return "Exactitud: "
        return "Exactitud: " + str(custom_accuracy_logReg.get()) + "%"

    @output
    @render.text
    def custom_logistic_regression_recall():
        if custom_recall_logReg.get() == -1:
            return "Sensibilidad o TVP: "
        return "Sensibilidad o TVP: " + str(custom_recall_logReg.get()) + "%"
    
    @output
    @render.text
    def custom_logistic_regression_precision():
        if custom_precision_logReg.get() == -1:
            return "Precisión: "
        return "Precisión: " + str(custom_precision_logReg.get()) + "%"
    
    @output
    @render.text
    def custom_logistic_regression_f1():
        if custom_f1_logReg.get() == -1:
            return "F1 Score: "
        return "F1 Score: " + str(custom_f1_logReg.get()) + "%"
        
    @output
    @render.text
    def custom_logistic_regression_accuracy_test():
        if custom_accuracy_logReg_test.get() == -1:
            return "Exactitud: "
        return "Exactitud: " + str(custom_accuracy_logReg_test.get()) + "%"

    @output
    @render.text
    def custom_logistic_regression_recall_test():
        if custom_recall_logReg_test.get() == -1:
            return "Sensibilidad o TVP: "
        return "Sensibilidad o TVP: " + str(custom_recall_logReg_test.get()) + "%"
    
    @output
    @render.text
    def custom_logistic_regression_precision_test():
        if custom_precision_logReg_test.get() == -1:
            return "Precisión: "
        return "Precisión: " + str(custom_precision_logReg_test.get()) + "%"
    
    @output
    @render.text
    def custom_logistic_regression_f1_test():
        if custom_f1_logReg_test.get() == -1:
            return "F1 Score: "
        return "F1 Score: " + str(custom_f1_logReg_test.get()) + "%"
    
    # WARNING MATRIZ DE CONFUSIÓN DE LA REGRESIÓN LOGÍSTICA
    @output
    @render.text
    def custom_logistic_regression_warning_conf_matrix_txt():
        return "¡No se puede mostrar la matriz de confusión de la regresión logística sin haber creado el modelo!"

#################################### UPDATES Y OTROS #########################################   

    # ACTUALIZAR CHECKBOX DE LA REGRESIÓN LOGÍSTICA
    def custom_update_logReg_checkbox_group():
        column_dict = {}
        for col in clean_custom_df.columns:
            if col != input.outcomeSelectorCustom():
                column_dict[col] = col
        ui.update_checkbox_group("custom_log_reg_features_sel", choices=column_dict, selected=list(column_dict))

    

##############################################################################################
############################## CUSTOM: RESET Y FUNCIONES EXTRA ###############################
##############################################################################################

#################################### UPDATES Y OTROS #########################################
    def update_all_selectors_custom():
        update_outcomeSelector_custom()
        custom_update_decTree_checkbox_group()
        custom_update_logReg_checkbox_group()
        custom_update_ranForest_checkbox_group()