from shiny import module, ui, reactive, render
from shiny.types import ImgData
from pathlib import Path

explanation_img_path = Path(__file__).parent.parent / "images"

@module.ui
def shared_varImp_ui():
    return ui.div(
        ui.div(
            ui.markdown("Una vez creado el modelo, podemos observar **cuáles son las características que más afectan al resultado** y cuáles son innecesarias. Las características con poca importancia para nuestro modelo pueden aportar ruido y generar resultados menos precisos.")
            , style="padding-right:50px; text-align: justify; text-justify: inter-word;" 
        ),
        ui.div(
            ui.markdown("Puedes seleccionar las características que consideres importantes de forma manual o usar el slider para declarar una importancia mínima y deseleccionar automáticamente las características por debajo de ese umbral. Si cambias las características seleccionadas, **¡no olvides de generar el modelo otra vez!**")
            , style="padding-right:50px; text-align: justify; text-justify: inter-word;" 
        ),
    )

@module.ui
def shared_conf_mat_metrics_ui(label = ""):
    return ui.div(
        {"id": "metrics_info"+str(label)},
        ui.div(
            ui.markdown("Para realizar el análisis de los resultados del algoritmo utilizaremos una matriz de confusión junto con varias métricas que calcularemos a partir de ella: precisión, sensibilidad, f1 score y exactitud.")
            , style="padding-top:10px; padding-right:50px; text-align: justify; text-justify: inter-word;" 
        ),
        ui.div(
            ui.markdown("La **matriz de confusión** es una tabla que se utiliza para evaluar el rendimiento de un modelo de clasificación. Es una **representación visual de la precisión de la clasificación del modelo** en función de las clases reales y las clases predichas.")
            , style="padding-right:50px; text-align: justify; text-justify: inter-word;" 
        ),
        ui.output_image("conf_mat_image", height="560px"),
        ui.input_action_button("show_metrics_info", "Conocer más sobre las métricas básicas ▽"
                                , style="padding: 0px 0px 10px 0px; background: white; border: none; font-weight: bold; text-decoration: underline; border: 0 !important; box-shadow: 0 0 !important; transition: 0.1s !important; background-color: transparent !important;"),       
    )

@module.ui
def shared_fitting_ui(label = ""):
    return ui.div(
        {"id": "fitting_info"+str(label)},
        ui.input_action_button("show_fitting_info", "Conocer más sobre el ajuste de los modelos ▽"
                        , style="padding: 20px 0px 10px 0px; background: white; border: none; font-weight: bold; text-decoration: underline; border: 0 !important; box-shadow: 0 0 !important; transition: 0.1s !important; background-color: transparent !important;"),
    )

@module.ui
def shared_tree_rep_info_ui(label = ""):
    return ui.div(
        {"id": "tree_rep_info"+str(label)},
        ui.input_action_button("show_tree_rep_info", "¿Qué significan los valores representados en el árbol de decisión? ▽"
                        , style="padding: 20px 0px 10px 0px; background: white; border: none; font-weight: bold; text-decoration: underline; border: 0 !important; box-shadow: 0 0 !important; transition: 0.1s !important; background-color: transparent !important;"),
    )


@module.server
def shared_algs_server(input, output, session, label = ""):
    
    @reactive.Effect
    @reactive.event(input.show_metrics_info)
    def _():
        show_metrics_info_button = input.show_metrics_info()
        if show_metrics_info_button % 2 == 1:
            ui.update_action_button("show_metrics_info", label="Conocer más sobre las métricas básicas △")
            ui.insert_ui(
                ui.div({"id": "inserted-metrics-info"+str(label)},
                    ui.markdown("""Como se puede observar en la imagen, a partir de la matriz de confusión, se pueden calcular diversas métricas de evaluación del modelo:
- **Precisión**: es una medida que evalúa la proporción de predicciones verdaderas positivas en relación con el total de predicciones positivas realizadas por un modelo, es decir, el porcentaje de casos positivos reales entre los detectados. Es útil cuando el énfasis está en minimizar los falsos positivos, como suele ocurrir en casos relacionados con salud.
- **Sensibilidad o Tasa de Verdaderos Positivos**: es una medida que evalúa la proporción de verdaderos positivos identificados correctamente en relación con el total de verdaderos positivos presentes en los datos. En el ámbito sanitario se puede definir como la capacidad de detectar correctamente la enfermedad entre los enfermos.
- **F1 Score**: es una medida que combina tanto la precisión como la sensibilidad en una única métrica. Es útil cuando se busca un equilibrio entre la precisión y la sensibilidad en un modelo.
- **Exactitud**: representa la proporción de predicciones correctas realizadas por un modelo en relación con el total de predicciones realizadas."""
                    ),
                    style="border: solid 0px grey; border-radius: 10px; background:#eceef1 ;margin-right:50px; padding:15px 20px 10px 20px; text-align: justify; text-justify: inter-word;",
                ),
                selector="#metrics_info"+str(label),
                where="beforeEnd",
            )
        else:
            ui.update_action_button("show_metrics_info", label="Conocer más sobre las métricas básicas ▽")
            ui.remove_ui("#inserted-metrics-info"+str(label))

    @reactive.Effect
    @reactive.event(input.show_fitting_info)
    def _():
        show_fitting_info_button = input.show_fitting_info()
        if show_fitting_info_button % 2 == 1:
            ui.update_action_button("show_fitting_info", label="Conocer más sobre el ajuste de los modelos △")
            ui.insert_ui(
                ui.div({"id": "inserted-fitting-info"+str(label)},
                    ui.markdown("""Antes de dar por terminado nuestro modelo y ahora que hemos obtenido sus resultados, debemos asegurarnos que obtenga los resultados deseados tanto con los datos de entrenamiento como con los de prueba. Si los resultados son muy distintos entre sí o deficientes en ambos casos, quiere decir que nuestro modelo presenta uno de estos problemas:
- **Sobreajuste (Overfitting)**: Ocurre cuando un modelo se ajusta demasiado a los datos de entrenamiento, capturando incluso el ruido o las fluctuaciones aleatorias presentes en los datos. Esto se traduce en un modelo que se adapta perfectamente a los datos de entrenamiento, pero tiene un rendimiento deficiente en los datos de prueba. En otras palabras, el modelo sobreajustado **memoriza los datos de entrenamiento en lugar de aprender patrones generales** que se apliquen a nuevos datos. Para abordar el sobreajuste, se pueden considerar modelos más simples, reducir la complejidad del modelo, utilizar técnicas de regularización (como la penalización de parámetros) o aplicar técnicas de validación cruzada para evaluar mejor el rendimiento del modelo en datos no vistos. En nuestro caso podemos probar simplificando el modelo introduciéndole menos características o cambiando los ajustes elegidos.
- **Subajuste (Underfitting)**: Ocurre cuando un modelo no puede capturar adecuadamente las relaciones y patrones presentes en los datos de entrenamiento. Esto se traduce en un modelo demasiado simple que no se ajusta bien a los datos y tiene un rendimiento deficiente tanto en los datos de entrenamiento como en los datos de prueba. En otras palabras, el modelo subajustado **no logra capturar la complejidad inherente en los datos y no generaliza bien** a nuevos datos. Para abordar el subajuste, se pueden considerar modelos más complejos, utilizar características más relevantes o aplicar técnicas de mejora, como aumentar la cantidad de datos de entrenamiento.

El objetivo es **encontrar un punto óptimo** en el que el modelo tenga un buen rendimiento tanto en los datos de entrenamiento como en los datos de prueba, lo que se conoce como un buen **equilibrio entre la capacidad del modelo y su capacidad de generalización**."""
                    ),
                    ui.output_image("overfit_image", height="260px"),
                    style="border: solid 0px grey; border-radius: 10px; background:#eceef1 ;margin-right:50px; padding:15px 20px 0px 20px; text-align: justify; text-justify: inter-word;",
                ),
                selector="#fitting_info"+str(label),
                where="beforeEnd",
            )
        else:
            ui.update_action_button("show_fitting_info", label="Conocer más sobre el ajuste de los modelos ▽")
            ui.remove_ui("#inserted-fitting-info"+str(label))

    @reactive.Effect
    @reactive.event(input.show_tree_rep_info)
    def _():
        show_tree_rep_info_button = input.show_tree_rep_info()
        if show_tree_rep_info_button % 2 == 1:
            ui.update_action_button("show_tree_rep_info", label="¿Qué significan los valores representados en el árbol de decisión? △")
            ui.insert_ui(
                ui.div({"id": "inserted-tree-rep-info"+str(label)},
                    ui.markdown("""¿Qué información está presente en cada nodo?
- **Condición de división**: contiene la característica con su condición utilizada para dividir los datos en el nodo.
- **Impureza** o criterio de impureza ('gini', 'entropy' o 'log_loss'): muestra el valor de la medida de impureza utilizada en el nodo. Esta medida ayuda a evaluar cómo de mezcladas están las clases en el nodo.
- **Número de muestras** ('samples'): representa la cantidad de muestras en el conjunto de datos que llegan a ese nodo.
- **Valor objetivo** ('value'): indica la distribución real de las clases en el nodo (cuantas muestras pertenecen realmente a cada clase). No tiene por qué coincidir con la división de ese nodo en el siguiente nivel.
- **Valor de predicción**: el valor de predicción o la clase mayoritaria que se asigna a las instancias que llegan a ese nodo."""
                    ),
                    style="border: solid 0px grey; border-radius: 10px; background:#eceef1 ;margin-right:50px; padding:15px 20px 10px 20px; text-align: justify; text-justify: inter-word;",
                ),
                selector="#tree_rep_info"+str(label),
                where="beforeEnd",
            )
        else:
            ui.update_action_button("show_tree_rep_info", label="¿Qué significan los valores representados en el árbol de decisión? ▽")
            ui.remove_ui("#inserted-tree-rep-info"+str(label))

    @output
    @render.image
    def conf_mat_image():
        img: ImgData = {"src": str(explanation_img_path / "Confusion matrix and metrics.png"), "height":"550px", "style":"display:block; margin-left:15%;"}
        return img
    
    @output
    @render.image
    def overfit_image():
        img: ImgData = {"src": str(explanation_img_path / "overfit underfit.png"), "height":"250px", "style":"display:block; margin-left:15%;"}
        return img