from shiny import module, ui, reactive, render
from shiny.types import ImgData
from pathlib import Path

explanation_img_path = Path(__file__).parent.parent / "images"


@module.ui
def ranForest_def_ui():
    return ui.div(
        ui.div(
            ui.markdown("Un bosque aleatorio (random forest) es un clasificador de conjunto que está **compuesto por múltiples árboles de decisión**. Los árboles de decisión individuales pueden sufrir sobreajuste cuando son muy profundos, lo que lleva a una alta variación en los resultados de clasificación para pequeños cambios en los datos de entrada. En un bosque aleatorio, los árboles se entrenan utilizando diferentes subconjuntos del conjunto de datos de entrenamiento. Para clasificar una nueva muestra, se pasa a través de cada árbol del bosque, y cada árbol genera un resultado de clasificación basado en una parte específica de la muestra. El bosque aleatorio selecciona la clasificación con la mayor cantidad de 'votos' en caso de clasificación discreta, o el promedio de las clasificaciones en caso de clasificación numérica. Al considerar los resultados de múltiples árboles, puede reducir la variación y el sobreajuste, mejorando la estabilidad y precisión de las clasificaciones.")
            , style="padding-right:50px; text-align: justify; text-justify: inter-word;" 
        ),
        ui.div(
            ui.markdown("A continuación, se muestra cómo un bosque aleatorio está formado por múltiples árboles de decisión (tres en este caso).")
            , style="padding-right:50px; padding-bottom:10px; text-align: justify; text-justify: inter-word;" 
        ),
        ui.output_image("ran_forest_expl_image", height="260px"),
    )

@module.ui
def ranForest_howTo_ui():
    return ui.div(
        {"id": "ran_forest_how_generate"},
        ui.input_action_button("ran_forest_show_how_info", "¿Cómo se genera el modelo de bosque aleatorio? ▽"
                        , style="padding: 30px 0px 10px 0px; background: white; border: none; font-weight: bold; text-decoration: underline; border: 0 !important; box-shadow: 0 0 !important; transition: 0.1s !important; background-color: transparent !important;"),
                            
    )

@module.ui
def ranForest_performance_ui():
    return ui.div(
        ui.div(
            ui.markdown("""**No hay un umbral exacto para considerar un modelo como bueno**, ya que depende del contexto y las necesidades del problema. En general, en aplicaciones relacionadas con el ámbito sanitario se busca maximizar tanto la precisión (para minimizar falsos positivos) como la sensibilidad o TVP (para minimizar falsos negativos), por lo que **se busca obtener un valor alto de F1**. En este ejemplo el valor de F1 puede llegar a superar el 95% utilizando los ajustes y características correctos. Aunque más complicado, al igual que ocurre con el árbol de decisión, el modelo puede ser sobreajustado.

*Consejo: editar la profundidad máxima del árbol es un buen punto de inicio para evitar el sobreajuste.*""")
            , style="padding-top:30px; padding-right:50px; text-align: justify; text-justify: inter-word;" 
        ),
    )


@module.server
def ranForest_server(input, output, session):
    
    @reactive.Effect
    @reactive.event(input.ran_forest_show_how_info)
    def _():
        show_ran_forest_how_gen_button = input.ran_forest_show_how_info()
        if show_ran_forest_how_gen_button % 2 == 1:
            ui.update_action_button("ran_forest_show_how_info", label="¿Cómo se genera el modelo de bosque aleatorio? △")
            ui.insert_ui(
                ui.div({"id": "inserted-ran-forest-how-gen-info"},
                    ui.markdown("""Todos los modelos siguen los mismos pasos para ser creados:
- Primero debemos **elegir los ajustes del modelo** que queremos crear. En este caso, disponemos de los siguientes ajustes (la mayoría son análogos a los de árbol de decisión):
    - **Num estimators**: el número de árboles para generar el bosque.
    - **Criterion**: La función utilizada para medir la calidad de una división.
    - **Max Depth**: La profundidad máxima del árbol. Si es None, los nodos se expandirán hasta que todas las hojas sean puras o hasta que todas las hojas contengan menos muestras que min_samples_split.
    - **Min samples split**: El número mínimo de muestras requeridas para dividir un nodo interno.
    - **Min samples leaf**: El número mínimo de muestras requeridas para estar en un nodo hoja.
    - **Max features**: El número de características a considerar al buscar la mejor división.
- Después debemos **elegir las características** que queremos usar para predecir el resultado. No todas las características pueden ser relevantes para el modelo y puede que nos encontremos algunas que aporten ruido a nuestros resultados. Si es la primera vez que creas el modelo, selecciona todas las características de momento.
- Por último, **¡genera el modelo!**"""
                    ),
                    style="border: solid 0px grey; border-radius: 10px; background:#eceef1 ;margin-right:50px; padding:15px 20px 10px 20px; text-align: justify; text-justify: inter-word;",
                ),
                selector="#ran_forest_how_generate",
                where="beforeEnd",
            )
        else:
            ui.update_action_button("ran_forest_show_how_info", label="¿Cómo se genera el modelo de bosque aleatorio? ▽")
            ui.remove_ui("#inserted-ran-forest-how-gen-info")

    @output
    @render.image
    def ran_forest_expl_image():
        img: ImgData = {"src": str(explanation_img_path / "ran_forest_expl.png"), "height":"250px", "style":"display:block; margin-left:15%;"}
        return img