from shiny import module, ui, reactive, render
from shiny.types import ImgData
from pathlib import Path

explanation_img_path = Path(__file__).parent.parent / "images"


@module.ui
def logReg_def_ui():
    return ui.div(
        ui.div(
            ui.markdown("El análisis de regresión logística es un método ampliamente utilizado para la clasificación supervisada. **Se basa en la regresión ordinaria** y se utiliza para modelar variables dicotómicas, que representan la presencia o ausencia de un evento en particular. El objetivo principal es **calcular la probabilidad de que una nueva instancia pertenezca a una clase específica**. Los resultados de una regresión logística son valores de probabilidad en el rango de 0 a 1. Para utilizarlo como clasificador binario, se establece un umbral que ayuda a distinguir entre las dos clases.")
            , style="padding-right:50px; text-align: justify; text-justify: inter-word;" 
        ),
        ui.div(
            ui.markdown("A continuación, se muestra cómo la regresión logística genera una curva de la probabilidad de que el valor en el eje X suponga que la muestra sea de una de las dos clases (representadas en el eje Y con valores 1 y 0). Cuando existen múltiples características es complicado representar una regresión logística ya que requeriría una representación multidimensional.")
            , style="padding-right:50px; padding-bottom:10px; text-align: justify; text-justify: inter-word;" 
        ),
        ui.output_image("log_reg_expl_image", height="260px"),
    )

@module.ui
def logReg_howTo_ui():
    return ui.div(
        {"id": "log_reg_how_generate"},
        ui.input_action_button("log_reg_show_how_info", "¿Cómo se genera el modelo de regresión logística? ▽"
                        , style="padding: 30px 0px 10px 0px; background: white; border: none; font-weight: bold; text-decoration: underline; border: 0 !important; box-shadow: 0 0 !important; transition: 0.1s !important; background-color: transparent !important;"),
                            
    )

@module.ui
def logReg_performance_ui():
    return ui.div(
        ui.div(
            ui.markdown("""**No hay un umbral exacto para considerar un modelo como bueno**, ya que depende del contexto y las necesidades del problema. En general, en aplicaciones relacionadas con el ámbito sanitario se busca maximizar tanto la precisión (para minimizar falsos positivos) como la sensibilidad o TVP (para minimizar falsos negativos), por lo que **se busca obtener un valor alto de F1**. En este ejemplo el valor de F1 puede llegar a superar el 90% utilizando los ajustes y características correctos. El algoritmo de regresión logística es mucho más complicado de sobreajustar que los dos anteriores.""")
            , style="padding-top:30px; padding-right:50px; text-align: justify; text-justify: inter-word;" 
        ),
    )


@module.server
def logReg_server(input, output, session):
    
    @reactive.Effect
    @reactive.event(input.log_reg_show_how_info)
    def _():
        show_log_reg_how_gen_button = input.log_reg_show_how_info()
        if show_log_reg_how_gen_button % 2 == 1:
            ui.update_action_button("log_reg_show_how_info", label="¿Cómo se genera el modelo de regresión logística? △")
            ui.insert_ui(
                ui.div({"id": "inserted-log-reg-how-gen-info"},
                    ui.markdown("""Todos los modelos siguen los mismos pasos para ser creados:
- Primero debemos **elegir los ajustes del modelo** que queremos crear. En este caso, disponemos de los siguientes ajustes:
    - **Solver**: algoritmo utilizado para la optimización del problema. No todas las opciones y permiten todos los tipos de penalty.
    - **Penalty**: Especifica la norma de la penalización.
    - **Tolerance**: Tolerancia para el criterio de detención.
    - **C**: Inversa de la fuerza de regularización. Valores más pequeños especifican una regularización más fuerte.
    - **Max iter**: Número máximo de iteraciones realizadas para que los solucionadores converjan.
- Después debemos **elegir las características** que queremos usar para predecir el resultado. No todas las características pueden ser relevantes para el modelo y puede que nos encontremos algunas que aporten ruido a nuestros resultados. Si es la primera vez que creas el modelo, selecciona todas las características de momento.
- Por último, **¡genera el modelo!**"""
                    ),
                    style="border: solid 0px grey; border-radius: 10px; background:#eceef1 ;margin-right:50px; padding:15px 20px 10px 20px; text-align: justify; text-justify: inter-word;",
                ),
                selector="#log_reg_how_generate",
                where="beforeEnd",
            )
        else:
            ui.update_action_button("log_reg_show_how_info", label="¿Cómo se genera el modelo de regresión logística? ▽")
            ui.remove_ui("#inserted-log-reg-how-gen-info")

    @output
    @render.image
    def log_reg_expl_image():
        img: ImgData = {"src": str(explanation_img_path / "log_reg_expl.png"), "height":"250px", "style":"display:block; margin-left:25%;"}
        return img