from shiny import module, ui, reactive, render
from shiny.types import ImgData
from pathlib import Path

explanation_img_path = Path(__file__).parent.parent / "images"


@module.ui
def decTree_def_ui():
    return ui.div(
        ui.div(
            ui.markdown("Un árbol de decisión es un algoritmo de aprendizaje automático que se utiliza para **clasificar elementos de datos siguiendo una estructura similar a un árbol**. Partiendo del nodo más alto, el nodo raíz, cada nodo del árbol representa una prueba en una variable de entrada o atributo. Dependiendo del resultado de dicha prueba, el algoritmo se bifurca hacia el siguiente nodo correspondiente en un nuevo nivel hasta llegar a un nodo hoja, que representa la decisión final de clasificación. Los nodos hoja contienen los resultados de clasificación. El árbol de decisión es una **forma intuitiva de modelar la lógica de toma de decisiones** y ha sido uno de los algoritmos más utilizados en el aprendizaje automático.")
            , style="padding-right:50px; text-align: justify; text-justify: inter-word;" 
        ),
        ui.div(
            ui.markdown("A continuación, se muestra un ejemplo de una representación de un árbol de decisión simple, que cuenta con 3 variables (C1, C2 y C3). Según los resultados de las pruebas con dichas variables se termina clasificando la muestra en una de las dos clases existentes.")
            , style="padding-right:50px; padding-bottom:10px; text-align: justify; text-justify: inter-word;" 
        ),
        ui.output_image("dec_tree_expl_image", height="260px"),
    )

@module.ui
def decTree_howTo_ui():
    return ui.div(
        {"id": "dec_tree_how_generate"},
        ui.input_action_button("dec_tree_show_how_info", "¿Cómo se genera el modelo de árbol de decisión? ▽"
                        , style="padding: 30px 0px 10px 0px; background: white; border: none; font-weight: bold; text-decoration: underline; border: 0 !important; box-shadow: 0 0 !important; transition: 0.1s !important; background-color: transparent !important;"),
                            
    )

@module.ui
def decTree_performance_ui():
    return ui.div(
        ui.div(
            ui.markdown("""**No hay un umbral exacto para considerar un modelo como bueno**, ya que depende del contexto y las necesidades del problema. En general, en aplicaciones relacionadas con el ámbito sanitario se busca maximizar tanto la precisión (para minimizar falsos positivos) como la sensibilidad o TVP (para minimizar falsos negativos), por lo que **se busca obtener un valor alto de F1**. En este ejemplo el valor de F1 puede superar el 90% pero es muy fácil sobreajustar el modelo con un árbol de decisión.
                        
        
*Consejo: editar la profundidad máxima del árbol es un buen punto de inicio para evitar el sobreajuste.*""")
            , style="padding-top:30px; padding-right:50px; text-align: justify; text-justify: inter-word;" 
        ),
    )


@module.server
def decTree_server(input, output, session):
    
    @reactive.Effect
    @reactive.event(input.dec_tree_show_how_info)
    def _():
        show_dec_tree_how_gen_button = input.dec_tree_show_how_info()
        if show_dec_tree_how_gen_button % 2 == 1:
            ui.update_action_button("dec_tree_show_how_info", label="¿Cómo se genera el modelo de árbol de decisión? △")
            ui.insert_ui(
                ui.div({"id": "inserted-dec-tree-how-gen-info"},
                    ui.markdown("""Todos los modelos siguen los mismos pasos para ser creados:
- Primero debemos **elegir los ajustes del modelo** que queremos crear. En este caso, disponemos de los siguientes ajustes:
    - **Criterion**: La función utilizada para medir la calidad de una división.
    - **Splitter**: La estrategia utilizada para elegir la división en cada nodo.
    - **Max Depth**: La profundidad máxima del árbol. Si es None, los nodos se expandirán hasta que todas las hojas sean puras o hasta que todas las hojas contengan menos muestras que min_samples_split.
    - **Min samples split**: El número mínimo de muestras requeridas para dividir un nodo interno.
    - **Min samples leaf**: El número mínimo de muestras requeridas para estar en un nodo hoja.
    - **Max features**: El número de características a considerar al buscar la mejor división.
- Después debemos **elegir las características** que queremos usar para predecir el resultado. No todas las características pueden ser relevantes para el modelo y puede que nos encontremos algunas que aporten ruido a nuestros resultados. Si es la primera vez que creas el modelo, selecciona todas las características de momento.
- Por último, **¡genera el modelo!**"""
                    ),
                    style="border: solid 0px grey; border-radius: 10px; background:#eceef1 ;margin-right:50px; padding:15px 20px 10px 20px; text-align: justify; text-justify: inter-word;",
                ),
                selector="#dec_tree_how_generate",
                where="beforeEnd",
            )
        else:
            ui.update_action_button("dec_tree_show_how_info", label="¿Cómo se genera el modelo de árbol de decisión? ▽")
            ui.remove_ui("#inserted-dec-tree-how-gen-info")

    @output
    @render.image
    def dec_tree_expl_image():
        img: ImgData = {"src": str(explanation_img_path / "dec_tree_expl.png"), "height":"250px", "style":"display:block; margin-left:25%;"}
        return img