import gradio as gr
import numpy as np
import matplotlib.pyplot as plt
import tensorflow as tf

# Helper functions
def cos_mat(theta, n):
    return np.sqrt(1 - (np.sin(np.radians(theta))/n)**2)

def calculate_R(theta, n):
  numerator = n * cos_mat(theta,n) - np.cos(np.radians(theta))
  denominator = n * cos_mat(theta, n) + np.cos(np.radians(theta))
  return (numerator/denominator)**2

def calculate_phase(theta, n, l, lamb):
  return (2*np.pi*n/lamb) *(2*l*cos_mat(theta, n))

# Calculate transmission - ground truth
def trans(lamb, n, l, theta):
  R = calculate_R(theta, n)
  F = 4*R/((1 - R)**2)
  delta = calculate_phase(theta, n, l, lamb)
  return F, delta[0] * lamb[0], (1 + F*np.sin(delta/2)**2)**(-1)

# create swish activation function
def swish(x):
    return x*tf.keras.activations.sigmoid(x)
tf.keras.utils.get_custom_objects()['swish'] = tf.keras.layers.Activation(swish)

# Normalizations of Fd to interval [0, 1]
def Fd_normalize(double):
  double = double - np.array([11.8966, 21697])
  double = np.divide(2*double, [23.8, 43394])
  return double

def Fd_unnormalize(double):
  double = np.multiply(double/2, [23.8, 43394])
  double = double + np.array([11.8966, 21697])
  return double

# Main function
def get_transmission(n, l, theta):

    # Compute ground truth
    lamb = np.arange(400, 800, 2).astype('float32')
    F, delta, transmission = trans(lamb, n, l, theta)

    # Make predictions
    Fulcon = tf.keras.models.load_model('network_fabry_perot.hdf5',
                                        custom_objects={'Activation' : tf.keras.layers.Activation(swish)})
    input = Fd_normalize(np.array([[F, delta]]))
    pred = Fulcon.predict(tf.constant(input))

    # Clear figure
    plt.clf()

    # Plot ground truth
    plt.plot(lamb, transmission, label='Ground truth')

    # Plot network prediction
    plt.plot(lamb, pred[0], 'r', label='Prediction')

    # Layout
    plt.xlabel("Wavelength (nm)", fontsize=14)
    plt.ylabel("Transmission", fontsize=14)
    plt.xticks(fontsize=14)
    plt.yticks(fontsize=14)
    plt.ylim(0, 1.05)
    plt.title(r"Transmission T($\lambda$)", fontsize=16)
    plt.legend()

    return plt.gcf()

iface = gr.Interface(fn=get_transmission,
                     inputs=[gr.inputs.Slider(minimum=1.05, maximum=3.55, label='Index of refraction (n)'),
                     gr.inputs.Slider(minimum=100, maximum=1000, label='Thickness (l)'),
                     gr.inputs.Slider(minimum=-70,maximum=70, label=r'Incident angle (theta)')],
                     outputs="plot",
                     title="Fabry-Pérot resonator",
                     description=r"Gradio demo for a neural network trained to predict the transmission spectrum " +
                     "of a Fabry-Pérot resonator. More info can be found on https://github.com/Joeri38/inverse-design.",
                     allow_flagging='never')
iface.launch() 
