import gradio as gr
import torch
import torchvision.transforms as transforms
from model import ImageClassificationModel
import os
from matplotlib import pyplot as plt
import numpy as np
import io
from PIL import Image
import seaborn as sns

# Get example images
example_images = [
    os.path.join(root, file)
    for root, dirs, files in os.walk("example_images")
    for file in files
]
# random sort
example_images.sort(key=lambda x: np.random.random())

HF_MODEL_NAME = "google/vit-base-patch16-224-in21k"

# Load model and set to evaluation mode
model = ImageClassificationModel.load_from_checkpoint(
    "models/DOCNone vit-base-patch16-224-in21k-finetune224.ckpt",
    HF_MODEL_NAME=HF_MODEL_NAME,
)
model.eval()


def plot_logits(scores: dict[str, float]):
    """Visualize the logits using Seaborn."""
    labels = list(scores.keys())
    logits = list(scores.values())
    sns.set_style("whitegrid")
    plt.figure(figsize=(10, 6))

    sns.barplot(x=labels, y=logits, palette="coolwarm", hue=labels)
    plt.ylim(
        -max(abs(min(logits)), abs(max(logits))),
        max(abs(min(logits)), abs(max(logits))),
    )
    plt.axhline(0, color="black", linewidth=0.8)
    plt.yticks(fontsize=10)
    plt.xticks(rotation=45, fontsize=25)
    plt.title("Model Logits", fontsize=45)
    plt.tight_layout(pad=1.0)
    buf = io.BytesIO()
    plt.savefig(buf, format="png")
    buf.seek(0)
    img = Image.open(buf)
    return np.array(img)


def predict(img) -> tuple[str, np.ndarray]:
    """Predict the image label and visualize logits."""
    try:
        transform = transforms.Compose(
            [
                transforms.Resize((224, 224)),
                transforms.ToTensor(),
                transforms.Normalize(mean=0.5, std=0.5),
            ]
        )
        img = img.convert("RGB")
        inputs = transform(img).unsqueeze(0)
        labels = ["None", "Cat", "Mouse", "Possum", "Rat", "Stoat"]

        with torch.no_grad():
            logits = model(inputs).logits
            raw_scores = {labels[i]: float(logits[0][i]) for i in range(6)}
        predicted_label = labels[np.argmax(logits)]
        plot_img = plot_logits(raw_scores)
        return predicted_label, plot_img
    except Exception as e:
        return f"Error: {e}", np.zeros((224, 224, 3))


def main():
    with gr.Blocks() as app:
        # Header markdown
        gr.Markdown(
            """
            <center><h1>Identify Pest Predators Demo</h1></span> 
            <span font-size:16px;">Upload an image or use an example image at the bottom to get the classification prediction and logits visualisation.</span>  
            </center>
            """
        )

        # Row container
        with gr.Row():
            # First column
            with gr.Column():
                input_img = gr.Image(label="Upload Image", type="pil")
                gr.Examples(examples=example_images, inputs=input_img, examples_per_page=6)
                run_btn = gr.Button("Predict")

            # Second column
            with gr.Column():
                pred_label = gr.Label(label="Prediction")
                logits_plot = gr.Image(label="Logits Plot", interactive=False)

        run_btn.click(predict, inputs=[input_img], outputs=[pred_label, logits_plot])

    with gr.Blocks() as main_app:
        with gr.Tab("Demo"):
            app.render()

    main_app.launch()


if __name__ == "__main__":
    main()
