##############################################################################################################
##################
### 1. Imports ###
##################

# General Imports
from PIL import Image

# Deeplearning 
import torch 
import gradio as gr
from transformers import pipeline # will use for image captioning 
from diffusers import StableDiffusionPipeline # Text to image 


##############################################################################################################
########################
### 2. Model loading ###
########################

# load image captioning model
img_cap_pipe = pipeline("image-to-text",model="Salesforce/blip-image-captioning-base")

# load image generator model
img_diff_pipe = StableDiffusionPipeline.from_pretrained("runwayml/stable-diffusion-v1-5", torch_dtype=torch.float16)
img_diff_pipe = img_diff_pipe.to("cuda")


##############################################################################################################
###########################
### 3. Helper Functions ###
###########################

def caption_img(image:str):
    """
    Wrapper for image captioning with model Salesforce/blip-image-captioning-base
    """
    # use the loaded pipeline to produce the NER predictions
    output = img_cap_pipe(image)

    # extract the output
    output = output[0]['generated_text']

    return output

def generate_img(prompt, negative_prompt, num_inference_steps, guidance_scale, width, height):
    """
    Wrapper that allows for more parameters passed to the diffussion model.
    """

    # call model with input parameters
    output =  img_diff_pipe(prompt = prompt,
                        negative_prompt= negative_prompt,
                        num_inference_steps=num_inference_steps,
                        guidance_scale = guidance_scale,
                        width = width,
                        height = height
                        )

    # extract the first image
    output = output.images[0]

    return output

##############################################################################################################
##############
### 4. App ###
##############

# Create app markdown
markdown = """
# Captioning and Stable Diffusion Generation 

Welcome to my app! This app uses a very simple Generative AI constructed from pretrained deep-learning models in two steps:
1. Image captioning (image-to-text) using Salesforce/blip-image-captioning-base and the `transformers` HugginFace API
2. Image generation (text-to-image) using runwayml/stable-diffusion-v1-5 and `StableDiffusionPipeline` from `diffusers`

Instructions: 
1. Copy-paste or upload an image in the input box. 
2. Click on "Generate caption" to see what the model thinks your image is about. 
3. Either re-write the prompt, or leave it as is. 
4. (Optional) Write a negative prompt , which allows the model to avoid including content related to this prompt 
   in the generated image. 
5. (Optional) Adjust the number of inference steps and the guidance hyperparameters along with the desired 
width and height of the output image. 
6. Click on "Generate Image" 

Enjoy! :) 
"""

# Encapsulate the demo in Blocks
with gr.Blocks() as demo:

    ### 0. General

    # Initialize a markdown for the app
    gr.Markdown(markdown)

    ### 1. Image Captioning part

    # Input for image
    image_upload = gr.Image(label="Your first image",type="pil")

    # Trigger the caption
    btn_caption = gr.Button("Generate caption")

    # output field for the generated caption
    caption = gr.Textbox(label="Generated caption")

    ### 2. Image Generation from caption

    # optimize to avoid this content in the image
    negative_prompt = gr.Textbox(value="low quality", label="Negative prompt", 
                                 info="The generation algorithm will attempt to remove this content in the generated image")

    ## 3. Setup the generation pipeline

    # First row
    with gr.Row():

        # First column: Inputs
        with gr.Column():

            # Optional generation parameters
            steps = gr.Slider(label="Inference Steps", minimum=1, maximum=100, value=25,
                      info="In how many steps will the denoiser denoise the image?")
            guidance = gr.Slider(label="Guidance Scale", minimum=1, maximum=20, value=7,
                      info="Controls how much the text prompt influences the result")
            width = gr.Slider(label="Width", minimum=64, maximum=512, step=64, value=256)
            height = gr.Slider(label="Height", minimum=64, maximum=512, step=64, value=256)

        # Second column: Outputs
        with gr.Column():
            # trigger the image generation
            btn_image = gr.Button("Generate image")

            # Capture the image output
            image_output = gr.Image(label="Generated Image")

    # trigger image captioning nad generation on click
    btn_caption.click(fn=caption_img, inputs=[image_upload], outputs=[caption])
    btn_image.click(fn=generate_img, inputs=[caption,negative_prompt,steps,guidance,width,height], outputs=[image_output])

# Close all instances
gr.close_all()

# Launch demo
demo.launch()