import gradio as gr
import json
import pandas as pd
import os

import openai
import pinecone

INDEX_NAME = "index-poc-ada"
ML_MODEL_NAME = "text-embedding-ada-002"
PINECONE_API_KEY = os.environ["PINECONE_API_KEY"]
PINECONE_ENV = "us-west1-gcp"
APPLICATION_TITLE = "Accuracy Testing - GPT Ada"
OPENAI_API_KEY = os.environ["OPENAI_API_KEY"]

intent_filters = {
    "md-ada-21K": [
        "",
        "To report a payment related issue.",
        "Get Order Status",
        "Change Delivery Schedule",
        "Change Delivery Address",
        "Report Order Issue",
        "Get Product Details",
        "Check Delivery Status",
        "Check Refund Details",
        "Unknown",
        "Report Missing Item",
        "Make Purchase",
        "Get Process Details",
        "Check Return Details",
        "Cancel Order",
        "Get Discount",
        "Request Refund",
        "Request Return Order",
        "Report Delivery Issue",
        "Update Account",
        "Get Delivery Status",
        "Report Payment Issue",
        "Report Account Issue",
        "End A Conversation",
        "Request Contact Details",
        "Issue Resolved",
        "Choose Pickup Option",
        "Get Billing Details",
        "Cancel Return",
        "Make Payment",
        "Report Product Issue",
        "Report Tech Issue",
        "Report Service Issue",
        "Visit Store",
        "Request Return Order",
        "Find Product",
        "Request Order Details",
        "Get Pickup Information",
        "To report a payment related issue",
        "Contact Service Representative",
        "Report Refund Issue",
        "Get Account Details",
        "Update Payment Method",
        "Report Return Issue",
    ],
    "oportun-ada-45k": [
        "",
        "Update Account",
        "Raise Payment Issue",
        "Unlisted",
        "Check Status",
        "Modify Payment",
        "Setup Payment Plan",
        "Extend Payment",
        "Make Payment",
        "Refinance Loan",
        "Update Payment Method",
        "Request Document",
        "Request Loan",
        "Raise Record Issue",
        "Get Product Info",
        "Raise Customer Service Issue",
        "Raise Process Issue",
        "Agree To Terms",
        "Authorize Payment",
        "Cancel Loan",
        "Cancel Payment",
        "Check Account Status",
        "Check Application Status",
        "Check Loan Status",
        "Check Payment Date",
        "Check Payment Status",
        "Check Store Details",
        "Complete Process",
        "Confirm Account Details",
        "Confirm Transaction Details",
        "Create An Account",
        "Dissatisfied Customer",
        "Get Call Purpose",
        "Get Customer Support",
        "Get Process Details",
        "Get Promotional info",
        "Issue Resolved",
        "Make Purchase",
        "Raise Account Issue",
        "Reason For Missed Payment",
        "Reject Loan Offer",
        "Request Another Loan",
        "Request For Change",
        "Transfer Call",
        "Unknown",
        "Verify Authorization",
    ],
}

lob_filters = {
    "md-ada-21K": ["", "Services"],
    "oportun-ada-45k": [
        "",
        "Collections - UPL",
        "Customer Care - UPL",
        "Originations - SPL",
        "PAS - UPL",
        "Collections - SPL",
        "Customer Care - SPL",
        "Collections",
    ],
}

namespace_values = ["md-ada-21K", "oportun-ada-45k"]


def initialize_pinecone():
    openai.api_key = OPENAI_API_KEY
    pinecone.init(api_key=PINECONE_API_KEY, environment=PINECONE_ENV)


def get_index():
    pinecone_index = pinecone.Index(INDEX_NAME)

    print("Found Index: ", pinecone_index.describe_index_stats())
    return pinecone_index


def get_gpt_ada_embedding(text, engine=ML_MODEL_NAME):
    text = text.replace("\n", " ")
    try:
        return openai.Embedding.create(input=[text], model=engine)["data"][0][
            "embedding"
        ]
    except openai.error.APIError as e:
        print("APIError generating embedding for text" + text)
        return []


def validate_input(input):
    if input.strip() == "":
        raise ValueError("Input cannot be empty")
    return input


def search_query(
    query,
    nameSpace,
    lobFilter,
    authorTypeFilter,
    intentFilter,
    languageFilterterList,
    topK,
):
    topK = 1 if topK <= 0 else topK
    validate_input(query)

    print(
        f"Showing {int(topK)} TopK results for Query: '{query}' in '{nameSpace} namespace'\nFilters [{lobFilter, authorTypeFilter, intentFilter, languageFilterterList}]"
    )
    # vectors = vecotrize_search_query(query)
    vectors = get_gpt_ada_embedding(query)

    filterList = {
        k: v
        for k, v in {
            "lob": lobFilter,
            "authorType": authorTypeFilter,
            "emotion": intentFilter,
            "languageName": languageFilterterList,
        }.items()
        if v
    }

    res = pinecone_index.query(
        vector=vectors,
        filter=filterList,
        namespace=nameSpace,
        top_k=topK,
        include_metadata=True
        # include_values=True,
    )

    res_array = []

    for obj in res["matches"]:
        new_obj = {
            "postId": obj["id"],
            "score": obj["score"],
            "lob": obj["metadata"]["lob"],
            "language": obj["metadata"]["languageName"],
            "intent": obj["metadata"]["intent"],
            "authorType": obj["metadata"]["authorType"],
            "authorId": obj["metadata"]["authorId"],
            "conversationId": obj["metadata"]["conversationId"],
        }

        res_array.append(new_obj)

    return pd.DataFrame(res_array)


def get_lob_filter_choices(ns):
    return lob_filters[ns]


def get_intent_filter_choices(ns):
    return intent_filters[ns]


def launch_app(func):
    with gr.Blocks(
        title=APPLICATION_TITLE,
        css=".gradio-container {font-weight: bold;}",
    ) as demo:
        gr.components.Text(label="Application", value=APPLICATION_TITLE)

        with gr.Row(variant="compact"):
            nameSpace = gr.components.Dropdown(
                label="Namespace",
                choices=namespace_values,
                value=namespace_values[0],
            )
            topK = gr.components.Number(label="Top K result", value=1)

        name = gr.components.Textbox(label="Search Query")

        with gr.Row(variant="compact"):
            lobFilter = gr.components.Dropdown(
                label="LOB filter",
                choices=get_lob_filter_choices(nameSpace.value),
                value="",
            )

            intentFilter = gr.Dropdown(
                label="Intent Filter",
                choices=get_intent_filter_choices(nameSpace.value),
                value="",
            )

            authorTypeFilter = gr.components.Dropdown(
                label="Author Type", choices=["agent", "customer"]
            )

            languageFilter = gr.components.Dropdown(
                label="Language", choices=["English", "Spanish"]
            )

        greet_btn = gr.components.Button("SEARCH")
                
        header_data = [
            "postId",
            "score",
            "lob",
            "language",
            "authorType",
            "intent",
            "authorId",
            "conversationId",
        ]
        output_result = gr.components.DataFrame(
            label="Result", headers=header_data, type="array", interactive=False, overflow_row_behaviour = "paginate"
        )

        greet_btn.click(
            fn=func,
            inputs=[
                name,
                nameSpace,
                lobFilter,
                authorTypeFilter,
                intentFilter,
                languageFilter,
                topK,
            ],
            outputs=output_result,
        )

        def update_lob_filter_choices(curr_ns):
            updatedChoices = get_lob_filter_choices(curr_ns)

            return gr.Dropdown.update(choices=updatedChoices, value=updatedChoices[0])

        def update_intent_filter_choices(curr_ns):
            updatedChoices = get_intent_filter_choices(curr_ns)

            return gr.Dropdown.update(choices=updatedChoices, value=updatedChoices[0])

        nameSpace.change(update_lob_filter_choices, nameSpace, lobFilter)
        nameSpace.change(update_intent_filter_choices, nameSpace, intentFilter)

    demo.launch()


initialize_pinecone()
pinecone_index = get_index()
launch_app(search_query)
