#!/usr/bin/env python

from __future__ import annotations

import gradio as gr

from paper_list import PaperList

DESCRIPTION = """\
# ICCV 2023 Papers

- https://iccv2023.thecvf.com
- https://openaccess.thecvf.com/ICCV2023
"""

TUTORIAL = """\
#### Hugging Face ICCV 2023 event

Join the org using this [link](https://huggingface.co/organizations/ICCV2023/share/BXikCrBMdmnKzsFjVSulFrCsQAujoYGiNG)

ICCV 2023 organization is accepting paper claims from authors attending ICCV 2023.

This organization invites participants to claim their ICCV 2023 papers, upload models and datasets, and to build their Gradio demos for conference papers on huggingface.

#### Hugging Face Paper Pages ICCV 2023

- ICCV 2023 Paper Pages will allow authors to claim their papers on Huggingface. Claiming papers on ICCV 2023 organization will allow people to find artifacts related to a paper such as models, datasets and Gradio demos (in form of Spaces). This also enables the community to discuss about the paper.

#### Tutorial for claiming the ICCV 2023 papers

Visit the [demo](https://huggingface.co/spaces/ICCV2023/ICCV2023-papers) and find your paper, to claim your paper click on the pager page link in the table and then click on your name in the corresponding Paper page and click “claim authorship”.  This will automatically re-direct to your paper settings where you can confirm the request. The admin team will validate your request soon. Once confirmed, the Paper page will show as verified.

If you need further assistance, see the guide [here](https://huggingface.co/docs/hub/paper-pages#claiming-authorship-to-a-paper)

If your paper is not yet indexed on huggingface, you can index it by following this [guide](https://huggingface.co/docs/hub/paper-pages#can-i-have-a-paper-page-even-if-i-have-no-modeldatasetspace) and open a [PR](https://huggingface.co/spaces/ICCV2023/ICCV2023-papers/discussions) to add your paper to the hugging face demo.
"""


paper_list = PaperList()

with gr.Blocks(css="style.css") as demo:
    gr.Markdown(DESCRIPTION)
    with gr.Accordion(label="Tutorial", open=True):
        gr.Markdown(TUTORIAL)

    search_box = gr.Textbox(
        label="Search Title", placeholder="You can search for titles with regular expressions. e.g. (?<!sur)face"
    )
    case_sensitive = gr.Checkbox(label="Case Sensitive")
    filter_names = gr.CheckboxGroup(
        label="Filter",
        choices=[
            "arXiv",
            "GitHub",
            "Space",
            "Model",
            "Dataset",
        ],
    )
    presentation_type = gr.Radio(label="Presentation Type", choices=["(ALL)", "Oral", "Poster"], value="(ALL)")
    search_button = gr.Button("Search")

    number_of_papers = gr.Textbox(label="Number of Papers Found")
    table = gr.HTML(show_label=False)

    demo.load(
        fn=paper_list.render,
        inputs=[
            search_box,
            case_sensitive,
            filter_names,
            presentation_type,
        ],
        outputs=[
            number_of_papers,
            table,
        ],
        api_name=False,
    )

    gr.on(
        triggers=[
            search_box.submit,
            search_button.click,
        ],
        fn=paper_list.render,
        inputs=[
            search_box,
            case_sensitive,
            filter_names,
            presentation_type,
        ],
        outputs=[
            number_of_papers,
            table,
        ],
        api_name=False,
    )

if __name__ == "__main__":
    demo.queue(api_open=False).launch()
