# import libraries
from tensorflow import keras
import gradio as gr
import pickle


# define constants from training
MODEL_PATH = "./model.pickle"
TOKENIZER_PATH = "./tokenizer.pickle"
INDEX_CLASS = {1:'Suicidal', 0:'non-Suicidal'}
MAX_SEQ_DF = 128


# load the tokenizer used on data when training
with open(TOKENIZER_PATH, 'rb') as tokenizer_file:
    tokenizer = pickle.load(tokenizer_file)
# load the mode and read the trained model
with open(MODEL_PATH, 'rb') as model_file:
    model = pickle.load(model_file)


# define prediction function
def predict(input_text):
    # preprocessing
    tokenized_data = tokenizer.texts_to_sequences([input_text])
    text_data_padded = keras.preprocessing.sequence.pad_sequences(tokenized_data, 
                                                                     maxlen = MAX_SEQ_DF, 
                                                                     padding = 'post')

    # make prediction
    pred = model.predict(text_data_padded)
    prediction = INDEX_CLASS[round(pred[0][0])]

    return prediction

content = """In today's world, mental health is a growing concern, especially for young people. According to the World Health Organization, suicide is the second leading cause of death among 15-29-year-olds globally, and early identification and intervention can save lives.

The issue of suicides and mental health in Iraq is an understudied one due to a lack of organizations focused on measuring, evaluating, or intervening in such cases. This is concerning given that Iraq continues to experience instability.

Suicidal ideation, also known as suicidal thoughts, refers to an individual's plans to commit suicide and can be an indicator of suicide risk. Suicidal thoughts can range from brief to substantial and may include significant planning, role-playing, and failed attempts. Adolescents are particularly prone to experiencing suicidal ideation. Early detection of suicidal ideation is an effective technique for preventing suicides.

Globally, nearly 800,000 people commit suicide each year, with a suicide rate of 10.5 per 100,000 people. The majority of suicides (79%) occur in low- and middle-income countries where resources for identification and management of mental health issues are often scarce and insufficient. Suicide ideation ranges from depression to an intense preoccupation with self-destruction and can be categorized as suicide ideators or planners and suicide attempters or completers. While some studies indicate that most individuals with suicidal ideation do not attempt suicide, other studies suggest that there is significant overlap between suicide ideators and attempters.

Social media has become a valuable tool for monitoring the mental health and well-being of its users, particularly young individuals. Online communication platforms such as forums, tweets, and blogs provide a space for people to openly discuss their mental health and to seek support and advice. This includes communication of suicidal tendencies, making social media an important venue for early detection and prevention of suicides. By using machine learning techniques and Natural Language Processing (NLP) methodology, researchers are now able to detect suicidal ideation in social media posts and provide early intervention and support.
"""



example_1 = ("I am feeling very hopeless and helpless lately Everything seems "
            "to be going wrong in my life and I don't see any way out. I have "
            "been having thoughts of ending it all and I just can't shake them "
            "I am afraid of what will happen if I don't get help soon.")

example_2 = ("Today was a wonderful day, I woke up feeling refreshed and ready "
            "to tackle the day. I had a delicious breakfast and went for a walk "
            "in the park. I met some old friends and we chatted for hours "
            "The weather was perfect and I felt so grateful for the beautiful "
            "sunshine. Afterwards, I went shopping and treated myself to some "
            "new clothes. I can't wait to see what tomorrow brings")

img_url = "https://healthmatters.nyp.org/wp-content/uploads/2019/09/warning-signs-of-suicide-hero.jpg"

description = (f"<img src='{img_url}' width='100%' alt='title image' "
            "style='height: 250px; object-fit: cover; border-radius: 15px;'>"
            "<br> Our project aims to create an AI-powered web interface "
            "that can quickly and accurately detect signs of suicidal thoughts"
            " in text inputs. The model is trained on a dataset of texts "
            "and utilizes state-of-the-art NLP techniques to analyze the "
            "sentiment and context of the text. The interface is easy to use"
            " and can provide immediate feedback to individuals who may be at "
            "risk of suicide. The goal of this project is to raise awareness "
            "about mental health and to provide a tool that can help "
            "individuals get the help they need.")

resources = ["https://www.verywellmind.com/how-to-talk-to-a-friend-who-has-lost-someone-to-suicide-6543351",
             "https://www.verywellmind.com/best-depression-resources-and-organizations-5114534",
             "https://healthmatters.nyp.org/how-to-spot-the-potential-warning-signs-of-suicide/",
             "https://988lifeline.org/",]

articale = (f"{content}"
            "<br> <h1>Some resources might help</h1> <ul> <li>"
            f"<a href='{resources[0]}'>How to talk to a friend</a></li><li>"
            f"<a href='{resources[1]}'>Depression Resources and Organizations</a></li>"
            f"<li><a href='{resources[2]}'>Potential warning signs of suicide</a></li>"
            f"<li><a href='{resources[3]}'>For Support and Treatment</a></li> </ul>")

# make nice web interface
gr_intrfc = gr.Interface(fn=predict, 
                         inputs="text", 
                         title='Suicidal Thoughts Detector',
                         description=description,
                         article=articale,
                         css="* {font-family: sans-serif}",
                         examples=[example_1, example_2], 
                         outputs="text",
                         theme='dark')

gr_intrfc.launch()

