import streamlit as st
from streamlit_extras.tags import tagger_component
import re
import pickle
from bertopic import BERTopic
from keras.models import load_model
from keras.preprocessing.sequence import pad_sequences


# Load the BERTopic model
@st.cache_resource
def load_models():
    return (
        BERTopic.load(r"Models/topic_key_model_130.pkl"),
        load_model(r"Models/tag_model.h5"),
        pickle.load(open(r"Models/token.pkl", "rb")),
        pickle.load(open(r"Models/bin.pkl", "rb")),
    )


# Load the model into memory
bertopic_model, cnn_model, tokenizer, binarizer = load_models()


# Clean the input text
def clean_text(text):
    text = re.sub(r"<.*?>", "", text)
    text = re.sub(r"[^A-Za-z']", " ", text)
    text = re.sub(r"\s+", " ", text)
    return text


# Assign tags to the input text using the CNN model
def tag_cnn_model(text):
    text = clean_text(text)
    text = tokenizer.texts_to_sequences([text])
    text_padded = pad_sequences(text, maxlen=512)
    q_pred = cnn_model.predict(text_padded)
    q_pred = (q_pred >= 0.25).astype(int)
    return binarizer.inverse_transform(q_pred)


# Retrieve the keyphrases from the input text using the KeyBERT model
def output_keybert(text, n):
    new_review = text
    similar_topics, similarity = bertopic_model.find_topics(new_review, top_n=n)
    similar_topics = sorted(similar_topics)
    for i in range(n):
        tags = bertopic_model.get_topic(similar_topics[i], full=True)["KeyBERT"]
        tags = [tag[0] for tag in tags]
        tagger_component(f"Tags from cluster {i+1}:", tags)


# Retrieve the keyphrases from the input text using the Bertopics MMR model
def output_mmr(text, n):
    new_review = text
    similar_topics, similarity = bertopic_model.find_topics(new_review, top_n=n)
    similar_topics = sorted(similar_topics)
    for i in range(n):
        tags = bertopic_model.get_topic(similar_topics[i], full=True)["MMR"]
        tags = [tag[0] for tag in tags]
        tagger_component(f"Tags from cluster {i+1}:", tags)


# Find the most similar topics for the input text using the BERTopic model
def output_bertopic(text, n):
    new_review = text
    similar_topics, similarity = bertopic_model.find_topics(new_review, top_n=n)
    similar_topics = sorted(similar_topics)
    for i in range(n):
        tags = bertopic_model.get_topic(similar_topics[i])
        tags = [tag[0] for tag in tags]
        tagger_component(f"Tags from cluster {i+1}:", tags)


# Display the supervised model page of the app
def supervised_page():
    st.header("Supervised Model")
    text = st.text_area("Enter text to assign tags", height=200, key="supervised_text")
    text = clean_text(text)
    if st.button("Assign tags", key="supervised_button"):
        if text == "":
            st.error("Please enter some text to assign tags")
        else:
            with st.spinner("Assigning tags..."):
                tags = tag_cnn_model(text)[0]
                tagger_component("Tags:", tags)


# Display the unsupervised model using bertopic page of the app
def unsupervised_page_bertopic():
    st.header("Unsupervised Model Using BERTopic Model")
    text = st.text_area(
        "Enter text to assign tags", height=200, key="unsupervised_text_bertopic"
    )
    text = clean_text(text)
    n = st.number_input(
        "Enter number of tags to assign", value=5, key="unsupervised_n_bertopic"
    )
    if st.button("Assign tags", key="unsupervised_button_bertopic"):
        if text == "":
            st.error("Please enter some text to assign tags")
        else:
            with st.spinner("Assigning tags..."):
                output_bertopic(text, n)


def unsupervised_page_keybert():
    st.header("Unsupervised Model Using BERTopic Model")
    text = st.text_area(
        "Enter text to assign tags", height=200, key="unsupervised_text_keybert"
    )
    text = clean_text(text)
    n = st.number_input(
        "Enter number of tags to assign", value=5, key="unsupervised_n_keybert"
    )
    if st.button("Assign tags", key="unsupervised_button_keybert"):
        if text == "":
            st.error("Please enter some text to assign tags")
        else:
            with st.spinner("Assigning tags..."):
                output_keybert(text, n)


# Display the unsupervised model using bertopic page of the app
def unsupervised_page_mmr():
    st.header("Unsupervised Model Using BERTopic Model")
    text = st.text_area(
        "Enter text to assign tags", height=200, key="unsupervised_text_mmr"
    )
    text = clean_text(text)
    n = st.number_input(
        "Enter number of tags to assign", value=5, key="unsupervised_n_mmr"
    )
    if st.button("Assign tags", key="unsupervised_button_mmr"):
        if text == "":
            st.error("Please enter some text to assign tags")
        else:
            with st.spinner("Assigning tags..."):
                output_mmr(text, n)


# Display the model page of the app
def model_page():
    stype_for_page = """
    <style>
    [data-testid="baseButton-secondary"]:hover {
    scale: 1.05;
    transition-duration: 0.3s;
    }
    </style>
    """
    st.markdown(stype_for_page, unsafe_allow_html=True)

    st.title("Select a model to use:")
    with st.container():
        tab1, tab2, tab3, tab4 = st.tabs(
            [
                "Supervised Using CNN",
                "UnSupervised-KeyBERT",
                "UnSupervised-MMR",
                "UnSupervised-BERTopic",
            ]
        )
        with tab1:
            supervised_page()
        with tab2:
            unsupervised_page_keybert()
        with tab3:
            unsupervised_page_mmr()
        with tab4:
            unsupervised_page_bertopic()
    with st.container():
        st.info("Click on the arrow to expand the example texts.")
        with st.expander("Example Texts", expanded=False):
            st.markdown(
                """
                ### Here are 5 examples of questions from Stack Exchange. Try them out!
                - I am trying to build a model to predict the price of a house based on its features. I have a dataset with 79 features and 1460 observations. I have tried using linear regression but it is not giving me good results. What should I do?
                - Can you suggest me some techniques in NLP to improve the performance of my model? I am using a simple LSTM model but it is not giving me good results. I have tried using a GRU model but it is not giving me good results. What should I do?
                - I was doing data cleaning on my dataset and I found that there are some missing values in my dataset. I tried to impute them using mean and median but it is not giving me good results. What should I do?
                - Tell me the difference between a linear regression model and a logistic regression model. I am confused between the two. I have tried to read about them but I am not able to understand the difference between them. What should I do?
                - What is the meaning of p-value? I have tried to read about it but I am not able to understand it. Can you explain it to me in simple terms? In statistics, what is the meaning of pvalue?
                """
            )


if __name__ == "__main__":
    model_page()
