import threading

import av
import numpy as np
import streamlit as st
from streamlit_webrtc import webrtc_streamer

from app_utils import ImgContainer
from base_model import BaseRGBDModel
from color_selection_ui import color_selection_ui
from depth_model import BaseDepthModel
from depth_selection_ui import depth_selection_ui
from model import base_inference
from sod_selection_ui import sod_selection_ui

lock = threading.Lock()

rtc_configuration = {
    "iceServers": [{"urls": [
        "stun:stun1.l.google.com:19302",
        "stun:stun2.l.google.com:19302",
        "stun:stun3.l.google.com:19302",
        "stun:stun4.l.google.com:19302",
    ]}],
}

img_container = ImgContainer()
img_container.frame_rate.reset()

def webrtc_app(
    depth_model: BaseDepthModel, 
    sod_model: BaseRGBDModel, 
    color: np.ndarray
):
    def video_frame_callback(frame: av.VideoFrame) -> av.VideoFrame:
        img: np.ndarray = frame.to_ndarray(format="bgr24")
        with lock:
            img_container.img = img
            img_container.frame_rate.count()
            img = img_container.frame_rate.show_fps(img)
        _, pred_sod, _ = base_inference(
            depth_model, sod_model, img, None, color
        )
        return av.VideoFrame.from_ndarray(pred_sod, format="bgr24")

    
    st.session_state.ctx = webrtc_streamer(
        key="snapshot", 
        video_frame_callback=video_frame_callback,
        rtc_configuration=rtc_configuration
    )

if __name__ == '__main__':
    depth_model = depth_selection_ui(st)
    sod_model = sod_selection_ui(st)
    color = color_selection_ui(st)
    webrtc_app(depth_model, sod_model, color)