import os
import shutil
import time
from typing import List

import cv2
import numpy as np
import streamlit as st
from stqdm import stqdm

from app_utils import load_video
from base_model import BaseRGBDModel
from depth_model import BaseDepthModel
from model import batch_base_inference

video_tmp_path = os.path.join("tmp", 'tmp.mp4')

def video_batch_inference(
    depth_model: BaseDepthModel,
    sod_model: BaseRGBDModel,
    raw_images: List[np.ndarray],
    result_vw: cv2.VideoWriter,
    color: np.ndarray,
) -> None:
    pred_depths, pred_sods = batch_base_inference(
        depth_model,
        sod_model,
        raw_images, 
        color=color
    )
    for image, pred_depth, pred_sod in zip(raw_images, pred_depths, pred_sods):
        result: np.ndarray = cv2.vconcat([image, pred_depth])
        result = cv2.hconcat([
            result, 
            cv2.resize(pred_sod, (pred_sod.shape[1] * 2, pred_sod.shape[0] * 2))
        ])
        result_vw.write(result)

def video_inference(
    depth_model: BaseDepthModel, 
    sod_model: BaseRGBDModel, 
    color: np.ndarray
):
    uploadedfile = st.file_uploader("Upload file", type=['mp4'])

    if uploadedfile is not None:
        with st.spinner():
            with open(video_tmp_path, 'wb') as f:
                f.write(uploadedfile.getbuffer())
        st.video(video_tmp_path)

        vf = cv2.VideoCapture(video_tmp_path)
        fps = vf.get(cv2.CAP_PROP_FPS)
        st.info(f'Video {fps:.2f} FPS')
        length = int(vf.get(cv2.CAP_PROP_FRAME_COUNT))

        start_frame = st.slider(
            'Start frame',
            min_value=0, max_value=length, value=0, step=1,
            key='start_frame'
        )

        end_frame = st.slider(
            'End frame',
            min_value=0, max_value=length, value=length-1, step=1,
            key='end_frame'
        )

        batch_size = st.number_input(
            'Batch size', min_value=1, max_value=100, 
            step=1, value=6, key='batch_size'
        )
        
        name = st.text_input('Name', uploadedfile.name.split('.mp4')[0])

        is_predict = st.button(
            'Predict Salient Objects', 
            key='predict_salient_objects', 
            disabled=uploadedfile is None,
        )
        if is_predict:
            result_stframe = st.empty()

            with st.spinner():
                frame_width = int(vf.get(3))
                frame_height = int(vf.get(4))
                
                result_tmp_file_path = 'tmp/tmp_result.avi'
                result_out_dir_path = os.path.join('demo', 'video', name)
                shutil.rmtree(result_out_dir_path, ignore_errors=True)
                os.makedirs(result_out_dir_path)
                result_out_file_path = os.path.join(result_out_dir_path, 'result.mp4')
                
                MAX_WIDTH = 640
                MAX_HEIGHT = 360
                
                width_ratio = frame_width / MAX_WIDTH
                height_ratio = frame_height / MAX_HEIGHT
                if width_ratio > height_ratio:
                    width = MAX_WIDTH
                    height = int(frame_height / width_ratio)
                else:
                    height = MAX_HEIGHT
                    width = int(frame_width / height_ratio)
                    
                result_vw = cv2.VideoWriter(
                    result_tmp_file_path, 
                    cv2.VideoWriter_fourcc('M', 'J', 'P', 'G'),
                    fps, (width * 3, height * 2)
                )
                length = int(vf.get(cv2.CAP_PROP_FRAME_COUNT))

                images: List[np.ndarray] = []

                start_time = time.time()

                for i in stqdm(range(length)):
                    if not vf.isOpened():
                        break

                    ret, image = vf.read()
                    if not ret:
                        break

                    if i < int(start_frame):
                        continue

                    if i > int(end_frame):
                        break

                    images.append(cv2.resize(image, (width, height)))

                    if len(images) >= batch_size:
                        video_batch_inference(
                            depth_model, sod_model,
                            images, result_vw, color
                        )
                        images = []

                if len(images) > 0:
                    video_batch_inference(
                        depth_model, sod_model,
                        images, result_vw, color
                    )
                    images = []

                st.info(f'Running time {time.time() - start_time:.2f} (s)')

                vf.release()
                result_vw.release()
                result_stframe.empty()

            with st.spinner('Converting videos to mp4...'):
                os.system(f'ffmpeg -y -i {result_tmp_file_path} -vcodec libx264 "{result_out_file_path}" -hide_banner -loglevel error')
                
            load_video(result_out_file_path)
