import os

import cv2
import matplotlib.pyplot as plt
import numpy as np
import torch
import torchvision.transforms as transforms
from torch import Tensor

from device import device
from lib.multi_depth_model_woauxi import RelDepthModel
from lib.net_tools import load_ckpt


def scale_torch(img: np.ndarray) -> Tensor:
    """
    Scale the image and output it in torch.tensor.
    :param img: input rgb is in shape [H, W, C], input depth/disp is in shape [H, W]
    :param scale: the scale factor. float
    :return: img. [C, H, W]
    """
    if len(img.shape) == 2:
        img = img[np.newaxis, :, :]
    if img.shape[2] == 3:
        transform = transforms.Compose([
            transforms.ToTensor(),
		    transforms.Normalize(
                (0.485, 0.456, 0.406), 
                (0.229, 0.224, 0.225) 
            )
        ])
        img = transform(img)
    else:
        img = img.astype(np.float32)
        img = torch.from_numpy(img)
    return img

# create depth model
depth_model = RelDepthModel(backbone='resnext101')
depth_model.eval()
load_ckpt(
    os.path.join(
        'pretrained_models', 
        'adelai_depth', 
        'res101.pth'
    ), 
    depth_model
)
depth_model.to(device)

def test():
    rgb = cv2.imread(os.path.join('images', 'pexels-mark-neal-5496430.jpg'))
    rgb_c = rgb[:, :, ::-1].copy()
    # gt_depth = None
    A_resize = cv2.resize(rgb_c, (448, 448))
    # rgb_half = cv2.resize(rgb, (rgb.shape[1]//2, rgb.shape[0]//2), interpolation=cv2.INTER_LINEAR)

    img_torch = scale_torch(A_resize)[None, :, :, :]
    pred_depth: np.ndarray = depth_model.inference(img_torch).cpu().numpy().squeeze()
    pred_depth_ori = cv2.resize(pred_depth, (rgb.shape[1], rgb.shape[0]))

    # if GT depth is available, uncomment the following part to recover the metric depth
    #pred_depth_metric = recover_metric_depth(pred_depth_ori, gt_depth)

    cv2.imwrite(os.path.join('tmp', 'rgb.png'), rgb)
    # save depth
    plt.imsave(os.path.join('tmp', 'depth.png'), pred_depth_ori, cmap='rainbow')
    cv2.imwrite(
        os.path.join('tmp', 'depth_raw.png'), 
        (pred_depth_ori/pred_depth_ori.max() * 60000).astype(np.uint16)
    )

if __name__ == '__main__':
    test()
