from typing import Tuple

import streamlit as st

from app_env import SOD_MODEL_TYPE
from app_utils import count_parameters
from base_model import BaseRGBDModel
from bbsnet_model import BBSNetModel
from rgbd_multimae_model import RGBDMultiMAEModel
from s_multimae.configs.base_config import base_cfg
from s_multimae.configs.experiment_config import configs_dict
from spnet_model import SPNetModel


@st.experimental_singleton
def load_multimae_model(sod_model_config_key: str) -> Tuple[BaseRGBDModel, base_cfg]:
    """
    1. Construct model
    2. Load pretrained weights
    3. Load model into device
    """
    cfg = configs_dict[sod_model_config_key]()
    sod_model = RGBDMultiMAEModel(cfg)
    return sod_model, cfg

@st.experimental_singleton
def load_spnet_model() -> BaseRGBDModel:
    """
    1. Construct model
    2. Load pretrained weights
    3. Load model into device
    """
    sod_model = SPNetModel()
    return sod_model

@st.experimental_singleton
def load_bbsnet_model() -> BaseRGBDModel:
    """
    1. Construct model
    2. Load pretrained weights
    3. Load model into device
    """
    sod_model = BBSNetModel()
    return sod_model

def sod_selection_ui() -> BaseRGBDModel:
    sod_model_type = st.selectbox(
        'Choose SOD model',
        (
            SOD_MODEL_TYPE.S_MULTIMAE,
            SOD_MODEL_TYPE.SPNET,
            SOD_MODEL_TYPE.BBSNET,
        ),
        key='sod_model_type',
    )

    if sod_model_type == SOD_MODEL_TYPE.S_MULTIMAE:
        sod_model_config_key = st.selectbox(
            'Choose config',
            configs_dict.keys(),
            key='sod_model_config_key',
        )
        sod_model, cfg = load_multimae_model(sod_model_config_key)
        st.text(f'Model description: {cfg.description}')
    elif sod_model_type == SOD_MODEL_TYPE.SPNET:
        sod_model = load_spnet_model()
        st.text(f'Model description: SPNet (https://github.com/taozh2017/SPNet)')
    elif sod_model_type == SOD_MODEL_TYPE.BBSNET:
        sod_model = load_bbsnet_model()
        st.text(f'Model description: BBSNet (https://github.com/DengPingFan/BBS-Net)')

    st.text(f'Number of parameters {count_parameters(sod_model)}')
    return sod_model
