import random
from typing import List, Tuple
from torch import nn, Tensor
import os, shutil
import torch
import matplotlib.pyplot as plt
import numpy as np
import gc, cv2

from .visualizer import post_processing_depth

'''
This module should not depend on other s_multimae modules.
'''

num_format = "{:,}".format

def clean_cache() -> None:
    torch.cuda.empty_cache()
    gc.collect()

def count_parameters(model: nn.Module) -> str:
    '''Count the number of learnable parameters of a model'''
    return num_format(sum(p.numel() for p in model.parameters() if p.requires_grad))

def random_choice(p: float) -> bool:
    '''Return True if random float <= p '''
    return random.random() <= p

def list_files(
    root_dir_path: str, max_files: int = None
) -> Tuple[List[str], List[str], List[str]]:
    '''List all files in a directory that has extensions
    Folder structure:
        root_dir_path
        |---GT
        |    |---image1.jpg
        |    |---image2.jpg
        |---RGB
        |    |---image1.png
        |    |---image2.png
        |---depths
        |    |---image1.png
        |    |---image2.png
    Returns: rgbs, depths, gts
    '''
    depths_dir = os.path.join(root_dir_path, 'depths')
    gts_dir = os.path.join(root_dir_path, 'GT')
    rgbs_dir = os.path.join(root_dir_path, 'RGB')

    depth_files = list(sorted(os.listdir(depths_dir)))
    gt_files = list(sorted(os.listdir(gts_dir)))
    rgb_files = list(sorted(os.listdir(rgbs_dir)))

    depth_files_names = [f.split('.')[0] for f in depth_files]
    gt_files_names = [f.split('.')[0] for f in gt_files]
    rgb_files_names = [f.split('.')[0] for f in rgb_files]

    # Ensure integrity
    assert depth_files_names == gt_files_names == rgb_files_names, \
        f"Dataset {root_dir_path} not integrity"

    depths: List[str] = []
    gts: List[str] = []
    rgbs: List[str] = []

    if max_files is not None:
        depth_files = depth_files[:max_files]
        gt_files = gt_files[:max_files]
        rgb_files = rgb_files[:max_files]

    for depth_file, gt_file, rgb_file in zip(depth_files, gt_files, rgb_files):
        depths.append(os.path.join(depths_dir, depth_file))
        gts.append(os.path.join(gts_dir, gt_file))
        rgbs.append(os.path.join(rgbs_dir, rgb_file))

    return rgbs, depths, gts

def scale_saliency_maps(inputs: Tensor) -> Tensor:
    '''Input: Tensor, shape of (B, C, H, W)'''
    min_v = torch.min(torch.flatten(inputs, 1), dim=1)[0].unsqueeze(1).unsqueeze(1).unsqueeze(1)
    max_v = torch.max(torch.flatten(inputs, 1), dim=1)[0].unsqueeze(1).unsqueeze(1).unsqueeze(1)
    return (inputs - min_v) / (max_v - min_v + 1e-8)

def get_epoch_from_ckpt_path(ckpt_path: str) -> int:
    '''Example ckpt_path
    os.path.join(experiment_dir_path, 'exp_v2.3', 'checkpoint_100.pt')
    '''
    return int(ckpt_path.split('_')[-1].split('.')[0])

def clean_dir(dir_path: str) -> None:
    '''Remove a directory if existed and create an empty directory'''
    if os.path.isdir(dir_path):
        shutil.rmtree(dir_path)
    os.makedirs(dir_path, exist_ok=True)

def get_sota_type(experiment_name: str) -> int:
    ''' 0 for SOTAs, 4 for experiment version 4, e.g. ...'''
    if "exp_v" not in experiment_name:
        return 0

    half_right = experiment_name.split("exp_v")[1]
    return int(half_right.split('.')[0])

def get_production_ckpt_path(experiment_name: str, epoch: int) -> str:
    return os.path.join('pretrained_models', 'multimae', experiment_name, f'checkpoint_{epoch}.pt')

def convert_batch_tensors_to_numpy_images(images: Tensor) -> np.ndarray:
    ''' images of shape (batch_size, channels, width, height) '''
    images = torch.permute(images, (0, 2, 3, 1))
    images = images.numpy()
    if images.shape[3] == 1:
        return np.squeeze(images, axis=3)
    else:
        return images

def join_horizontally(lst: List[np.ndarray]) -> np.ndarray:
    return np.concatenate(lst, axis=1)

def join_vertically(lst: List[np.ndarray]) -> np.ndarray:
    return np.concatenate(lst, axis=0)

def plot_batch_of_pairs(
    images: Tensor, 
    depths: Tensor, 
    gts: Tensor,
    save_file_path: str,
) -> None:
    images = convert_batch_tensors_to_numpy_images(images)
    depths = convert_batch_tensors_to_numpy_images(depths)
    gts = convert_batch_tensors_to_numpy_images(gts)
    batch_size = images.shape[0]
    samples: List[np.ndarray] = []
    
    # fig, axes = plt.subplots(batch_size, 3, figsize=(3*batch_size, 20)) # (number of images, 3)
    for i in range(batch_size):
        samples.append(join_horizontally([
            ((images[i]+1.0)/2 * 255).astype(np.uint8), 
            post_processing_depth(depths[i]), 
            post_processing_depth(gts[i]),
        ]))
        # axes[i, 0].imshow(images[i])
        # axes[i, 1].imshow(depths[i])
        # axes[i, 2].imshow(gts[i])
    # plt.show()
    
    final = join_vertically(samples)
    cv2.imwrite(save_file_path, cv2.cvtColor(final, cv2.COLOR_RGB2BGR))
    print(f'Saved to file {save_file_path}')
    
def plot_pairs(image: np.ndarray, depth: np.ndarray, gt: np.ndarray) -> None:
    batch_size = 1
    fig, axes = plt.subplots(batch_size, 3, figsize=(3*batch_size, 20)) # (number of images, 3)
    for i in range(batch_size):
        axes[i, 0].imshow(image)
        axes[i, 1].imshow(depth)
        axes[i, 2].imshow(gt)
    plt.show()

