from torch import nn, Tensor
import torch
from tqdm import tqdm
from logging import Logger

from s_multimae.dataloader import WarmupDataLoader

from ..criterion import Criterion
from ..distributed_training import get_device, get_rank, is_master_proc
from ..seeds import setup_seed
from ..gradient_accumulation import GradientAccumulation
from ..flop import calculate_gflops
from ..wandb_manager import wandb_init
from ..logger_fn import create_logger
from ..cross_validation import CrossValidation, cross_validation_log
from ..configs.base_config import base_cfg, get_train_cfg
from ..utils import count_parameters, clean_cache
from ..dataset_fn import TrainDataset
from ..lr import LinearLRRestart
from ..checkpoint import load_checkpoint, save_checkpoint
from ..device import cpu_device
from ..rgbd_model import RGBDModel
from ..run_type import run_type
from ..logger_fn import train_epoch_log

def train_v2(cfg: base_cfg) -> None:
    """TrainV2
    Changelogs:
    - Fix: zero out gradients step is performed in the wrong order
    - Using setup_seed with non-deterministic
    - Support distributed training
    - Change from model.opt_params to model.parameters()
    """
    
    # Distributed training
    local_rank = get_rank()
    print('local_rank', local_rank)
    local_device = get_device()
    is_master_process = is_master_proc()
    
    # Setup seed
    setup_seed(cfg.seed + local_rank, True)
    
    model = RGBDModel(cfg, run_type=run_type.rt)

    if is_master_process:
        # Logger
        logger = create_logger(cfg.logs_dir, 'TRAIN')

        # Model
        no_params = count_parameters(model)
        gflops = None # calculate_gflops(model, cfg.image_size)
        logger.info(f'Number of params {no_params}')
        logger.info(f'{gflops} GLOP(s)')
    else:
        logger: Logger = None

    # Train dataset and Dataloader
    train_dataset = TrainDataset(cfg)
    warmup_dataloader = WarmupDataLoader(cfg, train_dataset)
    
    # Cross validation
    cross_val = CrossValidation(
        cfg, max_track=20, 
        data_augmentation_version=cfg.data_augmentation_version
    )

    # Learning rate policy
    cfg.num_train_imgs = len(train_dataset)
    cfg.niters_per_epoch = len(warmup_dataloader.train_dataloader)
    lr_policy = LinearLRRestart(
        cfg.lr, cfg.end_lr, cfg.warmup_epoch
    )
    
    # Gradient Accumulation
    gradient_accumulation = GradientAccumulation(
        cfg.accum_iter, cfg.niters_per_epoch
    )
    
    # Criterion
    criterion = Criterion(cfg)
    
    # Gradient Scaler
    scaler = torch.cuda.amp.GradScaler()
    
    # Optimizer
    optimizer = torch.optim.AdamW(
        model.opt_params, lr=cfg.lr, betas=cfg.betas, 
        weight_decay=cfg.weight_decay
    )
    
    model.to(local_device)
    model = nn.SyncBatchNorm.convert_sync_batchnorm(model)
    
    # ✨
    # This container parallelizes the application of the given module by
    # splitting the input across the specified devices by chunking in the batch
    # dimension. The module is replicated on each machine and each device, and
    # each such replica handles a portion of the input. During the backwards
    # pass, gradients from each node are averaged.
    # The batch size should be larger than the number of GPUs used locally.
    model = nn.parallel.DistributedDataParallel(
        model, 
        
        # For single-device modules, ``device_ids`` can
        # contain exactly one device id
        device_ids=[local_rank],
    )
    
    # Load checkpoint if checkpoint path exists
    start_epoch, global_step = load_checkpoint(
        model, optimizer, None, logger, cfg.ckpt_path,
        strict=False, scaler=scaler
    )

    if is_master_process:
        train_cfg = get_train_cfg(
            cfg, no_params=no_params, gflops=gflops
        )
        wandb_run = wandb_init(cfg.experiment_name, config=train_cfg)
        logger.info('Start training:')

    if (start_epoch-1) % cfg.save_checkpoints_after_each_n_epochs == 0 and start_epoch != 1 \
        and cfg.save_checkpoints_after_each_n_epochs != 1:
        cross_validation_log(cfg, model, logger, cross_val, start_epoch-1)
    # cross_validation_log(cfg, model, logger, cross_val, start_epoch-1)

    for epoch in range(start_epoch, cfg.nepochs+1):
        # Don't forget to set epoch!
        warmup_dataloader.set_epoch(epoch)
        cfg.niters_per_epoch = len(warmup_dataloader.train_dataloader)
        lr_policy.set_epoch(epoch, cfg.niters_per_epoch)
        gradient_accumulation.niters_per_epoch = cfg.niters_per_epoch

        clean_cache()
        model.train()

        sum_loss: float = 0.0
        
        # ⭕ Zero the parameter gradients
        optimizer.zero_grad()

        # batch index starting from 1
        for batch_idx, (images, depths, gts, indices) in tqdm(
            enumerate(warmup_dataloader.train_dataloader, start=1), 
            total=cfg.niters_per_epoch,
            disable=not is_master_process,
        ):
            global_step += 1
            
            # Update learning rate
            current_idx = (epoch - 1) * cfg.niters_per_epoch + (batch_idx - 1) 
            lr = lr_policy.get_lr(current_idx)
            for i in range(len(optimizer.param_groups)):
                optimizer.param_groups[i]['lr'] = lr * optimizer.param_groups[i]['lr_scale']
            
            # Move to GPU
            images: Tensor = images.to(local_device)
            depths: Tensor = depths.to(local_device)
            gts: Tensor = gts.to(local_device)

            # Calculate loss
            with torch.cuda.amp.autocast(enabled=cfg.is_fp16):
                gpu_out: Tensor = model(images, depths)
                loss = criterion(gpu_out['semseg'], gts)
                if 'depth' in cfg.outputs:
                    loss += criterion(gpu_out['depth'], depths)

            clone_loss: Tensor = loss

            loss = loss / cfg.accum_iter
            
            # 📈 Scales loss.  
            # Calls backward() on scaled loss to create scaled gradients 
            # (accumulating scaled gradients if gradients are not zero)
            scaler.scale(loss).backward()
                
            # BackProp
            if gradient_accumulation.is_back_prop(batch_idx):
                # ⭐ Unscale gradients
                # - scaler.step() first unscales gradients of the optimizer's params.
                # - If gradients don't contain infs/NaNs, optimizer.step() is then called,
                # otherwise, optimizer.step() is skipped.
                # - optimizer.step(): updates the parameters
                # 
                # - Each parameter’s gradient (.grad attribute) should be unscaled 
                # before the optimizer updates the parameters, so the scale factor 
                # does not interfere with the learning rate.
                scaler.step(optimizer)
                
                # ⭐ Updates the scale for next iteration.
                scaler.update()
                
                # ⭕ Zero the parameter gradients
                optimizer.zero_grad()

            loss: float = clone_loss.to(cpu_device).item()
            sum_loss += loss

            if batch_idx % 10 == 0 and is_master_process:
                train_epoch_log(
                    cfg, epoch, logger, batch_idx, 
                    lr, loss, sum_loss, warmup_dataloader
                )

            del loss, images, gts, depths
            clean_cache()

        # Save checkpoint after each epoch
        if is_master_process:
            save_checkpoint(
                cfg, epoch, global_step, 
                model, optimizer, lr_policy, scaler,
            )

        if epoch % cfg.save_checkpoints_after_each_n_epochs == 0:
            cross_validation_log(cfg, model, logger, cross_val, epoch)
    
    if is_master_process:    
        wandb_run.finish()
