from torch import nn, Tensor
import wandb
import torch.backends.cudnn as cudnn
import random
import torch
from torch.utils.data import DataLoader
from tqdm import tqdm

from ..run_type import run_type
from ..rgbd_model import RGBDModel
from ..wandb_manager import wandb_init
from ..logger_fn import create_logger
from ..cross_validation import CrossValidation, cross_validation_log
from ..configs.base_config import base_cfg, get_train_cfg
from ..utils import count_parameters, clean_cache
from ..dataset_fn import TrainDataset
from ..lr import LinearLRRestart
from ..checkpoint import load_checkpoint, save_checkpoint
from ..device import device, cpu_device

def train_v1(cfg: base_cfg) -> None:
    cudnn.benchmark = True

    # Setup seed
    seed = random.randrange(100)
    torch.manual_seed(seed)
    if torch.cuda.is_available():
        torch.cuda.manual_seed(seed)

    # Logger
    logger = create_logger(cfg.logs_dir, 'TRAIN')
    

    # Model
    model = RGBDModel(cfg, run_type=run_type.rt)
    no_params = count_parameters(model)
    gflops = None # calculate_gflops(model, cfg.image_size)
    logger.info(f'Number of params {no_params}')
    logger.info(f'{gflops} GLOP(s)')
    
    # Cross Validation
    cross_val = CrossValidation(
        cfg, max_track=20, 
        data_augmentation_version=cfg.data_augmentation_version
    )
    
    # Group weight and config optimizer
    base_lr = cfg.lr
    
    optimizer = torch.optim.AdamW(
        model.opt_params, lr=base_lr, betas=cfg.betas, 
        weight_decay=cfg.weight_decay
    )

    # Train dataset and Dataloader
    train_dataset = TrainDataset(
        cfg.train_dataset_working_dir_path, cfg.image_size,
        version=cfg.data_augmentation_version, is_padding=cfg.is_padding,
        inputs=cfg.inputs, outputs=cfg.outputs,
    )
    train_dataloader = DataLoader(
        train_dataset, batch_size=cfg.batch_size, shuffle=True, 
        num_workers=cfg.num_workers
    )

    # config lr policy
    cfg.num_train_imgs = len(train_dataset)
    cfg.niters_per_epoch = cfg.num_train_imgs // cfg.batch_size  + 1
    total_iteration = cfg.nepochs * cfg.niters_per_epoch
    
    lr_policy = LinearLRRestart(
        base_lr, cfg.end_lr, cfg.niters_per_epoch,
    )
    
    model.to(device)
    optimizer.zero_grad()
    scaler = torch.cuda.amp.GradScaler()

    # Load checkpoint if checkpoint path exists
    start_epoch, global_step = load_checkpoint(
        model, optimizer, None, logger, cfg.ckpt_path,
        strict=False, scaler=scaler
    )

    train_cfg = get_train_cfg(
        cfg, no_params=no_params, gflops=gflops
    )
    wandb_init(cfg.experiment_name, config=train_cfg)
    logger.info('Start training:')

    if (start_epoch-1) % cfg.save_checkpoints_after_each_n_epochs == 0 and start_epoch != 1:
        cross_validation_log(cfg, model, logger, cross_val, start_epoch-1)

    for epoch in range(start_epoch, cfg.nepochs+1):
        # Don't forget to set epoch!
        lr_policy.set_epoch(epoch)

        clean_cache()
        model.train()

        sum_loss: float = 0.0


        for idx, (images, depths, gts, indices) in tqdm(
            enumerate(train_dataloader, start=1), 
            total=cfg.niters_per_epoch
        ):
            global_step += 1
            optimizer.zero_grad()

            # Move to GPU
            images: Tensor = images.to(device)
            depths: Tensor = depths.to(device)
            gts: Tensor = gts.to(device)

            # Calculate loss
            with torch.cuda.amp.autocast(enabled=cfg.is_fp16):
                loss = model(images, depths, gts)

            clone_loss = loss

            loss = loss / cfg.accum_iter
            # ⭐️ ⭐️ Scale Gradients
            scaler.scale(loss).backward()
                
            # BackProp
            if (idx % cfg.accum_iter == 0) or (idx == len(train_dataloader)):
                # ⭐️ ⭐️ Update Optimizer
                scaler.step(optimizer)
                scaler.update()

            current_idx = (epoch - 1) * cfg.niters_per_epoch + idx 
            lr = lr_policy.get_lr(current_idx)

            # print('global_step', global_step, 'current_idx', 'current_idx')

            for i in range(len(optimizer.param_groups)):
                optimizer.param_groups[i]['lr'] = lr * optimizer.param_groups[i]['lr_scale']

            loss: float = clone_loss.to(cpu_device).item()
            sum_loss += loss

            if idx % 10 == 0:
                print_str = 'Epoch {}/{}'.format(epoch, cfg.nepochs) \
                        + ' Iter {}/{}:'.format(idx + 1, cfg.niters_per_epoch) \
                        + ' lr=%.4e' % (lr * cfg.lr_scale) \
                        + ' loss=%.4f total_loss=%.4f' % (loss, (sum_loss / idx))
                logger.info(print_str)

                wandb.log({
                    'loss': loss,
                    'avg_loss': sum_loss / idx,
                    'lr': (lr * cfg.lr_scale),
                    # 'inputs': wandb.Image(images),
                    # "loss_hist": wandb.Histogram(ouputs),
                })

            del loss, images, gts, depths
            clean_cache()

        # Save checkpoint after each epoch
        save_checkpoint(
            cfg, epoch, global_step, 
            model, optimizer, lr_policy, scaler,
        )

        if epoch % cfg.save_checkpoints_after_each_n_epochs == 0:
            cross_validation_log(cfg, model, logger, cross_val, epoch)
        

    wandb.finish()
