from typing import Dict, Optional
from torch import nn, Tensor

from .model.multimae import generate_smultimae_model
from .configs.base_config import base_cfg
from .run_type import RUN_TYPE

class RGBDModel(nn.Module):
    def __init__(self, cfg: base_cfg, run_type=RUN_TYPE.COLAB):
        super(RGBDModel, self).__init__()

        self.inputs = cfg.inputs
        self.outputs = cfg.outputs

        self.is_no_depth = cfg.is_inference_with_no_depth

        self.model, self.opt_params = generate_smultimae_model(cfg, run_type=run_type)

    def encode_decode(
        self, images: Tensor, depths: Tensor
    ) -> Dict[str, Tensor]:
        """Encode images with backbone and decode into a semantic segmentation
        map of the same size as input.
        
        Returns:
        {
            "semseg": Tensor,
            "depth": Optional[Tensor],
            "rgb": Optional[tensor],
        }
        """
        inputs = {
            'rgb': images
        }
        if 'depth' in self.inputs:
            inputs['depth'] = depths
        return self.model.forward(inputs)

    def forward(self, images: Tensor, depths: Tensor) -> Dict[str, Tensor]:
        return self.encode_decode(images, depths)

    def inference(
        self, images: Tensor, depths: Optional[Tensor],
    ) -> Tensor:
        return self.encode_decode(images, depths)['semseg']

def test_model():
    from .run_type import run_type
    import torch
    from .utils import count_parameters
    from .configs.experiment_config import cfgv4_0_18
    cfg = cfgv4_0_18()
    model = RGBDModel(run_type=run_type)
    image_size = cfg.image_size # cfg.image_size
    images = torch.randn((3, 3, image_size, image_size))
    depths = torch.randn((3, 1, image_size, image_size))
    gts = torch.ones((3, 1, image_size, image_size))
    outs = model(images, depths, gts)
    print(outs)
    print(count_parameters(model))

# test_model()
