import os
from pprint import pprint
from typing import List, Tuple
import pandas as pd
import numpy as np

from .ablation_study import ABLATION_STUDY
from .eval import get_existed_tracking_tables_from_csv, save_tracking_csv
from .configs.base_config import base_cfg

def generate_benchmark_report(cfg: base_cfg) -> None:
    version = cfg.datasets_set
    additional_name = 'benchmark'
    our_model_name = 'Ours'
    
    print(f'generate_benchmark_report version={version}')
    csv_file_path = os.path.join(
        cfg.benchmark_csv_dir_path, 
        f'sotas_v{version}.csv'
    ) # GoogleDrive
    tracking_results_lst = get_existed_tracking_tables_from_csv(csv_file_path)
    # pprint(tracking_results_lst[:10], width=500)

    if version == 1:
        # Due to insufficient test saliency maps to guarantee the results are accurate
        filtered_out_sota_model_names = [
            'EFNet', 'BTSNet'
        ] 
        
        # Only keep base model
        # experiment_name = 'exp_v4.0.19_epoch175'
        experiment_name = 'exp_v4.0.35_epoch136'
    elif version == 2:
        # Due to insufficient test saliency maps to guarantee the results are accurate
        filtered_out_sota_model_names = [
            'EFNet', 'BTSNet'
        ] 
        
        # Only keep base model
        experiment_name = 'exp_v4.0.25_epoch195'
    elif version == 4:
        # Due to insufficient test saliency maps to guarantee the results are accurate
        filtered_out_sota_model_names = []
        
        # Only keep base model
        experiment_name = 'exp_v4.0.37_epoch259'
    else:
        raise NotImplementedError()
    
    final_results = []
    for tracking_results in tracking_results_lst:
        sota_name = tracking_results[0]
        sota_type = tracking_results[-5]
        if sota_type == 0 and sota_name not in filtered_out_sota_model_names:
            final_results.append(tracking_results)
        elif sota_name == experiment_name:
            tracking_results[0] = our_model_name
            final_results.append(tracking_results)
            
    # save results
    save_tracking_csv(
        cfg.test_dataset_names, 
        final_results, 
        csv_file_path=os.path.join(
            cfg.benchmark_csv_dir_path, 
            f'sotas_v{version}_{additional_name}.csv'
        ), # GoogleDrive, 
        version=version,
        additional_name=additional_name
    )

def generate_ablation_report(
    cfg: base_cfg,
    additional_name: str,
    mapping_experiment_names: List[List[str]] = []
) -> None:
    """
    
    Examples:
    additional_name = 'ablation_inputs_outputs'
        mapping_experiment_names = [
            ['exp_v4.0.19_epoch175', 'RGB+Depth->SM'],
            ['exp_v4.0.21_epoch160', 'RGB->SM+Depth'],
            ['exp_v4.0.22_epoch185', 'RGB->SM'],
            ['exp_v4.0.23_epoch195', 'Depth->SM'],
        ]
    """
    version = cfg.datasets_set
    
    print(f'generate_ablation_report version={version}')
    csv_file_path = os.path.join(
        cfg.benchmark_csv_dir_path, 
        f'sotas_v{version}.csv'
    ) # GoogleDrive
    tracking_results_lst = get_existed_tracking_tables_from_csv(csv_file_path)
    pprint(tracking_results_lst[:10], width=500)

    final_results = []
    for tracking_results in tracking_results_lst:
        sota_name = tracking_results[0]
        for mapping_experiment_name in mapping_experiment_names:
            if sota_name == mapping_experiment_name[0]:
                tracking_results[0] = mapping_experiment_name[1]
                final_results.append(tracking_results)
            
    # save results
    save_tracking_csv(
        cfg.test_dataset_names, 
        final_results, 
        csv_file_path=os.path.join(
            cfg.benchmark_csv_dir_path, 
            f'sotas_v{version}_{additional_name}.csv'
        ), # GoogleDrive, 
        version=version,
        additional_name=additional_name
    )

def generate_ablation_report_about_inputs_outputs(cfg: base_cfg) -> None:
    additional_name = ABLATION_STUDY.INPUTS_OUTPUTS
    if cfg.datasets_set == 1:
        # Only keep base model
        generate_ablation_report(
            cfg,
            additional_name,
            mapping_experiment_names = [
                ['exp_v4.0.19_epoch175', 'RGB+Depth->SM'],
                ['exp_v4.0.21_epoch160', 'RGB->SM+Depth'],
                ['exp_v4.0.22_epoch185', 'RGB->SM'],
                ['exp_v4.0.23_epoch195', 'Depth->SM'],
            ],
        )
    else:
        print(f'NotImplementedError! Ignored {additional_name} on set {cfg.datasets_set}')

def generate_ablation_report_about_data_augmentation(cfg: base_cfg) -> None:
    additional_name = ABLATION_STUDY.DATA_AUGMENTATION
    if cfg.datasets_set == 1:
        generate_ablation_report(
            cfg, 
            additional_name,
            mapping_experiment_names = [
                ['exp_v4.0.18_epoch180', 'DataAugmentationV4'],
                ['exp_v4.0.19_epoch175', 'DataAugmentationV2'],
            ],
        )
    else:
        print(f'NotImplementedError! Ignored {additional_name} on set {cfg.datasets_set}')

def generate_ablation_report_about_pretrained_backbone(cfg: base_cfg) -> None:
    additional_name = ABLATION_STUDY.PRETRAINED_BACKBONE
    if cfg.datasets_set == 1:
        generate_ablation_report(
            cfg, 
            additional_name,
            mapping_experiment_names = [
                ['exp_v4.0.43_epoch71', 'Self-supervised MultiMAE'],
                ['exp_v4.0.35_epoch136', 'Self-supervised MAE'],
                ['exp_v4.0.47_epoch68', 'Supervised ViT'],
                ['exp_v4.0.31_epoch174', 'No pretrained'],
            ],
        )
    else:
        print(f'NotImplementedError! Ignored {additional_name} on set {cfg.datasets_set}')

def generate_ablation_report_about_gpu_type(cfg: base_cfg) -> None:
    additional_name = ABLATION_STUDY.GPU_TYPE
    if cfg.datasets_set == 1:
        generate_ablation_report(
            cfg, 
            additional_name,
            mapping_experiment_names = [
                ['exp_v4.0.44_epoch86', 'RTX2070'],
                ['exp_v4.0.36_epoch70', 'P100'],
                ['exp_v4.0.43_epoch71', 'T4x2'],
                ['exp_v4.0.32_epoch241', 'A100'],
            ],
        )
    else:
        print(f'NotImplementedError! Ignored {additional_name} on set {cfg.datasets_set}')

def format_number(num: float) -> str:
    if num < 0:
        return '-'
    return "{:.4f}".format(num).lstrip('0')

def __generate_str(
    df: pd.DataFrame,
    dataset_name: str,
    evaluation_metric_name: str, # MAE, S, MaxF, MaxE
) -> str:
    if evaluation_metric_name not in ['MAE', 'S', 'MaxF', 'MaxE']:
        raise Exception(f'Unsupported evaluation_metric_name {evaluation_metric_name}')
    
    rs = ''
    lst = df[f'{dataset_name}_{evaluation_metric_name}'].to_numpy() if len(dataset_name) > 0 \
        else df[evaluation_metric_name].to_numpy()
    best_index = np.argmax(lst) if evaluation_metric_name in ['S', 'MaxF', 'MaxE'] \
        else np.argmin(lst)
    for i, e in enumerate(lst.tolist()):
        if i % 4 == 0:
            rs += '\n'
        rs += ' & ' + format_number(e) if i != best_index \
            else ' & ' + f'\\textcolor{{blue}}{{\\textbf{{{format_number(e)}}}}}'
    return rs

def sota_name_with_citation_latex(mapping_name: Tuple[str, str]) -> str:
    name, cite = mapping_name
    if '->' in name:
        name = name.replace('->', ' $\\rightarrow$ ')
    return f'\\textbf{{{name}}} \\cite{{{cite}}}' if cite is not None \
        else f'\\textbf{{{name}}}'

def dataset_name_with_citation_latex(mapping_name: Tuple[str, str]) -> str:
    name, cite = mapping_name
    return f'\\textbf{{{name}}} \\\\ \\cite{{{cite}}}' if cite is not None \
        else f'\\textbf{{{name}}} \\\\'

def get_mapping_sota_names(cfg: base_cfg, sota_names: List[str]) -> List[str]:
    mapping_sota_names: List[str] = []
    for sota_name in sota_names:
        try:
            index = cfg.sota_model_names.index(sota_name)
            mapping_sota_names.append(sota_name_with_citation_latex(cfg.mapping_sota_model_names[index]))
        except:
            mapping_sota_names.append(sota_name_with_citation_latex((sota_name, None))) # Ours
    return mapping_sota_names

def quantitative_evaluation_latex(cfg: base_cfg) -> None:
    df = pd.read_csv(os.path.join(
        cfg.benchmark_csv_dir_path,
        f'sotas_v{cfg.datasets_set}_benchmark.csv'
    ))
    mapping_sota_names = get_mapping_sota_names(cfg, df['Model'].tolist())
    test_dataset_names = cfg.test_dataset_names + ['']
    mapping_test_dataset_names = [dataset_name_with_citation_latex(e) for e in cfg.mapping_test_dataset_names] + ['']
    test_dataset_quantities = cfg.test_dataset_quantities + [sum(cfg.test_dataset_quantities)]
    
    export_latex(
        df, mapping_sota_names, mapping_test_dataset_names,
        test_dataset_names, test_dataset_quantities,  
        f'{cfg.quantitative_evaluation_latex_dir_path}.txt' 
    )

def quantitative_evaluation_ablation_study_latex(
    cfg: base_cfg, ablation_study_name: str
) -> None:
    df = pd.read_csv(os.path.join(
        cfg.benchmark_csv_dir_path,
        f'sotas_v{cfg.datasets_set}_{ablation_study_name}.csv'
    ))
    mapping_sota_names = get_mapping_sota_names(cfg, df['Model'].tolist())
    test_dataset_names = ['']
    mapping_test_dataset_names = ['']
    test_dataset_quantities = [sum(cfg.test_dataset_quantities)]
    
    export_latex(
        df, mapping_sota_names, mapping_test_dataset_names,
        test_dataset_names, test_dataset_quantities,
        f'{cfg.quantitative_evaluation_latex_dir_path}_{ablation_study_name}.txt'
    )
    
def export_latex(
    df: pd.DataFrame,
    mapping_sota_names: List[str],
    mapping_test_dataset_names: List[str],
    test_dataset_names: List[str],
    test_dataset_quantities: List[str],
    txt_path: str,
) -> None:
    print('mapping_sota_names', mapping_sota_names)
    print('test_dataset_names', test_dataset_names)
    
    first_line = ' & '.join(mapping_sota_names)  # sota_names

    latex_str = f'''
    \\begin{{tabularx}}{{\\textwidth}}{{
        |c|c| {'Y|' * len(mapping_sota_names)}
    }}
    \hline

    \multicolumn{{2}}{{|c|}}{{\\textbf{{Models}}}}
    & {first_line} \\\\
    \hline
    '''

    for mapping_dataset_name, dataset_name, quantity in \
        zip(mapping_test_dataset_names, test_dataset_names, test_dataset_quantities):
        mae_str = __generate_str(df, dataset_name, 'MAE')
        s_str = __generate_str(df, dataset_name, 'S')
        e_str = __generate_str(df, dataset_name, 'MaxF')
        f_str = __generate_str(df, dataset_name, 'MaxE')
        
        mapping_dataset_name = mapping_dataset_name \
            if len(mapping_dataset_name) > 0 \
            else dataset_name_with_citation_latex(('Average', None))
        
        latex_str += f'''
        \multirow{{4}}{{*}}{{\\rotatebox[origin=c]{{90}}{{ \makecell{{ {mapping_dataset_name} {quantity}}} }}}} 
        & M $\\downarrow$
        {mae_str} \\\\

        & S $\\uparrow$
        {s_str} \\\\

        & F $\\uparrow$
        {e_str} \\\\

        & E $\\uparrow$
        {f_str} \\\\
        \hline
        '''

    latex_str += f'''
    \end{{tabularx}}
    '''

    with open(txt_path, 'w') as f:
        f.write(latex_str)
    print(f'Saved to file {txt_path}')
