import shutil
import glob
import copy
from pprint import pprint
from typing import Dict, List, Tuple
from PIL import Image
import os
import numpy as np
import cv2
from py_sod_metrics import Smeasure
import pandas as pd
import wandb
from tqdm import tqdm

from .visualizer import apply_threshold, apply_vis_to_image, post_processing_depth
from .wandb_manager import wandb_init_sota_benchmark_sm, wandb_delete_artifacts
from .configs.base_config import base_cfg
from .dataset_fn import TestDataset
from .prepare_datasets import unzip_SOTAs, unzip_datasets

def __select_samples(
    df: pd.DataFrame,
    sota_names: List[str],
    our_model_name: str,
    no_samples: int,
    is_best: bool,
) -> Dict[str, List[str]]:
    best_rs = pd.DataFrame()
    best_rs[['image_name', 'dataset_name']] = df[['image_name', 'dataset_name']]
    for sota_name in sota_names:
        best_rs[sota_name] = df[our_model_name] - df[sota_name] if is_best \
            else df[sota_name] - df[our_model_name]
        
    best_rs['min'] = best_rs.min(axis=1, numeric_only=True)
    best_rs = best_rs[best_rs['min'] > 0]
    best_rs = best_rs.sort_values('min', ascending=False)
    best_rs = best_rs[['image_name', 'dataset_name', 'min']]
    print(best_rs.head(10))
    
    selected_samples: Dict[str, List[str]] = dict()
    for v in best_rs.iloc[:no_samples].values:
        dataset_name = v[1]
        image_name = v[0]
        if dataset_name not in selected_samples:
            selected_samples[dataset_name] = [image_name]
        else:
            selected_samples[dataset_name].append(image_name)
    
    return selected_samples 

def qualitative_evaluation(
    cfg: base_cfg,
    no_samples: int,
    dataset_names: List[str],
    sota_model_names: List[str],
) -> Dict[str, List[str]]:
    unzip_SOTAs(
        cfg, 
        [dataset_names for _ in range(len(sota_model_names))],
        sota_model_names
    )
    clean_qualitative_evaluation_latex(cfg)
    
    df = pd.read_csv(os.path.join(
        cfg.benchmark_csv_dir_path, f'smeasure_set{cfg.datasets_set}.csv'
    ))
    
    columns: List[str] = df.columns.tolist()
    our_model_names = [column for column in columns \
        if column.startswith('exp_v')]
    
    assert len(our_model_names) == 1, 'Should contain only one of our model'
    
    our_model_name = our_model_names[0]
    print(f'Our model name {our_model_name}')
    
    sota_names = [column for column in columns \
        if column not in ['index', 'image_name', 'dataset_name'] \
            and not column.startswith('exp_v')
    ]
    print('SOTA names:', sota_names)
    
    best_samples = __select_samples(
        df, sota_names, our_model_name, no_samples, is_best=True
    )
    print('Best samples:')
    pprint(best_samples)
    best_table, best_vis_table, best_s_measure_table = __qualitative_evaluation(
        cfg, best_samples, sota_model_names, is_best=True
    )
    
    worst_samples = __select_samples(
        df, sota_names, our_model_name, no_samples, is_best=False
    )
    print('Worst samples:')
    pprint(worst_samples)
    worst_table, worst_vis_table, worst_s_measure_table = __qualitative_evaluation(
        cfg, worst_samples, sota_model_names, is_best=False
    )
    
    wandb_run = wandb_init_sota_benchmark_sm(cfg.datasets_set)
    wandb_delete_artifacts(cfg, wandb_run)
    wandb_run.log({
        f'benchmark_best_sm': best_table,
        f'benchmark_best_vis': best_vis_table,
        f'benchmark_best_s_measure': best_s_measure_table,
        f'benchmark_worst_sm': worst_table,
        f'benchmark_worst_vis': worst_vis_table,
        f'benchmark_worst_s_measure': worst_s_measure_table,
    })
    wandb_run.finish()

def __format_number(num: float) -> str:
    return "{:.4f}".format(num).lstrip('0')

def __qualitative_evaluation_latext(cfg: base_cfg, is_best: bool, scale=0.1):
    df = pd.read_csv(__qualitative_evaluation_csv_path(cfg, is_best=is_best))
    text = f"""
\\begin{{tabularx}}{{\\textwidth}}{{
    c {"Y " * df.shape[0]}
}} 
"""
    no_samples = df.shape[0]
    range_indices = range(0, no_samples)
    for i in range_indices:
        text += f" & Sample {i+1}"
    text += "\\\\ \\midrule \n"

    text += "Color \n"
    for i in range_indices:
        text += f" & \includegraphics[scale={scale}]{{Images/QualitativeEvaluationSet{cfg.datasets_set}/{'best' if is_best else 'worst'}_{df['Index'][i]}_RGB.png}}"
    text += f"\\\\ \\midrule \n"
        
    text += "Depth \n"
    for i in range_indices:
        text += f" & \includegraphics[scale={scale}]{{Images/QualitativeEvaluationSet{cfg.datasets_set}/{'best' if is_best else 'worst'}_{df['Index'][i]}_DEPTH.png}}"
    text += f"\\\\ \\midrule \n"
        
    text += "GT \n"
    for i in range_indices:
        text += f" & \includegraphics[scale={scale}]{{Images/QualitativeEvaluationSet{cfg.datasets_set}/{'best' if is_best else 'worst'}_{df['Index'][i]}_GT.png}}"
    text += f"\\\\ \\midrule \n"
        
    columns = df.columns
    sota_names = columns[6:]
    
    for sota_name in sota_names:
        text += f"{sota_name} \n"
        for i in range_indices:
            text += f" & \includegraphics[scale={scale}]{{Images/QualitativeEvaluationSet{cfg.datasets_set}/{'best' if is_best else 'worst'}_{df['Index'][i]}_{sota_name}.png}} {__format_number(df[sota_name][i])}" 
        text += f"\\\\ \\midrule \n"
        
    text += f"\n \\end{{tabularx}}"
    
    txt_path = os.path.join(
        cfg.source_code_dir, 'latex', 
        f'{"best" if is_best else "worst"}_qualitative_evaluation_set{cfg.datasets_set}.txt'
    )
    with open(txt_path, 'w') as f:
        f.write(text)

def qualitative_evaluation_latex(cfg: base_cfg) -> None:
    __qualitative_evaluation_latext(cfg, is_best=True)
    __qualitative_evaluation_latext(cfg, is_best=False)

def clean_qualitative_evaluation_latex(cfg: base_cfg) -> None:
    """Clean: remove directory cfg.qualitative_evaluation_latex

    Args:
        cfg (base_cfg): Config
    """
    shutil.rmtree(cfg.qualitative_evaluation_latex_dir_path, ignore_errors=True)

def save_qualitative_evaluation_latex(
    cfg: base_cfg,
    i: int,
    img: np.ndarray,
    label: str,
    is_best: bool,
) -> None:
    os.makedirs(cfg.qualitative_evaluation_latex_dir_path, exist_ok=True)
    cv2.imwrite(
        os.path.join(
            cfg.qualitative_evaluation_latex_dir_path, 
            f'{"best" if is_best else "worst"}_{i}_{label}.png'
        ),
        cv2.cvtColor(cv2.resize(img, (150, 150)), cv2.COLOR_BGR2RGB)
    )

def __qualitative_evaluation_csv_path(cfg: base_cfg, is_best: bool) -> str:
    """Return the csv path to the best/worst qualitative evaluation of our experiment

    Args:
        cfg (base_cfg): Config
        is_best (bool): best/worst

    Returns:
        str: csv path
    """
    return os.path.join(
        cfg.benchmark_csv_dir_path, 
        f'qualitative_evaluation_{"best" if is_best else "worst"}_set{cfg.datasets_set}.csv'
    )
    
def __mapping_sota_model_names(sota_model_names: List[str]) -> List[str]:
    """Map our experiment name "exp_v4.0.19_epoch175" to "Ours"

    Args:
        sota_model_names (List[str]): List of SOTA names, must include only one "exp_v*"

    Returns:
        List[str]: List of SOTA names after mapping our experiment name to "Ours"
    """
    our_experiments = [sota_model_name for sota_model_name in sota_model_names if sota_model_name.startswith("exp_v")]
    assert len(our_experiments) == 1, "must include only one 'exp_v*'"
    return [
        "Ours" if sota_model_name.startswith("exp_v") else sota_model_name for sota_model_name in sota_model_names
    ]

def __qualitative_evaluation(
    cfg: base_cfg,
    selected_samples: Dict[str, List[str]],
    sota_model_names: List[str],
    is_best: bool,
) -> Tuple[wandb.Table, wandb.Table, wandb.Table]:
    """
    Note: Make sure unzip_SOTAs before execute this function
    
    selected_samples = {
        'COME-E': [
            'COME_Hard_2050',
            'COME_Hard_1961',
            ...
        ],
        ....  
    }
    
    dataset_names can be ['COME-E', 'COME-H']
    """
    mapped_sota_model_names = __mapping_sota_model_names(sota_model_names)
    columns = ['Index', 'Image name', 'Dataset', 'RGB', 'Depth', 'GT'] + mapped_sota_model_names
    results = []
    vis_results = []
    s_measure_results = []
    color = np.array([0., 0.97647059, 0.])
    
    for dataset_name, image_names in selected_samples.items():
        print(f'Dataset {dataset_name}')
        dataset_dir_path = os.path.join(
            cfg.test_datasets_working_dir_path, 
            dataset_name
        )
        dataset = TestDataset(cfg, dataset_dir_path)
        
        for image_name in tqdm(image_names):
            possible_image_names = [
                existed_image_name for existed_image_name in dataset.images \
                if os.path.basename(existed_image_name).startswith(f'{image_name}.')
            ]
            assert len(possible_image_names) == 1, \
                f'Should have exactly one image {image_name} - No. len {len(possible_image_names)}'
            
            i = dataset.images.index(possible_image_names[0])
        
            image, depth, gt, image_name = dataset.get_raw_item(i)
            postprocessed_depth = post_processing_depth(np.asarray(depth))
            
            row = [
                i, image_name, dataset_name, 
                wandb.Image(image), 
                wandb.Image(postprocessed_depth), 
                wandb.Image(gt),
            ]
            vis_row = copy.deepcopy(row)

            binary_mask = apply_threshold(np.asarray(gt))
            vis_image = apply_vis_to_image(image, binary_mask, color)
            vis_row[5] = wandb.Image(vis_image)
            
            save_qualitative_evaluation_latex(cfg, i, np.array(image), 'RGB', is_best)
            save_qualitative_evaluation_latex(cfg, i, postprocessed_depth, 'DEPTH', is_best)
            save_qualitative_evaluation_latex(cfg, i, vis_image, 'GT', is_best)

            s_measure_row = copy.deepcopy(row)

            for sota_model_name, mapped_sota_model_name in zip(sota_model_names, mapped_sota_model_names):
                file_path_pattern = os.path.join(cfg.sotas_working_dir, sota_model_name, dataset_name, f'{image_name}.*')
                files = glob.glob(file_path_pattern)
                assert len(files) == 1, f'Can not find the salient map {image_name} \
                    of SOTA {sota_model_name} on dataset {dataset_name}, \
                    Length = {len(files)}, File path pattern = {file_path_pattern}'
                pred = Image.open(files[0]).convert('L')
                pred = pred.resize(image.size)
                pred = np.asarray(pred)

                # SM
                row.append(wandb.Image(pred))

                # Visualization
                binary_mask = apply_threshold(pred)
                vis_image = apply_vis_to_image(image, binary_mask, color)
                vis_row.append(wandb.Image(vis_image))
                
                save_qualitative_evaluation_latex(cfg, i, vis_image, mapped_sota_model_name, is_best)

                # S-measure
                sm = Smeasure()
                sm.step(pred, np.asarray(gt))
                s_measure_row.append(sm.get_results()["sm"])
                
            results.append(row)
            vis_results.append(vis_row)
            s_measure_results.append(s_measure_row)

    table = wandb.Table(data=results, columns=columns)
    vis_table = wandb.Table(data=vis_results, columns=columns)
    s_measure_table = wandb.Table(data=s_measure_results, columns=columns)
    
    df = pd.DataFrame(s_measure_results, columns=columns)
    df.to_csv(__qualitative_evaluation_csv_path(cfg, is_best), index=False)
    
    return table, vis_table, s_measure_table

def generate_s_measure_csv(
    cfg: base_cfg,
    dataset_names: List[str], 
    sota_model_names: List[str],
) -> None:
    """Calculate S-measure for the entire dataset
    Each dataset has a seperated csv file.
    
    dataset_names can be ['COME-E', 'COME-H']
    """    
    unzip_SOTAs(
        cfg, 
        [dataset_names for _ in range(len(sota_model_names))],
        sota_model_names
    )
    
    s_measure_results = []

    for dataset_name in dataset_names:
        dataset_dir_path = os.path.join(
            cfg.test_datasets_working_dir_path, 
            dataset_name
        )
        print(f'Dataset: {dataset_name}')
        dataset = TestDataset(cfg, dataset_dir_path)

        length = len(dataset)

        for i in tqdm(range(length)):
            image, depth, gt, image_name = dataset.get_raw_item(i)
            
            s_measure_row = [
                i, image_name, dataset_name, 
            ]

            for sota_model_name in sota_model_names:
                file_path_pattern = os.path.join(
                    cfg.sotas_working_dir, sota_model_name, 
                    dataset_name, f'{image_name}.*'
                )
                files = glob.glob(file_path_pattern)
                assert len(files) == 1, f'Can not find the salient map {image_name} \
                    of SOTA {sota_model_name} on dataset {dataset_name}, \
                    Length = {len(files)}, File path pattern = {file_path_pattern}'
                pred = Image.open(files[0]).convert('L')
                pred = pred.resize(image.size)
                pred = np.asarray(pred)

                # S-measure
                sm = Smeasure()
                sm.step(pred, np.asarray(gt))
                s_measure_row.append(sm.get_results()["sm"])
            
            s_measure_results.append(s_measure_row)
    
    df = pd.DataFrame(
        s_measure_results, 
        columns=['index', 'image_name', 'dataset_name'] + sota_model_names
    )
    csv_file_path = os.path.join(
        cfg.benchmark_csv_dir_path, 
        f'smeasure_set{cfg.datasets_set}.csv'
    )
    print(f'Saved S-measure results into {csv_file_path}')
    df.to_csv(csv_file_path, index=False)
