import os
from typing import List
from zipfile import ZipFile

from .configs.base_config import base_cfg
from .run_type import run_type, RUN_TYPE

def unzip(zip_path: str, dest_dir: str) -> None:
    """Note: it will ignore if this zip file has been unzipped before

    Args:
        zip_path (str): the path to zip file
        dest_dir (str): the destination directory to be unzipped to.
    """
    if not os.path.isdir(dest_dir):
        with ZipFile(zip_path, 'r') as zip_obj:
            zip_obj.extractall(dest_dir)

def unzip_datasets(cfg: base_cfg) -> None:
    """
    Note: If the unzip process is abruptedly stopped. 
        You need to delete the whole folder '/content/datasets' and rerun again

    Directory format:
    datasets:
     |-- previous_train
        |---- GT
        |---- RGB
        |---- depths
     |-- previous_dev
        |---- GT
        |---- RGB
        |---- depths
     |-- previous_test
        |---- dataset1
            |---- GT
            |---- RGB
            |----depths
        |---- dataset2
            |---- GT
            |---- RGB
            |----depths
    """
    os.makedirs(cfg.datasets_working_dir_path, exist_ok=True)
    
    unzip(cfg.train_dataset_zip_path, cfg.train_dataset_working_dir_path)
    if cfg.dev_dataset_zip_path is not None:
        unzip(cfg.dev_dataset_zip_path, cfg.dev_dataset_working_dir_path)

    os.makedirs(cfg.test_datasets_working_dir_path, exist_ok=True)
    for test_dataset_name in cfg.test_dataset_names:
        unzip(
            os.path.join(cfg.test_datasets_dir_path, f'{test_dataset_name}.zip'),
            os.path.join(cfg.test_datasets_working_dir_path, test_dataset_name)
        )

def unzip_SOTAs(
    cfg: base_cfg,
    sotas_datasets: List[List[str]], # [['COME-E', 'COME-H'], [...], ...]
    sota_model_names: List[str], # ['SPNet', 'BBSNet', ...]
) -> None:
    """Directory format:

    {benchmark_dir_path}
        |---- {sota_model_name}
                    |---- {dataset}.zip
                    |---- COME-E.zip
                    |---- COME-H.zip

    Unzip to 

    {cfg.sotas_working_dir}
        |---- v1
              |---- {sota_model_name}
                        |---- {dataset}
                                |---- pred_1.png
                                |---- pred_2.png
              |---- SPNet
                        |---- COME-E
                                |---- pred_1.png
                                |---- pred_2.png
        |---- v2
              |---- {sota_model_name}
                        |---- {dataset}
                                |---- pred_1.png
                                |---- pred_2.png

    """
    print('run_type', run_type.rt)
    if not run_type.rt == RUN_TYPE.COLAB:
        return
    
    working_dir = cfg.sotas_working_dir # Colab
    os.makedirs(working_dir, exist_ok=True)
    assert len(sotas_datasets) == len(sota_model_names), \
        "sotas_datasets and sota_model_names must have the same size"

    for datasets_per_sota, sota_model_name in zip(sotas_datasets, sota_model_names):
        model_working_dir_path = os.path.join(working_dir, sota_model_name) # Colab
        os.makedirs(model_working_dir_path, exist_ok=True)

        for dataset_name in datasets_per_sota:
            zip_path = os.path.join(
                cfg.benchmark_dir_path, sota_model_name, f'{dataset_name}.zip'
            ) # GoogleDrive
            pred_working_dir_path = os.path.join(
                model_working_dir_path,
                dataset_name,
            )
            unzip(zip_path, pred_working_dir_path)

def prepare_datasets(cfg: base_cfg):
    if run_type.rt == RUN_TYPE.COLAB:
        print('Unzip datasets ...')
        unzip_datasets(cfg)
