import torch
from torch import Tensor
import math
import warnings
from typing import Tuple, Union
from einops import rearrange

import os
from ..run_type import RUN_TYPE
from ..configs.base_config import base_cfg

def pair(t: Union[int, Tuple[int, int]]) -> Tuple[int, int]:
    return t if isinstance(t, tuple) else (t, t)

def build_2d_sincos_posemb(h: int, w: int, embed_dim=1024, temperature=10000.):
    """Sine-cosine positional embeddings from MoCo-v3

    Source: https://github.com/facebookresearch/moco-v3/blob/main/vits.py
    """
    grid_w = torch.arange(w, dtype=torch.float32)
    grid_h = torch.arange(h, dtype=torch.float32)
    grid_w, grid_h = torch.meshgrid(grid_w, grid_h, indexing='xy')
    assert embed_dim % 4 == 0, 'Embed dimension must be divisible by 4 for 2D sin-cos position embedding'
    pos_dim = embed_dim // 4
    omega = torch.arange(pos_dim, dtype=torch.float32) / pos_dim
    omega = 1. / (temperature ** omega)
    out_w = torch.einsum('m,d->md', [grid_w.flatten(), omega])
    out_h = torch.einsum('m,d->md', [grid_h.flatten(), omega])
    pos_emb = torch.cat([torch.sin(out_w), torch.cos(out_w), torch.sin(out_h), torch.cos(out_h)], dim=1)[None, :, :]
    pos_emb = rearrange(pos_emb, 'b (h w) d -> b d h w', h=h, w=w, d=embed_dim)
    return pos_emb


def _no_grad_trunc_normal_(tensor: Tensor, mean: float, std: float, a: float, b: float):
    # Cut & paste from PyTorch official master until it's in a few official releases - RW
    # Method based on https://people.sc.fsu.edu/~jburkardt/presentations/truncated_normal.pdf
    def norm_cdf(x):
        # Computes standard normal cumulative distribution function
        return (1. + math.erf(x / math.sqrt(2.))) / 2.

    if (mean < a - 2 * std) or (mean > b + 2 * std):
        warnings.warn("mean is more than 2 std from [a, b] in nn.init.trunc_normal_. "
                    "The distribution of values may be incorrect.",
                    stacklevel=2)

    with torch.no_grad():
        # Values are generated by using a truncated uniform distribution and
        # then using the inverse CDF for the normal distribution.
        # Get upper and lower cdf values
        l = norm_cdf((a - mean) / std)
        u = norm_cdf((b - mean) / std)

        # Uniformly fill tensor with values from [l, u], then translate to
        # [2l-1, 2u-1].
        tensor.uniform_(2 * l - 1, 2 * u - 1)

        # Use inverse cdf transform for normal distribution to get truncated
        # standard normal
        tensor.erfinv_()

        # Transform to proper mean, std
        tensor.mul_(std * math.sqrt(2.))
        tensor.add_(mean)

        # Clamp to ensure it's in the proper range
        tensor.clamp_(min=a, max=b)
        return tensor


def trunc_normal_(tensor: Tensor, mean=0., std=1., a=-2., b=2.):
    # type: (Tensor, float, float, float, float) -> Tensor
    r"""Fills the input Tensor with values drawn from a truncated
    normal distribution. The values are effectively drawn from the
    normal distribution :math:`\mathcal{N}(\text{mean}, \text{std}^2)`
    with values outside :math:`[a, b]` redrawn until they are within
    the bounds. The method used for generating the random values works
    best when :math:`a \leq \text{mean} \leq b`.
    Args:
        tensor: an n-dimensional `Tensor`
        mean: the mean of the normal distribution
        std: the standard deviation of the normal distribution
        a: the minimum cutoff value
        b: the maximum cutoff value
    Examples:
        >>> w = torch.empty(3, 5)
        >>> nn.init.trunc_normal_(w)
    """
    return _no_grad_trunc_normal_(tensor, mean, std, a, b)


def drop_path(x: Tensor, drop_prob: float = 0., training: bool = False):
    """Drop paths (Stochastic Depth) per sample (when applied in main path of residual blocks).
    This is the same as the DropConnect impl I created for EfficientNet, etc networks, however,
    the original name is misleading as 'Drop Connect' is a different form of dropout in a separate paper...
    See discussion: https://github.com/tensorflow/tpu/issues/494#issuecomment-532968956 ... I've opted for
    changing the layer and argument names to 'drop path' rather than mix DropConnect as a layer name and use
    'survival rate' as the argument.
    """
    if drop_prob == 0. or not training:
        return x
    keep_prob = 1 - drop_prob
    shape = (x.shape[0],) + (1,) * (x.ndim - 1)  # work with diff dim tensors, not just 2D ConvNets
    random_tensor = keep_prob + torch.rand(shape, dtype=x.dtype, device=x.device)
    random_tensor.floor_()  # binarize
    output = x.div(keep_prob) * random_tensor
    return output

def get_pretrained_backbone_path(cfg: base_cfg, run_type: str) -> str:
    if cfg.pretrained_backbone == 'multi-vit':
        if run_type == RUN_TYPE.COLAB:
            pretrained_backbone_path = os.path.join(
                cfg.source_code_dir, 'backbones', 'multimae',
                'multimae-b_98_rgb+-depth-semseg_1600e_multivit-afff3f8c.pth'
            )
        elif run_type == RUN_TYPE.KAGGLE:
            pretrained_backbone_path = os.path.join(
                '/', 'kaggle', 'input', 'pretrainedweightsmultimae',
                'multimae-b_98_rgb-depth-semseg_1600e_multivit-afff3f8c.pth'
            )
        elif run_type == RUN_TYPE.UBUNTU:
            pretrained_backbone_path = os.path.join(
                '.', 'pretrained_weights', 
                'multimae-b_98_rgb+-depth-semseg_1600e_multivit-afff3f8c.pth'
            )
        else:
            raise Exception(f'Unsupported load pretrained-backbone for run_type {run_type}')
    elif cfg.pretrained_backbone == 'vit':
        if run_type == RUN_TYPE.COLAB:
            pretrained_backbone_path = os.path.join(
                cfg.source_code_dir, 'backbones', 'multimae',
                'vit_base_patch16_224_blocks.pth'
            )
        elif run_type == RUN_TYPE.KAGGLE:
            pretrained_backbone_path = os.path.join(
                '/', 'kaggle', 'input', 'pretrainedweightsmultimae',
                'vit_base_patch16_224_blocks.pth'
            )
        elif run_type == RUN_TYPE.UBUNTU:
            pretrained_backbone_path = os.path.join(
                '.', 'pretrained_weights', 
                'vit_base_patch16_224_blocks.pth'
            )
        else:
            raise Exception(f'Unsupported load pretrained-backbone for run_type {run_type}')
    elif cfg.pretrained_backbone == 'large-mae':
        if run_type == RUN_TYPE.COLAB:
            pretrained_backbone_path = os.path.join(
                cfg.source_code_dir, 'backbones', 'multimae',
                'selected_mae_pretrain_vit_large.pth'
            )
        elif run_type == RUN_TYPE.KAGGLE:
            pretrained_backbone_path = os.path.join(
                '/', 'kaggle', 'input', 'pretrainedweightsmultimae',
                'selected_mae_pretrain_vit_large.pth'
            )
        elif run_type == RUN_TYPE.UBUNTU:
            pretrained_backbone_path = os.path.join(
                '.', 'pretrained_weights',
                'selected_mae_pretrain_vit_large.pth'
            )
        else:
            raise Exception(f'Unsupported load pretrained-backbone for run_type {run_type}')
    elif cfg.pretrained_backbone == 'huge-mae':
        if run_type == RUN_TYPE.COLAB:
            pretrained_backbone_path = os.path.join(
                cfg.source_code_dir, 'backbones', 'multimae',
                'selected_mae_pretrain_vit_huge.pth'
            )
        elif run_type == RUN_TYPE.KAGGLE:
            pretrained_backbone_path = os.path.join(
                '/', 'kaggle', 'input', 'pretrainedweightsmultimae',
                'selected_mae_pretrain_vit_huge.pth'
            )
        elif run_type == RUN_TYPE.UBUNTU:
            pretrained_backbone_path = os.path.join(
                '.', 'pretrained_weights',
                'selected_mae_pretrain_vit_huge.pth'
            )
        else:
            raise Exception(f'Unsupported load pretrained-backbone for run_type {run_type}')
    else:
        raise Exception(f'Unsupported {cfg.pretrained_backbone}')
        
    return pretrained_backbone_path
