import logging
from logging import Logger
import functools
from termcolor import colored
import os, sys
from logging import Logger
import wandb

from .dataloader import WarmupDataLoader
from .configs.base_config import base_cfg

@functools.lru_cache()
def create_logger(output_dir: str, name: str = '') -> Logger:
    os.makedirs(output_dir, exist_ok=True)
    
    # create logger
    logger = logging.getLogger(name)
    logger.setLevel(logging.DEBUG)
    logger.propagate = False

    # create formatter
    fmt = '[%(asctime)s %(name)s] (%(filename)s %(lineno)d): %(levelname)s %(message)s'
    color_fmt = colored('[%(asctime)s %(name)s]', 'green') + \
                colored('(%(filename)s %(lineno)d)', 'yellow') + ': %(levelname)s %(message)s'

    # create console handlers for master process
    console_handler = logging.StreamHandler(sys.stdout)
    console_handler.setLevel(logging.DEBUG)
    console_handler.setFormatter(logging.Formatter(fmt=color_fmt, datefmt='%Y-%m-%d %H:%M:%S'))
    logger.addHandler(console_handler)

    # create file handlers
    file_handler = logging.FileHandler(os.path.join(output_dir, f'log.txt'), mode='a')
    file_handler.setLevel(logging.DEBUG)
    file_handler.setFormatter(logging.Formatter(fmt=fmt, datefmt='%Y-%m-%d %H:%M:%S'))
    logger.addHandler(file_handler)

    return logger

def train_epoch_log(
    cfg: base_cfg,
    epoch: int,
    logger: Logger,
    batch_idx: int,
    lr: float,
    loss: float,
    sum_loss: float,
    warmup_dataloader: WarmupDataLoader,
) -> None:
    print_str = 'Epoch {}/{}'.format(epoch, cfg.nepochs) \
        + ' Iter {}/{}:'.format(batch_idx + 1, cfg.niters_per_epoch) \
        + ' lr=%.4e' % (lr * cfg.lr_scale) \
        + ' loss=%.4f total_loss=%.4f' % (loss, (sum_loss / batch_idx))
    logger.info(print_str)

    wandb.log({
        'loss': loss,
        'avg_loss': sum_loss / batch_idx,
        'lr': (lr * cfg.lr_scale),
        'batch_size': warmup_dataloader.batch_size,
        # 'inputs': wandb.Image(images),
        # "loss_hist": wandb.Histogram(ouputs),
    })