from typing import List
import torch, os
from tqdm import tqdm
from torch.utils.data import DataLoader
from torch import nn, Tensor
import cv2
import numpy as np
import torch.nn.functional as F

from .rgbd_model import RGBDModel
from .configs.base_config import base_cfg
from .checkpoint import load_checkpoint
from .utils import clean_dir
from .dataset_fn import TestDataset
from .device import device

@torch.no_grad()
def generate_saliency_maps_per_dataloader(
    cfg: base_cfg,
    dataloader: DataLoader, 
    model: RGBDModel, 
    save_dataset_dir: str,
    is_padding: bool = True, is_fp16: bool = False
) -> None:
    os.makedirs(save_dataset_dir, exist_ok=True)
    for i_batch, (images, depths, gts, indices, image_sizes, image_names) in tqdm(
        enumerate(dataloader), total=len(dataloader.dataset) // dataloader.batch_size
    ):
        gpu_images: Tensor = images.to(device)
        gpu_depths: Tensor = depths.to(device)
        gpu_gts: Tensor = depths.to(device)
        with torch.cuda.amp.autocast(enabled=is_fp16):
            preds_no_sigmoid: Tensor = model.inference(gpu_images, gpu_depths)

        for pred_no_sigmoid, image_name, image_size in zip(preds_no_sigmoid, image_names, image_sizes):
            if is_padding:
                w, h = image_size.numpy()
                k = max(w, h)
                res: Tensor = F.interpolate(
                    pred_no_sigmoid.unsqueeze(0), size=(k, k), 
                    mode='bilinear', align_corners=False
                )
                res = res[:, :, int((k-h)/2.): int((k+h)/2.), int((k-w)/2.): int((k+w)/2.)]
            else:
                res: Tensor = F.interpolate(
                    pred_no_sigmoid.unsqueeze(0), size=(image_size[1], image_size[0]), 
                    mode='bilinear', align_corners=False
                )
            res = res.sigmoid().data.cpu().numpy().squeeze()
            res = (res - res.min()) / (res.max() - res.min() + 1e-8)

            if is_fp16:
                res = np.float32(res)
            cv2.imwrite(os.path.join(save_dataset_dir, image_name),res*255)

        del gpu_images, gpu_depths, gpu_gts, preds_no_sigmoid

def get_experiment_saliency_maps_working_dir(cfg: base_cfg, epoch: int) -> str:
    rs = f'{cfg.experiment_name}_epoch{epoch}{"_padding" if cfg.is_padding_for_test else ""}'
    if cfg.is_inference_with_no_depth:
        rs += '_nodepth'
    return rs

@torch.no_grad()
def generate_saliency_maps(
    cfg: base_cfg, model: nn.Module, 
    epochs_lst: List[int], # List of epochs [400, 500, ...]
    data_augmentation_version: int,
    set_version: int = 1, # Set version 1, 2
) -> List[str]:
    experiment_names: List[str] = []

    test_dataset_names = cfg.test_dataset_names

    for epoch in epochs_lst:
        ckpt_path = os.path.join(cfg.experiment_dir_path, cfg.experiment_name, f'checkpoint_{epoch}.pt')
        load_checkpoint(model, None, None, None, ckpt_path)
        experiment_name = get_experiment_saliency_maps_working_dir(cfg, epoch)
        experiment_names.append(experiment_name)
        experiment_saliency_maps_working_dir = os.path.join(
            cfg.sotas_working_dir, experiment_name
        )
        clean_dir(experiment_saliency_maps_working_dir)
        print(f'Output to directory {experiment_saliency_maps_working_dir}')

        model.to(device)
        model.eval()

        batch_size = cfg.test_batch_size

        dataset_working_dir_paths: List[str] = [
            os.path.join(cfg.test_datasets_working_dir_path, dataset_name) \
            for dataset_name in test_dataset_names
        ] # Colab

        for dataset_name in test_dataset_names:
            print(f'Dataset {dataset_name}')
            dataset_working_dir_path = os.path.join(
                cfg.test_datasets_working_dir_path, dataset_name
            )
            dataset = TestDataset(cfg, dataset_working_dir_path)
            dataloader = DataLoader(
                dataset, batch_size=batch_size, 
                shuffle=False, num_workers=cfg.num_workers
            )
            generate_saliency_maps_per_dataloader(
                cfg, dataloader, model, 
                os.path.join(experiment_saliency_maps_working_dir, dataset_name),
                is_fp16 = cfg.is_fp16,
                is_padding = cfg.is_padding_for_test
            )
    return experiment_names
    