import os
import json
from typing import Dict, Optional

class ExperimentManager:
    def __init__(
        self, 
        experiment_name: str, 
        json_dir_path: str,
        experiment_dir_path: str,
    ):
        self.experiment_name = experiment_name
        
        self.latest_epoch: Optional[int] = None
        self.best_epoch: Optional[int] = None
        self.best_mae: float = 1.0
        os.makedirs(json_dir_path, exist_ok=True)
        self.json_file_path = os.path.join(
            json_dir_path, 
            f'{self.experiment_name}.json'
        )
        self.experiment_dir_path = experiment_dir_path
        self.load()

    def load(self) -> None:
        d = self.get_exp_json_info()
        if d is not None:
            self.latest_epoch = d['latest_epoch']
            self.best_epoch = d['best_epoch']
            self.best_mae = d['best_mae']
        
    def get_exp_json_info(self) -> Dict:
        if not os.path.isfile(self.json_file_path):
            return None
        with open(self.json_file_path, 'r') as f:
            return json.load(f)

    def write_exp_json_info(self) -> None:
        json_object = json.dumps(
            dict(
                latest_epoch=self.latest_epoch,
                best_epoch=self.best_epoch,
                best_mae=self.best_mae,
            ), 
            indent=4
        )
        with open(self.json_file_path, 'w') as f:
            f.write(json_object)

    def __str__(self) -> str:
        return f'Latest epoch: {self.latest_epoch} | Best epoch: {self.best_epoch} | Best MAE: {self.best_mae}'

    def clean(self) -> None:
        dir_path = os.path.join(self.experiment_dir_path, self.experiment_name)
        if not os.path.isdir(dir_path):
            return
        epochs = list(sorted([int(f.split('_')[1].split('.')[0]) for f in os.listdir(dir_path)]))
        for epoch in epochs:
            if epoch not in [self.latest_epoch, self.best_epoch]:
                file_path = os.path.join(
                    dir_path,
                    f'checkpoint_{epoch}.pt'
                )
                with open(file_path, 'w') as f:
                    f.write('_')
                os.remove(file_path)
                print('removed', file_path)

    def update(self, new_epoch: int, mae: Optional[float] = None) -> None:
        self.latest_epoch = new_epoch
        if mae is not None:
            if mae <= self.best_mae:
                self.best_epoch = new_epoch
                self.best_mae = mae
        
        self.write_exp_json_info()
        self.clean()

        print(str(self))
