import os
from typing import List, Optional, Tuple
import shutil
import wandb
import pandas as pd
from pprint import pprint
from tqdm import tqdm
import numpy as np

from .prepare_datasets import unzip
from .configs.base_config import base_cfg
from .evaluation_metrics import CalTotalMetric
from .dataset_fn import EvalDataset
from .utils import get_sota_type, clean_dir
from .run_type import run_type
from .rgbd_model import RGBDModel
from .generate_saliency_maps import generate_saliency_maps
from .wandb_manager import wandb_init_sota_benchmark

def get_existed_tracking_tables_from_csv(csv_file_path: str) -> List[
    List # str, (float, float, float, float) x n, int
]:
    if os.path.isfile(csv_file_path):
        df = pd.read_csv(
            csv_file_path, 
            header=0
        )
        return df.to_numpy().tolist()
    else:
        return []

def __save_tracking_each_df(
    results: List,
    benchmark_csv_path: str,
    columns: List[str] = [],
) -> List:
    df = pd.DataFrame(data=results, columns=columns)
    df.sort_values(by=['Type', 'Model'], inplace=True)
    df.to_csv(benchmark_csv_path, index=False)
    print('save csv', benchmark_csv_path)
    return df

def __generate_list_of_columns_names(dataset_names: List[str]) -> List[str]:
    columns = ['Model']
    for dataset_name in dataset_names:
        columns += [
            f'{dataset_name}_MAE',
            f'{dataset_name}_S',
            f'{dataset_name}_MaxF',
            f'{dataset_name}_MaxE'
        ]     

    columns += ['Type']
    
    # Average MAE over all test datasets
    columns += [
        'MAE',
        'S',
        'MaxF',
        'MaxE'
    ]   

    return columns

def save_tracking_csv(
    dataset_names: List[str], # to generate list of columns names
    tracking_results_lst: List,
    csv_file_path: str, 
    version: int = 1,
    additional_name: Optional[str] = None,
) -> None:
    # Full version: MAE, S, MaxF, MaxE
    columns = __generate_list_of_columns_names(dataset_names)
    df = __save_tracking_each_df(
        tracking_results_lst, 
        csv_file_path,
        columns=columns
    )
    
    # Minimal version: only S_measure
    minimal_columns = [column for column in columns \
        if 'MAE' not in column and 'MaxF' not in column and 'MaxE' not in column]
    minimal_df = df[minimal_columns]

    run = wandb_init_sota_benchmark(
        version, additional_name=additional_name
    )
    run.log({
        'benchmark': wandb.Table(
            dataframe=df, columns=columns
        ),
        'benchmark_s': wandb.Table(
            dataframe=minimal_df, columns=minimal_columns
        )
    })
    run.finish()
    
def refresh_evaluate_SOTAs(
    cfg: base_cfg,
    # sotas_datasets: List[List[str]], # [['COME-E', 'COME-H'], [...], ...]
    # sota_model_names: List[str], # ['SPNet', 'BBSNet', ...]
    version: int, # Set version: 1, 2
    # test_datasets_working_dir_path: str, # Colab
    all_test_dataset_names: List[str], # ['COME-E', 'COME-H']
    # benchmark_dir_path: str, # GoogleDrive
) -> None:
    csv_file_path = os.path.join(cfg.benchmark_csv_dir_path, f'sotas_v{version}.csv') # GoogleDrive
    tracking_results_lst = get_existed_tracking_tables_from_csv(csv_file_path)
    pprint(tracking_results_lst[:10], width=500)

    save_tracking_csv(
        all_test_dataset_names, tracking_results_lst, 
        csv_file_path, version=version
    )

def evaluate_SOTAs(
    cfg: base_cfg,
    sotas_datasets: List[List[str]], # [['COME-E', 'COME-H'], [...], ...]
    sota_model_names: List[str], # ['SPNet', 'BBSNet', ...]
    version: int, # Set version: 1, 2
    test_datasets_working_dir_path: str, # Colab
    all_test_dataset_names: List[str], # ['COME-E', 'COME-H']
    benchmark_dir_path: str, # GoogleDrive
) -> None:
    """Directory format:

    {benchmark_dir_path}
        |---- {sota_model_name}
                    |---- {dataset}.zip
                    |---- COME-E.zip
                    |---- COME-H.zip

    Unzip to 

    {cfg.sotas_working_dir}
        |---- v1
              |---- {sota_model_name}
                        |---- {dataset}
                                |---- pred_1.png
                                |---- pred_2.png
              |---- SPNet
                        |---- COME-E
                                |---- pred_1.png
                                |---- pred_2.png
        |---- v2
              |---- {sota_model_name}
                        |---- {dataset}
                                |---- pred_1.png
                                |---- pred_2.png

    """
    working_dir = cfg.sotas_working_dir # Colab
    os.makedirs(working_dir, exist_ok=True)
    assert len(sotas_datasets) == len(sota_model_names), \
        "sotas_datasets and sota_model_names must have the same size"

    csv_file_path = os.path.join(cfg.benchmark_csv_dir_path, f'sotas_v{version}.csv') # GoogleDrive
    tracking_results_lst = get_existed_tracking_tables_from_csv(csv_file_path)
    pprint(tracking_results_lst[:5], width=1000)

    inconsistent_sota_datasets: List[Tuple[str, str]] = [
        # ['sota_model_name', 'dataset_name']
    ] 

    for datasets_per_sota, sota_model_name in zip(sotas_datasets, sota_model_names):
        model_working_dir_path = os.path.join(working_dir, sota_model_name) # Colab
        os.makedirs(model_working_dir_path, exist_ok=True)
        print(f'Method {sota_model_name} :')

        tracking_results_dict = dict()

        for dataset_name in datasets_per_sota:
            zip_path = os.path.join(
                benchmark_dir_path, sota_model_name, f'{dataset_name}.zip'
            ) # GoogleDrive
            pred_working_dir_path = os.path.join(
                model_working_dir_path,
                dataset_name,
            )
            unzip(zip_path, pred_working_dir_path)

            cal_total_seg_metrics = CalTotalMetric()
            print(f'\nEvaluating SOTA {sota_model_name} on dataset {dataset_name}')
            dataset = EvalDataset(
                pred_root = pred_working_dir_path,
                gt_root = os.path.join(
                    test_datasets_working_dir_path, 
                    dataset_name,
                    'GT'
                )
            )
            if not dataset.is_consistent:
                inconsistent_sota_datasets.append((sota_model_name, dataset_name))
            for i, (pred, gt, image_path) in tqdm(enumerate(dataset), total=len(dataset)):
                pred = np.array(pred)
                gt = np.array(gt)
                cal_total_seg_metrics.step(pred, gt)

                # if i > 10:
                #     break

            numerical_results = cal_total_seg_metrics.get_results()
            mae,s,max_f,max_e = numerical_results['MAE'], numerical_results['SM'], \
                numerical_results['maxF'], numerical_results['maxE']

            tracking_results_dict[dataset_name] = [mae,s,max_f,max_e]
            print('MAE:',mae,'| Smeasure:',s,'| max_f:',max_f,'| max_e:',max_e)

        tracking_results = [sota_model_name]

        for dataset_name in all_test_dataset_names:
            if dataset_name in tracking_results_dict:
                tracking_results += tracking_results_dict[dataset_name]
            else:
                tracking_results += [-1, -1, -1, -1]

        tracking_results.append(get_sota_type(sota_model_name)) # type of SOTA
        
        calculate_average_evaluation_metrics(tracking_results, cfg)

        tracking_results_lst.append(tracking_results)

        save_tracking_csv(
            all_test_dataset_names, tracking_results_lst, 
            csv_file_path, version=version
        )
        print('\n\n')

    print('inconsistent_sota_datasets')
    pprint(inconsistent_sota_datasets)

def evaluate_my_experiments(
    cfg: base_cfg,
    epochs_lst: List[int],
    is_save_salient_maps: bool = False,
    is_calculate_evaluation_metrics: bool = True,
) -> None:
    sota_model_names = generate_saliency_maps(
        cfg, 
        model = RGBDModel(cfg, run_type=run_type.rt),
        epochs_lst = epochs_lst,
        data_augmentation_version = cfg.data_augmentation_version,
        set_version=cfg.datasets_set,
    ) # ['exp_v4.3_epoch100', ...]
    
    sotas_datasets = [
        cfg.test_dataset_names for _ in range(len(sota_model_names))
    ]

    if is_save_salient_maps:
        save_experiment_salient_maps(cfg, sota_model_names)

    if is_calculate_evaluation_metrics:
        evaluate_SOTAs(
            cfg,
            sotas_datasets = sotas_datasets,
            sota_model_names = sota_model_names,
            version = cfg.datasets_set,
            test_datasets_working_dir_path = cfg.test_datasets_working_dir_path,
            all_test_dataset_names = cfg.test_dataset_names,
            benchmark_dir_path = cfg.benchmark_dir_path,
        )

def save_experiment_salient_maps(
    cfg: base_cfg,
    sota_model_names: List[str],
) -> None:
    for sota_model_name in sota_model_names:
        results_src_dir_path = os.path.join(cfg.sotas_working_dir, sota_model_name)
        dataset_names = os.listdir(results_src_dir_path)

        results_dest_dir_path = os.path.join(cfg.benchmark_dir_path, sota_model_name)
        clean_dir(results_dest_dir_path)

        for dataset_name in dataset_names:
            dataset_dir_path = os.path.join(results_src_dir_path, dataset_name)
            shutil.make_archive(
                os.path.join(results_dest_dir_path, dataset_name),
                'zip',
                os.path.join(results_src_dir_path, dataset_name),
            )

def refactor_evaluation_metrics_benchmark(cfg: base_cfg):
    csv_file_path = os.path.join(cfg.benchmark_csv_dir_path, f'sotas_v{cfg.datasets_set}.csv') # GoogleDrive
    tracking_results_lst = get_existed_tracking_tables_from_csv(csv_file_path)
    no_test_datasets = len(cfg.test_dataset_names)
    for tracking_results in tracking_results_lst:
        calculate_average_evaluation_metrics(tracking_results, cfg)
    
    save_tracking_csv(
        cfg.test_dataset_names, tracking_results_lst, 
        csv_file_path, version=cfg.datasets_set
    )
    
def calculate_average_evaluation_metrics(tracking_results: List, cfg: base_cfg):
    '''Calculate average evaluation metrics [mae, s, max_f, max_e]
    '''
    no_test_datasets = len(cfg.test_dataset_names)
    mae = 0.
    s = 0.
    max_f = 0.
    max_e = 0.
    total_quantities = 0
    for i in range(no_test_datasets):
        if tracking_results[1 + i*4] > 0:
            total_quantities += cfg.test_dataset_quantities[i]
            mae += tracking_results[1 + i*4] * cfg.test_dataset_quantities[i]
            s += tracking_results[2 + i*4] * cfg.test_dataset_quantities[i]
            max_f += tracking_results[3 + i*4] * cfg.test_dataset_quantities[i]
            max_e += tracking_results[4 + i*4] * cfg.test_dataset_quantities[i]
        
    mae /= total_quantities
    s /= total_quantities
    max_f /= total_quantities
    max_e /= total_quantities
        
    tracking_results.extend([mae, s, max_f, max_e])
