import os
import torch.distributed as dist
import torch

def get_rank() -> int:
    """
    Get the rank of the current process.
    """
    if not dist.is_available():
        return 0
    if not dist.is_initialized():
        return 0
    return dist.get_rank()

def get_device() -> torch.device:
    """
    Get the device of the current process.
    """
    return torch.device(f'cuda:{get_rank()}')


def is_master_proc() -> bool:
    """
    Determines if the current process is the master process.
    """
    if dist.is_initialized():
        return dist.get_rank() == 0
    else:
        return True

def get_world_size():
    """
    Get the size of the world.
    """
    if not dist.is_available():
        return 1
    if not dist.is_initialized():
        return 1
    return dist.get_world_size()


def init_distributed() -> None:
    '''The package needs to be initialized using 
    the torch.distributed.init_process_group() function 
    before calling any other methods. 
    This blocks until all processes have joined.
    '''
    
    # Initializes the distributed backend 
    # which will take care of synchronizing nodes/GPUs
    dist_url = "env://" # default

    # only works with torch.distributed.launch // torch.run
    rank = int(os.environ["RANK"])
    world_size = int(os.environ['WORLD_SIZE'])
    local_rank = int(os.environ['LOCAL_RANK'])
    dist.init_process_group(
        backend="nccl", # Stands for NVIDIA Collective Communications Library
        init_method=dist_url, # Mutually exclusive with `store`
        world_size=world_size, # Number of processes participating in the job
        rank=rank # Rank of the current process
    )

    # this will make all .cuda() calls work properly
    torch.cuda.set_device(local_rank)

    # synchronizes all the threads to reach this point 
    # before moving on
    dist.barrier()
