from typing import Optional, Tuple
from torch.utils.data import Dataset
import numpy as np
from PIL import Image
import abc
from torch import Tensor
import os

from .configs.base_config import base_cfg
from .utils import list_files
from .data_augmentation import (
    DataAugmentationV2, 
    DataAugmentationV4, 
    DataAugmentationV5,
)

class DATASET_TYPE:
    TRAIN = 'train'
    DEV = 'dev'
    TEST = 'test'

class RGBDDataset(Dataset):
    def __init__(
        self,
        cfg: base_cfg,
        root_dir_path: str, # path to 'datasets/train'
        dataset_type: DATASET_TYPE,
        max_size: Optional[int] = None, # for debug only
    )->None:
        if dataset_type == DATASET_TYPE.TRAIN:
            self.image_size = cfg.image_size
            is_padding = cfg.is_padding
        elif dataset_type == DATASET_TYPE.DEV:
            self.image_size = cfg.image_size
            is_padding = cfg.is_padding
        elif dataset_type == DATASET_TYPE.TEST:
            self.image_size = cfg.test_image_size
            is_padding = cfg.is_padding_for_test
        else:
            raise NotImplementedError(f'Unsupported dataset_type {dataset_type}')
        
        self.images, self.depths, self.gts = list_files(root_dir_path, max_size)

        self.size: int = len(self.images)
        self.indices = np.arange(self.size)

        self.inputs = cfg.inputs
        self.outputs = cfg.outputs
        self.cfg = cfg

        if cfg.data_augmentation_version == 2:
            print('Using DataAugmentationV2')
            self.data_augmentation = DataAugmentationV2(
                self.image_size, is_padding=is_padding,
                inputs=self.inputs, outputs=self.outputs,
            )
        elif cfg.data_augmentation_version == 4:
            print('Using DataAugmentationV4')
            self.data_augmentation = DataAugmentationV4(
                self.image_size, is_padding=is_padding,
                inputs=self.inputs, outputs=self.outputs,
            )
        elif cfg.data_augmentation_version == 5:
            print('Using DataAugmentationV5')
            self.data_augmentation = DataAugmentationV5(cfg)
        else:
            raise NotImplementedError(f'Unsupported DataAugmentation version {cfg.data_augmentation_version}')

    def get_raw_item(self, index: int)->Tuple[Image.Image, Image.Image, Image.Image, str]:
        index = self.indices[index]

        image = self.rgb_loader(self.images[index])
        depth = self.binary_loader(self.depths[index])
        gt = self.binary_loader(self.gts[index])
        image_name = self.images[index].split('/')[-1].split('.')[0]

        return image, depth, gt, image_name

    def get_image_name(self, index: int) -> Tuple[str, str, str, str]:
        index = self.indices[index]
        image_name = self.images[index].split('/')[-1].split('.')[0]
        return self.images[index], self.depths[index], self.gts[index], image_name

    @abc.abstractmethod
    def __getitem__(self, index: int) -> Tuple:
        pass

    def rgb_loader(self, path: str) -> Image.Image:
        with open(path, 'rb') as f:
            img = Image.open(f)
            return img.convert('RGB')

    def binary_loader(self, path: str) -> Image.Image:
        with open(path, 'rb') as f:
            img = Image.open(f)
            return img.convert('L')

    def __len__(self)->int:
        return self.size

class TrainDataset(RGBDDataset):
    def __init__(
        self, 
        cfg: base_cfg,
        max_size: Optional[int] = None, # for debug only
    )->None:
        super().__init__(
            cfg, cfg.train_dataset_working_dir_path, 
            DATASET_TYPE.TRAIN, max_size=max_size
        )

    def __getitem__(self, index: int, is_transform = True)->Tuple[Tensor, Tensor, Tensor, int]:
        index = self.indices[index]

        image = self.rgb_loader(self.images[index])
        depth = self.binary_loader(self.depths[index])
        gt = self.binary_loader(self.gts[index])

        image, depth, gt = self.data_augmentation(image, depth, gt, is_transform)
        return image, depth, gt, index

class DevDataset(RGBDDataset): 
    def __init__(
        self, 
        cfg: base_cfg,
        max_size: Optional[int] = None, # for debug only
    )->None:
        super().__init__(
            cfg, cfg.dev_dataset_working_dir_path, 
            DATASET_TYPE.DEV, max_size=max_size
        )

    def __getitem__(self, index: int)->Tuple[Tensor, Tensor, Tensor, int]:
        index = self.indices[index]

        image = self.rgb_loader(self.images[index])
        depth = self.binary_loader(self.depths[index])
        gt = self.binary_loader(self.gts[index])

        image, depth, gt = self.data_augmentation(image, depth, gt, False)
        
        return image, depth, gt, index

class TestDataset(RGBDDataset):
    def __init__(
        self,
        cfg: base_cfg,
        root_dir_path: str, # path to 'datasets/test'
        max_size: Optional[int] = None, # for debug only
    )->None:
        super().__init__(
            cfg, root_dir_path, 
            DATASET_TYPE.TEST, max_size=max_size
        )

    def __getitem__(self, index: int) -> Tuple[Tensor, Tensor, Tensor, int, np.ndarray, str]:
        index = self.indices[index]

        image = self.rgb_loader(self.images[index])
        depth = self.binary_loader(self.depths[index])
        gt = self.binary_loader(self.gts[index])

        image_shape = np.array([image.size[0], image.size[1]])

        image, depth, gt = self.data_augmentation(image, depth, gt, False)

        file_name = self.gts[index].split('/')[-1]
        
        return image, depth, gt, index, image_shape, file_name

class EvalDataset(Dataset):
    def __init__(self, pred_root: str, gt_root: str):
        pred_name_lst = list(sorted(os.listdir(pred_root)))

        filtered_name_lst = []

        self.is_consistent=True
        for pred_name in pred_name_lst:
            if not os.path.isfile(os.path.join(gt_root, pred_name)):
                self.is_consistent=False
                print(f'pred_name {pred_name} not found in gt_root. FileNotFound {os.path.join(gt_root, pred_name)}')
            else:
                filtered_name_lst.append(pred_name)

        # if not is_consistent:
        #     raise Exception('2 datasets are not consistent!')

        self.pred_path_lst = list(map(lambda x: os.path.join(pred_root, x), filtered_name_lst))
        self.gt_path_lst = list(map(lambda x: os.path.join(gt_root, x), filtered_name_lst))
        

    def __getitem__(self, item: int) -> Tuple[Image.Image, Image.Image, str]:
        pred = Image.open(self.pred_path_lst[item]).convert('L')
        gt = Image.open(self.gt_path_lst[item]).convert('L')
        if pred.size != gt.size:
            # print(f'[WARN] Saliency map and GT are not in the same size {self.pred_path_lst[item]}, sm size {pred.size}, gt size {gt.size}')
            pred = pred.resize(gt.size, Image.BILINEAR)
        return pred, gt, self.pred_path_lst[item]

    def __len__(self) -> int:
        return len(self.pred_path_lst)
