from typing import Optional
from torch.utils.data.distributed import DistributedSampler
from torch import nn
from torch.utils.data import DataLoader

from .configs.base_config import base_cfg
from .dataset_fn import TrainDataset

class WarmupDataLoader():
    def __init__(
        self, cfg: base_cfg, 
        train_dataset: TrainDataset
    ) -> None:
        self.cfg = cfg
        self.train_dataset = train_dataset
        self.train_sampler = DistributedSampler(
            dataset=train_dataset, shuffle=True
        )
        self.is_warmup_batch_size = cfg.warmup_min_batch_size is not None
        
        self.train_dataloader = DataLoader(
            self.train_dataset, 
            batch_size=self.cfg.batch_size, 
            sampler=self.train_sampler, 
            num_workers=self.cfg.num_workers,
            
            # If ``True``, the data loader will copy Tensors
            # into device/CUDA pinned memory before returning them
            pin_memory=True,
        )
        self.batch_size = cfg.batch_size
        
        if self.is_warmup_batch_size:
            assert cfg.warmup_epoch_batch_size is not None, \
                f'cfg.warmup_epoch_batch_size should not be None when self.is_warmup_batch_size is not None'
            assert cfg.warmup_min_batch_size >= 1 and cfg.warmup_min_batch_size < cfg.batch_size, \
                f'warmup_min_batch_size {cfg.warmup_min_batch_size} must be greater or equal to 1 and smaller batch_size {cfg.batch_size}'
            assert cfg.warmup_epoch_batch_size <= cfg.warmup_epoch, \
                f'cfg.warmup_epoch_batch_size must be smaller or equal to cfg.warmup_epoch'
            
            max_b = cfg.batch_size
            min_b = cfg.warmup_min_batch_size
            warmup_epoch = cfg.warmup_epoch_batch_size
            
            self.a = (max_b - min_b) / (warmup_epoch - 1)
            self.b = min_b - self.a
        
    def set_epoch(self, epoch: int) -> None:
        if not self.is_warmup_batch_size:
            # In distributed mode, calling the data_loader.sampler.set_epoch() 
            # method at the beginning of each epoch before creating the DataLoader iterator
            # is necessary to make shuffling work properly across multiple epochs. 
            # Otherwise, the same ordering will be always used
            self.train_sampler.set_epoch(epoch)
            return
        
        x = (epoch-1) % self.cfg.warmup_epoch + 1
        if x > self.cfg.warmup_epoch_batch_size:
            self.train_sampler.set_epoch(epoch)
            return
        
        y = int(self.a * x + self.b)
        print(f'Set batch_size {y}')
        self.batch_size = y
        
        self.train_dataloader = DataLoader(
            self.train_dataset, 
            batch_size=y, 
            sampler=self.train_sampler, 
            num_workers=self.cfg.num_workers,
            
            # If ``True``, the data loader will copy Tensors
            # into device/CUDA pinned memory before returning them
            pin_memory=True,
        )
        self.train_sampler.set_epoch(epoch)