from typing import List, Optional, Tuple
from PIL import Image
import torch
import torch.nn.functional as F
from torch import nn
import numpy as np
from torchvision import transforms
import albumentations
from torch import Tensor
import torchvision.transforms.functional as transformsF
from torchvision.transforms import InterpolationMode
import torch.nn.functional as F
import random

from .configs.base_config import base_cfg
from .utils import random_choice

class SquarePad:
    def __init__(self, fill_value=0.):
        self.fill_value = fill_value

    def __call__(self, image: Image) -> Image:
        _, w, h = image.shape
        max_wh = np.max([w, h])
        wp = int((max_wh - w) / 2)
        hp = int((max_wh - h) / 2)
        padding = (hp, hp, wp, wp)
        image = F.pad(image, padding, value=self.fill_value, mode='constant')
        return image

class DataAugmentationV5(torch.nn.Module):
    def __init__(
        self,
        cfg: base_cfg,
    ):
        super(DataAugmentationV5, self).__init__()
        self.inputs = cfg.inputs
        self.outputs = cfg.outputs
        self.image_size = cfg.image_size
        self.resize = transforms.Resize((self.image_size, self.image_size))
        self.to_tensor = transforms.ToTensor()
        self.normalize_image = transforms.Normalize(
            cfg.data_augmentation_config.mean_normalization, 
            cfg.data_augmentation_config.std_normalization,
        )
        self.cfg = cfg
        
    def random_horizontal_flip(self, lst: List[Tensor], p=0.5) -> List[Tensor]:
        if random_choice(p=p):
            return [transformsF.hflip(e) for e in lst]
        return lst
    
    def no_pad_resize(self, lst: List[Tensor]) -> List[Tensor]:
        return [self.resize(e) for e in lst]

    def process_transform_to_tensor(self, lst: List[Tensor]) -> List[Tensor]:
        return [self.to_tensor(e) for e in lst]
    
    def random_gaussian_blur(
        self, 
        tensor: Tensor, 
        p=0.5, 
        max_kernel_size: int = 19, # must be an odd positive integer
    ) -> Tensor:
        if random_choice(p=p):
            kernel_size = random.randrange(1, max_kernel_size, 2)
            return transformsF.gaussian_blur(tensor, kernel_size=kernel_size)
        return tensor
    
    def preprocessing(self, images: Tensor, depths: Tensor) -> Tuple[Tensor, Tensor]:
        images, depths = self.resize(images), self.resize(depths)
        return self.normalize_image(images), depths
    
    def forward(
        self, 
        image: Image.Image, 
        depth: Image.Image, 
        gt: Optional[Image.Image] = None,
        is_transform: Optional[bool] = True,
    ) -> Tuple[Tensor, Tensor, Optional[Tensor]]:
        lst = [image, depth, gt] if gt is not None else [image, depth]

        if not is_transform:
            # Dev or Test
            lst = self.no_pad_resize(lst)
            lst = self.process_transform_to_tensor(lst)
            if gt is not None:
                image, depth, gt = lst
                # gt[gt > 0.0] = 1.0
            else:
                image, depth = lst
            image = self.normalize_image(image)
            return image, depth, gt

        lst = self.random_horizontal_flip(
            lst,
            p = self.cfg.data_augmentation_config.random_horizontal_flip_prob
        )
        lst = self.no_pad_resize(lst)
        lst = self.process_transform_to_tensor(lst)
        
        if gt is not None:
            image, depth, gt = lst
        else:
            image, depth = lst
        image = self.random_gaussian_blur(
            image, 
            p=self.cfg.data_augmentation_config.image_gaussian_config.p, 
            max_kernel_size=self.cfg.data_augmentation_config.image_gaussian_config.max_gaussian_kernel,
        )
        if 'depth' in self.inputs:
            depth = self.random_gaussian_blur(
                depth, 
                p=self.cfg.data_augmentation_config.depth_gaussian_config.p, 
                max_kernel_size=self.cfg.data_augmentation_config.depth_gaussian_config.max_gaussian_kernel,
            )
        image = self.normalize_image(image)

        return image, depth, gt


class DataAugmentationV2(torch.nn.Module):
    def __init__(
        self, 
        image_size: int, 
        inputs: List[str], 
        outputs: List[str], 
        is_padding=True,
    ):
        super(DataAugmentationV2, self).__init__()
        self.image_size = image_size
        self.is_padding = is_padding
        self.inputs = inputs
        self.outputs = outputs

        self.to_tensor = transforms.ToTensor()
        self.to_image = transforms.ToPILImage()
        self.square_pad_0 = SquarePad(fill_value=0.) # for rgb, gt
        self.square_pad_1 = SquarePad(fill_value=1.) # for depth
        self.resize = transforms.Resize((self.image_size, self.image_size))

        self.random_perspective_0 = transforms.RandomPerspective(
            distortion_scale=0.2, p=1.0, fill=0.
        )
        self.random_perspective_1 = transforms.RandomPerspective(
            distortion_scale=0.2, p=1.0, fill=255
        )

        self.longest_max_size = albumentations.augmentations.geometric.resize.LongestMaxSize(
            max_size=self.image_size, p=1
        )

        # RGB, p = 0.5
        self.transform_color_jitter = transforms.ColorJitter(brightness=.5, hue=.3)

        # RGB, p = 1.0
        self.transform_contrast_sharpness = transforms.Compose([
            transforms.RandomAutocontrast(p=0.5),
            transforms.RandomAdjustSharpness(sharpness_factor=2, p=0.5),                             
        ])

        self.normalize_image = transforms.Normalize(
            [0.5, 0.5, 0.5], 
            [0.5, 0.5, 0.5]
        )

    def no_pad_resize(self, lst: List[Tensor]) -> List[Tensor]:
        return [self.resize(e) for e in lst]

    def pad_resize(self, lst: List[Tensor]) -> List[Tensor]:
        gt: Tensor = None
        if len(lst) == 3:
            image, depth, gt = lst
        else:
            image, depth = lst

        image = self.to_tensor(image)
        image = self.square_pad_0(image)
        image = self.resize(image)
        image = self.to_image(image)

        if gt is not None:
            gt = self.to_tensor(gt)
            gt = self.square_pad_0(gt)
            gt = self.resize(gt)
            gt = self.to_image(gt)

        depth = self.to_tensor(depth)
        depth = self.square_pad_1(depth)
        depth = self.resize(depth)
        depth = self.to_image(depth)

        if gt is not None:
            return [image, depth, gt]
        else:
            return [image, depth]

    def process_transform_to_tensor(self, lst: List[Tensor]) -> List[Tensor]:
        return [self.to_tensor(e) for e in lst]

    def random_horizontal_flip(self, lst: List[Tensor], p=0.5) -> List[Tensor]:
        if random_choice(p=p):
            return [transformsF.hflip(e) for e in lst]
        return lst

    def random_vertical_flip(self, lst: List[Tensor], p=0.5) -> List[Tensor]:
        if random_choice(p=p):
            return [transformsF.vflip(e) for e in lst]
        return lst

    def random_rotate(self, lst: List[Tensor], p=0.3) -> List[Tensor]:
        if random_choice(p=p):
            angle = transforms.RandomRotation.get_params(degrees=(0, 90))
            
            rs: List[Tensor] = []
            for i, e in enumerate(lst):
                if i == 1:
                    rs.append(transformsF.rotate(e, angle, InterpolationMode.BICUBIC, fill=255))
                else:
                    rs.append(transformsF.rotate(e, angle, InterpolationMode.BICUBIC))
            return rs
        return lst

    def random_resized_crop(self, lst: List[Tensor], p=0.3) -> List[Tensor]:
        if random_choice(p=p):
            i, j, h, w = transforms.RandomResizedCrop.get_params(
                lst[0], 
                scale=(0.5, 2.0), 
                ratio=(0.75, 1.3333333333333333)
            )
            return [transformsF.resized_crop(
                e, i, j, h, w , 
                [self.image_size, self.image_size], 
                InterpolationMode.BICUBIC
            ) for e in lst]
        return lst

    def random_gaussian_blur(
        self, 
        tensor: Tensor, 
        p=0.5, 
        max_kernel_size: int = 19, # must be an odd positive integer
    ) -> Tensor:
        if random_choice(p=p):
            kernel_size = random.randrange(1, max_kernel_size, 2)
            return transformsF.gaussian_blur(tensor, kernel_size=kernel_size)
        return tensor

    def color_jitter(self, tensor: Tensor, p=0.5) -> Tensor:
        if random_choice(p=p):
            return self.transform_color_jitter(tensor)
        return tensor

    def random_maskout_depth(self, tensor: Tensor, p=0.5) -> Tensor:
        if random_choice(p=p):
            _, h, w = tensor.shape
            xs = np.random.choice(w, 2)
            ys = np.random.choice(h, 2)
            tensor[:, min(ys):max(ys), min(xs):max(xs)] = torch.ones((max(ys)-min(ys), max(xs)-min(xs)))
            return tensor
        return tensor

    def random_perspective(self, lst: List[Tensor], p=0.2) -> List[Tensor]:
        if random_choice(p=p):
            gt: Tensor = None
            if len(lst) == 3:
                image, depth, gt = lst
            else:
                image, depth = lst

            image = self.random_perspective_0(image)

            if gt is not None:
                gt = self.random_perspective_0(gt)

            depth = self.random_perspective_1(depth)

            if gt is not None:
                return [image, depth, gt]
            else:
                return [image, depth]
        return lst

    def preprocessing(self, images: Tensor, depths: Tensor) -> Tuple[Tensor, Tensor]:
        images, depths = self.resize(images), self.resize(depths)
        return self.normalize_image(images), depths

    def forward(
        self, 
        image: Image.Image, 
        depth: Image.Image, 
        gt: Optional[Image.Image] = None,
        is_transform: Optional[bool] = True,
    ) -> Tuple[Tensor, Tensor, Optional[Tensor]]:
        lst = [image, depth, gt] if gt is not None else [image, depth]

        if not is_transform:
            # Dev or Test
            if self.is_padding:
                lst = self.pad_resize(lst)
            else:
                lst = self.no_pad_resize(lst)
            lst = self.process_transform_to_tensor(lst)
            if gt is not None:
                image, depth, gt = lst
                # gt[gt > 0.0] = 1.0
            else:
                image, depth = lst
            image = self.normalize_image(image)
            return image, depth, gt

        lst = self.random_horizontal_flip(lst)
        if random_choice(p=0.2):
            lst = self.pad_resize(lst)
        else:
            lst = self.no_pad_resize(lst)
        lst = self.random_perspective(lst, p=0.2)
        lst = self.random_rotate(lst)
        lst = self.random_resized_crop(lst)
        lst = self.process_transform_to_tensor(lst)
        
        if gt is not None:
            image, depth, gt = lst
        else:
            image, depth = lst

        image = self.color_jitter(image)
        image = self.transform_contrast_sharpness(image)
        image = self.random_gaussian_blur(image, p=0.5, max_kernel_size=19)
        if 'depth' in self.inputs:
            depth = self.random_gaussian_blur(depth, p=0.5, max_kernel_size=36)
        image = self.normalize_image(image)

        return image, depth, gt

class DataAugmentationV4(torch.nn.Module):
    def __init__(
        self, 
        image_size: int, 
        inputs: List[str], 
        outputs: List[str], 
        is_padding=True,
    ):
        super(DataAugmentationV4, self).__init__()
        self.image_size = image_size
        self.is_padding = is_padding
        self.inputs = inputs
        self.outputs = outputs

        self.to_tensor = transforms.ToTensor()
        self.to_image = transforms.ToPILImage()
        self.square_pad_0 = SquarePad(fill_value=0.) # for rgb, gt
        self.square_pad_1 = SquarePad(fill_value=1.) # for depth
        self.resize = transforms.Resize((self.image_size, self.image_size))

        self.random_perspective_0 = transforms.RandomPerspective(
            distortion_scale=0.2, p=1.0, fill=0.
        )
        self.random_perspective_1 = transforms.RandomPerspective(
            distortion_scale=0.2, p=1.0, fill=255
        )

        self.longest_max_size = albumentations.augmentations.geometric.resize.LongestMaxSize(
            max_size=self.image_size, p=1
        )

        # RGB, p = 0.5
        self.transform_color_jitter = transforms.ColorJitter(brightness=.5, hue=.3)

        # RGB, p = 1.0
        self.transform_contrast_sharpness = transforms.Compose([
            transforms.RandomAutocontrast(p=0.3), # TODO p=0.5
            transforms.RandomAdjustSharpness(sharpness_factor=1.2, p=0.3), # TODO p=0.5                           
        ])

        self.normalize_image = transforms.Normalize(
            [0.485, 0.456, 0.406], [0.229, 0.224, 0.225]
        )

        self.normalize_depth = transforms.Normalize(
            [0.5], [0.5]
        )

    def no_pad_resize(self, lst: List[Tensor]) -> List[Tensor]:
        # return [self.resize(e) for e in lst]
        return lst

    def pad_resize(self, lst: List[Tensor]) -> List[Tensor]:
        gt: Tensor = None
        if len(lst) == 3:
            image, depth, gt = lst
        else:
            image, depth = lst

        image = self.to_tensor(image)
        image = self.square_pad_0(image)
        # image = self.resize(image)
        image = self.to_image(image)

        if gt is not None:
            gt = self.to_tensor(gt)
            gt = self.square_pad_0(gt)
            # gt = self.resize(gt)
            gt = self.to_image(gt)

        depth = self.to_tensor(depth)
        depth = self.square_pad_1(depth)
        # depth = self.resize(depth)
        depth = self.to_image(depth)

        if gt is not None:
            return [image, depth, gt]
        else:
            return [image, depth]

    def process_transform_to_tensor(self, lst: List[Tensor]) -> List[Tensor]:
        return [self.to_tensor(e) for e in lst]

    def random_horizontal_flip(self, lst: List[Tensor], p=0.5) -> List[Tensor]:
        if random_choice(p=p):
            return [transformsF.hflip(e) for e in lst]
        return lst

    def random_vertical_flip(self, lst: List[Tensor], p=0.5) -> List[Tensor]:
        if random_choice(p=p):
            return [transformsF.vflip(e) for e in lst]
        return lst

    def random_rotate(self, lst: List[Tensor], degrees=(0, 35), p=0.3) -> List[Tensor]:
        if random_choice(p=p):
            angle = transforms.RandomRotation.get_params(degrees=degrees) # TODO 90
            
            rs: List[Tensor] = []
            for i, e in enumerate(lst):
                if i == 1:
                    rs.append(transformsF.rotate(e, angle, InterpolationMode.BICUBIC, fill=255))
                else:
                    rs.append(transformsF.rotate(e, angle, InterpolationMode.BICUBIC))
            return rs
        return lst

    def random_resized_crop(
        self, lst: List[Tensor], scale=(0.08, 1.0), 
        ratio=(0.75, 1.3333333333333333), p=0.5
    ) -> List[Tensor]: # TODO p = 0.3
        if random_choice(p=p):
            i, j, h, w = transforms.RandomResizedCrop.get_params(
                lst[0], 
                scale=scale, # TODO scale=(0.5, 2.0)
                ratio=ratio
            )
            return [transformsF.resized_crop(
                e, i, j, h, w , 
                [self.image_size, self.image_size], 
                InterpolationMode.BICUBIC
            ) for e in lst]
        return lst

    def random_gaussian_blur(
        self, 
        tensor: Tensor, 
        p=0.5, 
        max_kernel_size: int = 19, # must be an odd positive integer
    ) -> Tensor:
        if random_choice(p=p):
            kernel_size = random.randrange(1, max_kernel_size, 2)
            return transformsF.gaussian_blur(tensor, kernel_size=kernel_size)
        return tensor

    def color_jitter(self, tensor: Tensor, p=0.5) -> Tensor:
        if random_choice(p=p):
            return self.transform_color_jitter(tensor)
        return tensor

    def random_maskout_depth(self, tensor: Tensor, p=0.5) -> Tensor:
        if random_choice(p=p):
            _, h, w = tensor.shape
            xs = np.random.choice(w, 2)
            ys = np.random.choice(h, 2)
            tensor[:, min(ys):max(ys), min(xs):max(xs)] = \
                torch.ones((max(ys)-min(ys), max(xs)-min(xs)))
            return tensor
        return tensor

    def random_perspective(self, lst: List[Tensor], p=0.2) -> List[Tensor]:
        if random_choice(p=p):
            gt: Tensor = None
            if len(lst) == 3:
                image, depth, gt = lst
            else:
                image, depth = lst

            image = self.random_perspective_0(image)

            if gt is not None:
                gt = self.random_perspective_0(gt)

            depth = self.random_perspective_1(depth)

            if gt is not None:
                return [image, depth, gt]
            else:
                return [image, depth]
        return lst

    def preprocessing(self, images: Tensor, depths: Tensor) -> Tuple[Tensor, Tensor]:
        images, depths = self.resize(images), self.resize(depths)
        return self.normalize_image(images), self.normalize_depth(depths)

    def forward(
        self, 
        image: Image.Image, 
        depth: Image.Image, 
        gt: Optional[Image.Image] = None,
        is_transform: Optional[bool] = True,
    ) -> Tuple[Tensor, Tensor, Optional[Tensor]]:
        lst = [image, depth, gt] if gt is not None else [image, depth]

        if not is_transform:
            # Dev or Test
            if self.is_padding:
                lst = self.pad_resize(lst)
            else:
                lst = self.no_pad_resize(lst)
            lst = self.process_transform_to_tensor(lst)
            lst = [self.resize(e) for e in lst]
            if gt is not None:
                image, depth, gt = lst
                # gt[gt > 0.0] = 1.0
            else:
                image, depth = lst
            image = self.normalize_image(image)
            if 'depth' in self.inputs:
                depth = self.normalize_depth(depth)
            return image, depth, gt

        lst = self.random_horizontal_flip(lst, p=0.5)
        lst = self.random_perspective(lst, p=0.5)
        lst = self.random_resized_crop(
            lst, scale=(0.5, 1.0), ratio=(0.75, 1.3333333333333333), p=0.5
        )
        lst = self.random_rotate(lst, degrees=(0, 10), p=0.5)
        lst = self.process_transform_to_tensor(lst)

        lst = [self.resize(e) for e in lst]
        
        if gt is not None:
            image, depth, gt = lst
        else:
            image, depth = lst

        image = self.color_jitter(image, p=0.5)
        image = self.transform_contrast_sharpness(image)
        image = self.random_gaussian_blur(image, max_kernel_size=19, p=0.3)
        if 'depth' in self.inputs:
            depth = self.random_gaussian_blur(depth, max_kernel_size=19, p=0.3)
            depth = self.normalize_depth(depth)
        image = self.normalize_image(image)

        return image, depth, gt

def get_data_augmentation(
    cfg: base_cfg,
    image_size: int,
    is_padding: bool,
) -> nn.Module:
    if cfg.data_augmentation_version == 2:
        print('Using DataAugmentationV2')
        return DataAugmentationV2(
            image_size, 
            is_padding=is_padding,
            inputs=cfg.inputs, 
            outputs=cfg.outputs,
        )
    elif cfg.data_augmentation_version == 4:
        print('Using DataAugmentationV4')
        return DataAugmentationV4(
            image_size,
            is_padding=is_padding,
            inputs=cfg.inputs, 
            outputs=cfg.outputs,
        )
    elif cfg.data_augmentation_version == 5:
        print('Using DataAugmentationV5')
        return DataAugmentationV5(cfg)
    else:
        raise NotImplementedError(f'Unsupported DataAugmentation version {cfg.data_augmentation_version}')
