from typing import Optional
from torch.utils.data import DataLoader
from torch import nn, Tensor
import torch
from tqdm import tqdm
import wandb
from torch.utils.data.distributed import DistributedSampler
import torch.distributed as dist

from .criterion import DevCriterion
from .distributed_training import get_world_size, is_master_proc
from .utils import clean_cache
from .dataset_fn import DevDataset, RGBDDataset
from .configs.base_config import base_cfg
from .logger_fn import Logger
from .device import device, cpu_device

class CrossValidation:
    def __init__(
        self, cfg: base_cfg, max_size: Optional[int] = None, 
        max_track: Optional[int] = None,
        data_augmentation_version: int = 1,
    ) -> None:
        self.cfg = cfg
        self.dev_dataset = DevDataset(cfg)
        dev_sampler = DistributedSampler(
            dataset=self.dev_dataset, shuffle=False
        )
        self.dev_dataloader = DataLoader(
            self.dev_dataset, batch_size=cfg.val_batch_size, 
            sampler=dev_sampler, 
            num_workers=cfg.num_workers,
            pin_memory=True,
        )
        self.dev_criterion = DevCriterion()
        self.dev_num_iters = len(self.dev_dataloader)
        
        self.world_size = get_world_size()
        self.is_master_process = is_master_proc()

    def calculate_dev_mae(
        self, model: nn.Module, epoch: int, logger: Optional[Logger] = None
    ) -> float:
        dataloader = self.dev_dataloader
        dataset = self.dev_dataset
        num_iters = self.dev_num_iters
        return self.__calculate_mae(
            epoch, dataloader, dataset, 
            num_iters, model, 'dev', logger
        )

    @torch.no_grad()
    def __calculate_mae(
        self, epoch: int, dataloader: DataLoader, 
        dataset: RGBDDataset, 
        num_iters: int, model: nn.Module, log_attr: str, 
        logger: Optional[Logger] = None
    ) -> float:
        '''Given that the model is already loaded in GPU
        Note that the model will be in evaluation model after running this function
        '''
        model.eval()

        total_mae: float = 0.0
        if logger is not None and self.is_master_process:
            logger.info(f'Cross-validation [{log_attr}] ...')
        for i_batch, (gpu_images, gpu_depths, gpu_gts, indices) in tqdm(
            enumerate(dataloader, start=1), total=num_iters,
            disable=not self.is_master_process,
        ):
            gpu_images: Tensor = gpu_images.cuda()
            gpu_depths: Tensor = gpu_depths.cuda()
            gpu_gts: Tensor = gpu_gts.cuda()
            
            with torch.cuda.amp.autocast(enabled=self.cfg.is_fp16):
                gpu_out: Tensor = model(gpu_images, gpu_depths)
                mae: Tensor = self.dev_criterion(
                    gpu_out['semseg'].sigmoid(), gpu_gts
                )
                dist.all_reduce(mae)
            
            total_mae += mae.to(cpu_device).item() * indices.shape[0] # * self.world_size
            del gpu_images, gpu_depths, gpu_gts, indices
            clean_cache()

        return total_mae / len(dataset)

def cross_validation_log(
    cfg: base_cfg,
    model: nn.Module, 
    logger: Logger, 
    cross_val: CrossValidation, 
    epoch: int
) -> None:
    clean_cache()

    dev_mae = cross_val.calculate_dev_mae(model, epoch, logger)
    
    if is_master_proc():
        wandb.log({
            # 'train_mae': train_mae, 
            'dev_mae': dev_mae, 
            'epoch': epoch,
        })
        logger.info(f'Epoch {epoch}: Dev MAE {dev_mae:.4f}')
        cfg.em.update(epoch, dev_mae)

    clean_cache()

def test_cross_validation(cfg: base_cfg) -> None:
    from .rgbd_model import RGBDModel
    from .checkpoint import load_checkpoint
    from .run_type import run_type
    from .wandb_manager import wandb_login, wandb_init
    wandb_login(cfg)
    wandb_init('test_cross_validation')

    model = RGBDModel(cfg, run_type=run_type.rt) 
    load_checkpoint(model, None, None, None, ckpt_path = cfg.ckpt_path)

    model.to(device)

    cross_val = CrossValidation(cfg, max_track=10, max_size=100)
    cross_val.calculate_train_mae(model, 2)
    cross_val.calculate_dev_mae(model, 2)

    wandb.finish()
