from torch import nn, Tensor

from .configs.base_config import base_cfg

class Criterion(nn.Module):
    def __init__(self, cfg: base_cfg) -> None:
        super(Criterion, self).__init__()
        self.use_label_smoothing = cfg.label_smoothing > 0.
        self.label_smoothing = cfg.label_smoothing
        
        if self.label_smoothing >= 1.0 or self.label_smoothing < 0:
            raise Exception('Label smoothing must be in range [0.0, 1.0), exclude 1.0')
        
        self.loss = nn.BCEWithLogitsLoss(reduction='mean')
            
    def smooth_label(self, gts: Tensor) -> Tensor:
        '''
        Ref: https://github.com/tensorflow/tensorflow/blob/fcc4b966f1265f466e82617020af93670141b009/tensorflow/python/keras/losses.py#L1573
        '''
        return gts * (1.0 - self.label_smoothing) + 0.5 * self.label_smoothing
        
    def forward(self, preds: Tensor, gts: Tensor):
        if self.use_label_smoothing:
            return self.loss(preds, self.smooth_label(gts))
        else:
            return self.loss(preds, gts)

class DevCriterion(nn.Module):
    def __init__(self) -> None:
        super(DevCriterion, self).__init__()
        self.loss = nn.L1Loss(reduction='mean')
            
    def forward(self, preds: Tensor, gts: Tensor):
        max = preds.amax(dim=(2,3), keepdim=True)
        min = preds.amin(dim=(2,3), keepdim=True)
        preds = (preds-min) / (max - min + 1e-8)
        return self.loss(preds, gts)
