from functools import partial
from typing import Dict, List, Optional

from ..run_type import run_type
from .base_config import base_cfg

class cfg_debug_ubuntu(base_cfg):
    def __init__(self, epoch: Optional[int] = None):
        super().__init__(
            epoch,
            experiment_name = 'debug-training-on-ubuntu',
            datasets_set=1, 
            run_type=run_type.rt
        )
        
        self.description = 'debug-training-on-ubuntu'
        
        '''MultiMAE'''
        self.dim_tokens = 1280
        self.encoder_depth = 32
        self.num_heads = 16
        self.pretrained_backbone = 'huge-mae'
        self.input_patch_size = 14
        self.output_patch_size = 14

        '''Learning rate'''
        self.lr = 1e-5
        self.end_lr = 1e-11
        self.lr_scale = 100 
        
        # self.warmup_epoch_batch_size = 0
        # self.warmup_min_batch_size = 0
        
        self.batch_size = 1            # <---------------
        self.val_batch_size = 10         # <---------------
        self.nepochs = 1000
        
        self.decoder_main_tasks = [['rgb', 'depth']]

        self.data_augmentation_version = 5
        self.save_checkpoints_after_each_n_epochs = 1
        self.num_workers = 8
        self.train_function_version = 2

class cfg_debug_colab(base_cfg):
    def __init__(self, epoch: Optional[int] = None):
        super().__init__(
            epoch,
            experiment_name = 'debug-training-on-colab',
            datasets_set=1, 
            run_type=run_type.rt
        )
        
        self.description = 'debug-training-on-colab'

        '''Learning rate'''
        self.lr = 1e-5
        self.end_lr = 1e-11
        self.lr_scale = 100 
        
        self.batch_size = 20             # <---------------
        self.val_batch_size = 100         # <---------------
        self.nepochs = 1000
        
        
        self.decoder_main_tasks = [['rgb', 'depth']]

        self.data_augmentation_version = 2
        self.save_checkpoints_after_each_n_epochs = 1
        
        self.train_function_version = 2
        
class cfg_debug_kaggle(base_cfg):
    def __init__(self, epoch: Optional[int] = None):
        super().__init__(
            epoch,
            experiment_name = 'debug-training-on-kaggle',
            datasets_set=1, 
            run_type=run_type.rt
        )
        
        self.description = 'debug-training-on-kaggle'

        '''Learning rate'''
        self.lr = 1e-5
        self.end_lr = 1e-11
        self.lr_scale = 100 
        
        self.batch_size = 30             # <---------------
        self.val_batch_size = 200         # <---------------
        self.nepochs = 1000
        
        
        self.decoder_main_tasks = [['rgb', 'depth']]

        self.data_augmentation_version = 5
        self.save_checkpoints_after_each_n_epochs = 1
        
        self.train_function_version = 2

class cfg_set_1(base_cfg):
    def __init__(self, epoch: Optional[int] = None):
        super().__init__(
            epoch,
            experiment_name = 'cfg-set-1',
            datasets_set=1, 
            run_type=run_type.rt
        )

class cfg_set_2(base_cfg):
    def __init__(self, epoch: Optional[int] = None):
        super().__init__(
            epoch,
            experiment_name = 'cfg-set-2',
            datasets_set=2, 
            run_type=run_type.rt
        )

class cfg_set_3(base_cfg):
    def __init__(self, epoch: Optional[int] = None):
        super().__init__(
            epoch,
            experiment_name = 'cfg-set-3',
            datasets_set=3, 
            run_type=run_type.rt
        )

class cfg_set_4(base_cfg):
    def __init__(self, epoch: Optional[int] = None):
        super().__init__(
            epoch,
            experiment_name = 'cfg-set-4',
            datasets_set=4, 
            run_type=run_type.rt
        )


# Old class name: cfgv4_1_16
class cfgv4_0_18(base_cfg):
    def __init__(self, epoch: Optional[int] = None):
        super().__init__(
            epoch,
            experiment_name = 'exp_v4.0.18',
            datasets_set=1, 
            run_type=run_type.rt
        )

        self.description = 'DAv4-Base'
        self.accum_iter = 2              # <---------------

        '''Learning rate'''
        self.lr = 1e-5
        self.end_lr = 1e-11
        self.lr_scale = 100 
        
        self.batch_size = 32             # <---------------
        self.val_batch_size = 300         # <---------------
        self.nepochs = 200
        
        
        self.decoder_main_tasks = [['rgb', 'depth']]

        self.data_augmentation_version = 4
        self.save_checkpoints_after_each_n_epochs = 5

# Old class name: cfgv4_1_17
class cfgv4_0_19(base_cfg):
    def __init__(self, epoch: Optional[int] = None):
        super().__init__(
            epoch,
            experiment_name = 'exp_v4.0.19',
            datasets_set=1, 
            run_type=run_type.rt
        )
        
        self.description = 'DAv2-Base'
        self.accum_iter = 2              # <---------------

        '''Learning rate'''
        self.lr = 1e-5
        self.end_lr = 1e-11
        self.lr_scale = 100 
        
        self.batch_size = 32 # 25 # 32             # <---------------
        self.val_batch_size = 300 # 150 # 300         # <---------------
        self.nepochs = 300
        
        self.decoder_main_tasks = [['rgb', 'depth']]

        self.data_augmentation_version = 2
        self.save_checkpoints_after_each_n_epochs = 5

# Old class name: cfgv4_1_18
class cfgv4_0_21(base_cfg):
    def __init__(self, epoch: Optional[int] = None):
        super().__init__(
            epoch,
            experiment_name = 'exp_v4.0.21',
            datasets_set=1, 
            run_type=run_type.rt
        )

        self.description = 'DAv2-RGB->SOD+Depth'
        self.accum_iter = 2              # <---------------

        '''Learning rate'''
        self.lr = 1e-5
        self.end_lr = 1e-11
        self.lr_scale = 100 
        
        self.batch_size = 32             # <---------------
        self.val_batch_size = 300         # <---------------
        self.nepochs = 200
        
        

        self.inputs = ['rgb']
        self.outputs = ['semseg', 'depth']
        self.decoder_main_tasks = [['rgb'], ['rgb']]
        
        self.data_augmentation_version = 2
        self.save_checkpoints_after_each_n_epochs = 5

# Old class name: cfgv4_1_19
class cfgv4_0_22(base_cfg):
    def __init__(self, epoch: Optional[int] = None):
        super().__init__(
            epoch,
            experiment_name = 'exp_v4.0.22',
            datasets_set=1, 
            run_type=run_type.rt
        )

        self.description = 'DAv2-RGB->SOD'
        self.accum_iter = 2              # <---------------

        '''Learning rate'''
        self.lr = 1e-5
        self.end_lr = 1e-11
        self.lr_scale = 100 
        
        self.batch_size = 32             # <---------------
        self.val_batch_size = 300         # <---------------
        self.nepochs = 200
        
        

        self.inputs = ['rgb']
        self.outputs = ['semseg']
        self.decoder_main_tasks: List[List[str]] = [['rgb']]
        
        self.data_augmentation_version = 2
        self.save_checkpoints_after_each_n_epochs = 5

# Old class name: cfgv4_1_20
class cfgv4_0_23(base_cfg):
    def __init__(self, epoch: Optional[int] = None):
        super().__init__(
            epoch,
            experiment_name = 'exp_v4.0.23',
            datasets_set=1, 
            run_type=run_type.rt
        )
    
        self.description = 'DAv2-Depth->SOD'
        self.accum_iter = 2              # <---------------

        '''Learning rate'''
        self.lr = 1e-5
        self.end_lr = 1e-11
        self.lr_scale = 100 
        
        self.batch_size = 32             # <---------------
        self.val_batch_size = 300         # <---------------
        self.nepochs = 200
        
        

        self.inputs = ['depth']
        self.outputs = ['semseg']
        self.decoder_main_tasks: List[List[str]] = [['depth']]
        
        self.data_augmentation_version = 2
        self.save_checkpoints_after_each_n_epochs = 5

# Old class name: cfgv4_1_21
class cfgv4_0_25(base_cfg):
    def __init__(self, epoch: Optional[int] = None):
        super().__init__(
            epoch,
            experiment_name = 'exp_v4.0.25',
            datasets_set=2, 
            run_type=run_type.rt
        )
        
        self.description = 'DAv2-Base-Set2'
        self.accum_iter = 2              # <---------------

        '''Learning rate'''
        self.lr = 1e-5
        self.end_lr = 1e-11
        self.lr_scale = 100 
        
        self.batch_size = 30             # <---------------
        self.val_batch_size = 300         # <---------------
        self.nepochs = 200
        
        
        self.decoder_main_tasks = [['rgb', 'depth']]

        self.data_augmentation_version = 2
        self.save_checkpoints_after_each_n_epochs = 5

# Old class name: cfgv4_1_22
class cfgv4_0_26(base_cfg):
    def __init__(self, epoch: Optional[int] = None):
        super().__init__(
            epoch,
            experiment_name = 'exp_v4.0.26',
            datasets_set=1, 
            run_type=run_type.rt
        )
        
        self.description = 'DAv2-Large'
        self.accum_iter = 2              # <---------------

        self.image_size = 448
        self.test_image_size = 448
        self.embed_dim = 6144 * 4
        self.input_patch_size = 32
        self.output_patch_size: int = 64

        '''Learning rate'''
        self.lr = 1e-5
        self.end_lr = 1e-11
        self.lr_scale = 100 
        
        self.batch_size = 13 # 13 # 14     # 15       # <---------------
        self.val_batch_size = 70 # 70 # 80 # 100        # <---------------
        self.nepochs = 400
        
        
        self.decoder_main_tasks = [['rgb', 'depth']]

        self.data_augmentation_version = 2
        self.save_checkpoints_after_each_n_epochs = 5

# Old class name: cfgv4_1_24
class cfgv4_0_28(base_cfg):
    def __init__(self, epoch: Optional[int] = None):
        super().__init__(
            epoch,
            experiment_name = 'exp_v4.0.28',
            datasets_set=1, 
            run_type=run_type.rt
        )
        
        self.description = 'DAv2-Base-Decoder[RGB]'
        self.accum_iter = 2              # <---------------

        '''Learning rate'''
        self.lr = 1e-5
        self.end_lr = 1e-11
        self.lr_scale = 100 
        
        self.batch_size = 28 # 25 # 32             # <---------------
        self.val_batch_size = 200 # 150 # 300         # <---------------
        self.nepochs = 200
        
        
        self.decoder_main_tasks = [['rgb']]

        self.data_augmentation_version = 2
        self.save_checkpoints_after_each_n_epochs = 1
        
        self.train_function_version = 2

# Old class name: cfgv4_1_25
class cfgv4_0_29(base_cfg):
    def __init__(self, epoch: Optional[int] = None):
        super().__init__(
            epoch,
            experiment_name = 'exp_v4.0.29',
            datasets_set=1, 
            run_type=run_type.rt
        )
        
        self.description = 'DAv2-Base-Decoder[Depth]'
        self.accum_iter = 2              # <---------------

        '''Learning rate'''
        self.lr = 1e-5
        self.end_lr = 1e-11
        self.lr_scale = 100 
        
        self.batch_size = 28             # <---------------
        self.val_batch_size = 200         # <---------------
        self.nepochs = 200
        
        
        self.decoder_main_tasks = [['depth']]

        self.data_augmentation_version = 2
        self.save_checkpoints_after_each_n_epochs = 1
        
        self.train_function_version = 2

# Old class name: cfgv4_1_26
class cfgv4_0_30(base_cfg):
    def __init__(self, epoch: Optional[int] = None):
        super().__init__(
            epoch,
            experiment_name = 'exp_v4.0.30',
            datasets_set=1, 
            run_type=run_type.rt
        )
        
        self.description = 'DAv2-Base-GradAccum[4]'
        self.accum_iter = 4             # <---------------

        '''Learning rate'''
        self.lr = 1e-5
        self.end_lr = 1e-11
        self.lr_scale = 100 
        
        self.batch_size = 32 # 25 # 32             # <---------------
        self.val_batch_size = 300 # 150 # 300         # <---------------
        self.nepochs = 1000
        
        
        self.decoder_main_tasks = [['rgb', 'depth']]

        self.data_augmentation_version = 2
        self.save_checkpoints_after_each_n_epochs = 5

# Old class name: cfgv4_1_27
class cfgv4_0_31(base_cfg):
    def __init__(self, epoch: Optional[int] = None):
        super().__init__(
            epoch,
            experiment_name = 'exp_v4.0.31',
            datasets_set=1, 
            run_type=run_type.rt
        )
        
        self.description = 'DAv2-Base-NoPretrainedBB'

        '''Learning rate'''
        self.lr = 1e-5
        self.end_lr = 1e-11
        self.lr_scale = 100 
        
        '''Pretrained Backbone'''
        self.pretrained_backbone = None
        
        self.batch_size = 28 # 25 # 32             # <---------------
        self.val_batch_size = 200 # 150 # 300         # <---------------
        self.nepochs = 1000
        
        
        self.decoder_main_tasks = [['rgb', 'depth']]

        self.data_augmentation_version = 2
        self.save_checkpoints_after_each_n_epochs = 2
        
        self.train_function_version = 2

# Old class name: cfgv4_1_28
class cfgv4_0_32(base_cfg):
    def __init__(self, epoch: Optional[int] = None):
        super().__init__(
            epoch,
            experiment_name = 'exp_v4.0.32',
            datasets_set=1, 
            run_type=run_type.rt
        )
        
        self.description = 'DAv2-Base-LargerBatchSize'

        '''Learning rate'''
        self.lr = 1e-5
        self.end_lr = 1e-11
        self.lr_scale = 100 
        
        self.batch_size = 90 # 25 # 32             # <---------------
        self.val_batch_size = 400 # 150 # 300         # <---------------
        self.nepochs = 1000
        
        
        self.decoder_main_tasks = [['rgb', 'depth']]

        self.data_augmentation_version = 2
        self.save_checkpoints_after_each_n_epochs = 1

# Old class name: cfgv4_1_29
class cfgv4_0_33(base_cfg):
    def __init__(self, epoch: Optional[int] = None):
        super().__init__(
            epoch,
            experiment_name = 'exp_v4.0.33',
            datasets_set=3, 
            run_type=run_type.rt
        )
        
        self.description = 'DAv2-Base-Kaggle2GPUs-Set3'

        '''Learning rate'''
        self.lr = 1e-5
        self.end_lr = 1e-11
        self.lr_scale = 100 
        
        self.batch_size = 28             # <---------------
        self.val_batch_size = 200         # <---------------
        self.nepochs = 1000
        
        
        self.decoder_main_tasks = [['rgb', 'depth']]

        self.data_augmentation_version = 2
        self.save_checkpoints_after_each_n_epochs = 1
        
        self.train_function_version = 2

class cfgv4_0_34(base_cfg):
    def __init__(self, epoch: Optional[int] = None):
        super().__init__(
            epoch,
            experiment_name = 'exp_v4.0.34',
            datasets_set=1, 
            run_type=run_type.rt
        )
        
        self.description = 'DAv2-Base-ViTBB'

        '''Learning rate'''
        self.lr = 1e-5
        self.end_lr = 1e-11
        self.lr_scale = 100 
        
        '''Pretrained Backbone'''
        self.pretrained_backbone = 'vit'
        
        self.batch_size = 28 # 25 # 32             # <---------------
        self.val_batch_size = 200 # 150 # 300         # <---------------
        self.nepochs = 1000
        
        
        self.decoder_main_tasks = [['rgb', 'depth']]

        self.data_augmentation_version = 2
        self.save_checkpoints_after_each_n_epochs = 2
        
        self.train_function_version = 2

class cfgv4_0_35(base_cfg):
    def __init__(self, epoch: Optional[int] = None):
        super().__init__(
            epoch,
            experiment_name = 'exp_v4.0.35',
            datasets_set=1, 
            run_type=run_type.rt
        )
        
        self.description = 'DAv2-Base-MAEBB'

        '''Learning rate'''
        self.lr = 1e-5
        self.end_lr = 1e-11
        self.lr_scale = 100 
        
        '''Pretrained Backbone'''
        self.pretrained_backbone = 'mae'
        
        self.batch_size = 28 # 25 # 32             # <---------------
        self.val_batch_size = 200 # 150 # 300         # <---------------
        self.nepochs = 1000
        
        
        self.decoder_main_tasks = [['rgb', 'depth']]

        self.data_augmentation_version = 2
        self.save_checkpoints_after_each_n_epochs = 2
        
        self.train_function_version = 2

class cfgv4_0_36(base_cfg):
    def __init__(self, epoch: Optional[int] = None):
        super().__init__(
            epoch,
            experiment_name = 'exp_v4.0.36',
            datasets_set=1, 
            run_type=run_type.rt
        )
        
        self.description = 'DAv2-Base'

        '''Learning rate'''
        self.lr = 1e-5
        self.end_lr = 1e-11
        self.lr_scale = 100 
        
        self.batch_size = 28 # 25 # 32             # <---------------
        self.val_batch_size = 300 # 150 # 300         # <---------------
        self.nepochs = 300
        
        
        self.decoder_main_tasks = [['rgb', 'depth']]

        self.data_augmentation_version = 2
        self.save_checkpoints_after_each_n_epochs = 1
        self.train_function_version = 2
        
class cfgv4_0_37(base_cfg):
    def __init__(self, epoch: Optional[int] = None):
        super().__init__(
            epoch,
            experiment_name = 'exp_v4.0.37',
            datasets_set=4, 
            run_type=run_type.rt
        )
        
        self.description = 'DAv2-Base-Set4'

        '''Learning rate'''
        self.lr = 1e-5
        self.end_lr = 1e-11
        self.lr_scale = 100 
        
        self.batch_size = 30 # 25 # 32             # <---------------
        self.val_batch_size = 300 # 150 # 300         # <---------------
        self.nepochs = 300
        
        
        self.decoder_main_tasks = [['rgb', 'depth']]

        self.data_augmentation_version = 2
        self.save_checkpoints_after_each_n_epochs = 1
        self.train_function_version = 2

class cfgv4_0_38(base_cfg):
    def __init__(self, epoch: Optional[int] = None):
        super().__init__(
            epoch,
            experiment_name = 'exp_v4.0.38',
            datasets_set=1, 
            run_type=run_type.rt
        )
        
        self.description = 'DAv4-Base'

        '''Learning rate'''
        self.lr = 1e-5
        self.end_lr = 1e-11
        self.lr_scale = 100 
        
        self.batch_size = 30 # 25 # 32             # <---------------
        self.val_batch_size = 300 # 150 # 300         # <---------------
        self.nepochs = 300
        
        
        self.decoder_main_tasks = [['rgb', 'depth']]

        self.data_augmentation_version = 4
        self.save_checkpoints_after_each_n_epochs = 1
        self.train_function_version = 2

class cfgv4_0_39(base_cfg):
    def __init__(self, epoch: Optional[int] = None):
        super().__init__(
            epoch,
            experiment_name = 'exp_v4.0.39',
            datasets_set=2, 
            run_type=run_type.rt
        )
        
        self.description = 'DAv2-Base-Set2'

        '''Learning rate'''
        self.lr = 1e-5
        self.end_lr = 1e-11
        self.lr_scale = 100 
        
        self.batch_size = 30 # 25 # 32             # <---------------
        self.val_batch_size = 300 # 150 # 300         # <---------------
        self.nepochs = 200
        
        
        self.decoder_main_tasks = [['rgb', 'depth']]

        self.data_augmentation_version = 2
        self.save_checkpoints_after_each_n_epochs = 1
        self.train_function_version = 2

# Deprecated, suspended
class cfgv4_0_40(base_cfg):
    def __init__(self, epoch: Optional[int] = None):
        super().__init__(
            epoch,
            experiment_name = 'exp_v4.0.40',
            datasets_set=1, 
            run_type=run_type.rt
        )
        
        self.description = 'DAv2-Base-LabelSmoothing'

        '''Learning rate'''
        self.lr = 1e-5
        self.end_lr = 1e-11
        self.lr_scale = 100 
        
        self.batch_size = 30 # 25 # 32             # <---------------
        self.val_batch_size = 300 # 150 # 300         # <---------------
        self.nepochs = 100
        self.label_smoothing = 0.1
        
        
        self.decoder_main_tasks = [['rgb', 'depth']]

        self.data_augmentation_version = 2
        self.save_checkpoints_after_each_n_epochs = 1
        self.train_function_version = 2

# Deprecated, suspended
class cfgv4_0_41(base_cfg):
    def __init__(self, epoch: Optional[int] = None):
        super().__init__(
            epoch,
            experiment_name = 'exp_v4.0.41',
            datasets_set=1, 
            run_type=run_type.rt
        )
        
        self.description = 'DAv2-Base-LabelSmoothing'

        '''Learning rate'''
        self.lr = 1e-5
        self.end_lr = 1e-11
        self.lr_scale = 100 
        
        self.batch_size = 30 # 25 # 32             # <---------------
        self.val_batch_size = 300 # 150 # 300         # <---------------
        self.nepochs = 100
        self.label_smoothing = 0.1
        
        
        self.decoder_main_tasks = [['rgb', 'depth']]

        self.data_augmentation_version = 2
        self.save_checkpoints_after_each_n_epochs = 1
        self.train_function_version = 2
        
class cfgv4_0_42(base_cfg):
    def __init__(self, epoch: Optional[int] = None):
        super().__init__(
            epoch,
            experiment_name = 'exp_v4.0.42',
            datasets_set=1, 
            run_type=run_type.rt
        )
        
        self.description = 'DAv2-Base-LabelSmoothing-WarmUpBatchSize'

        '''Learning rate'''
        self.lr = 1e-5
        self.end_lr = 1e-11
        self.lr_scale = 100
        
        self.warmup_min_batch_size = 5
        self.warmup_epoch_batch_size = 6
        
        self.batch_size = 30 # 25 # 32             # <---------------
        self.val_batch_size = 300 # 150 # 300         # <---------------
        self.nepochs = 100
        self.label_smoothing = 0.1
        
        
        self.decoder_main_tasks = [['rgb', 'depth']]

        self.data_augmentation_version = 2
        self.save_checkpoints_after_each_n_epochs = 1
        self.train_function_version = 2
        
class cfgv4_0_43(base_cfg):
    def __init__(self, epoch: Optional[int] = None):
        super().__init__(
            epoch,
            experiment_name = 'exp_v4.0.43',
            datasets_set=1, 
            run_type=run_type.rt
        )
        
        self.description = 'DAv2-Base-v2'

        '''Learning rate'''
        self.lr = 1e-5
        self.end_lr = 1e-11
        self.lr_scale = 100
        
        self.batch_size = 28 # 25 # 32             # <---------------
        self.val_batch_size = 300 # 150 # 300         # <---------------
        self.nepochs = 100
        
        
        self.decoder_main_tasks = [['rgb', 'depth']]

        self.data_augmentation_version = 2
        self.save_checkpoints_after_each_n_epochs = 1
        self.train_function_version = 2

class cfgv4_0_44(base_cfg):
    def __init__(self, epoch: Optional[int] = None):
        super().__init__(
            epoch,
            experiment_name = 'exp_v4.0.44',
            datasets_set=1, 
            run_type=run_type.rt
        )
        
        self.description = 'DAv2-Base-v2-Ubuntu'
        self.accum_iter = 3             # <---------------

        '''Learning rate'''
        self.lr = 1e-5
        self.end_lr = 1e-11
        self.lr_scale = 100
        
        self.num_workers = 8
        self.batch_size = 10 # 25 # 32             # <---------------
        self.val_batch_size = 100 # 150 # 300         # <---------------
        self.nepochs = 100
        
        
        self.decoder_main_tasks = [['rgb', 'depth']]

        self.data_augmentation_version = 2
        self.save_checkpoints_after_each_n_epochs = 1
        self.train_function_version = 2

class cfgv4_0_45(base_cfg):
    def __init__(self, epoch: Optional[int] = None):
        super().__init__(
            epoch,
            experiment_name = 'exp_v4.0.45',
            datasets_set=1, 
            run_type=run_type.rt
        )
        
        self.description = 'DAv2-Base-NoPretrainedBB-v2'

        '''Learning rate'''
        self.lr = 1e-4
        self.end_lr = 1e-11
        self.lr_scale = 100 
        
        '''Pretrained Backbone'''
        self.pretrained_backbone = None
        
        self.batch_size = 30             # <---------------
        self.val_batch_size = 300         # <---------------
        self.nepochs = 100
        
        
        self.decoder_main_tasks = [['rgb', 'depth']]

        self.data_augmentation_version = 2
        self.save_checkpoints_after_each_n_epochs = 1
        self.train_function_version = 2
    
class cfgv4_0_46(base_cfg):
    def __init__(self, epoch: Optional[int] = None):
        super().__init__(
            epoch,
            experiment_name = 'exp_v4.0.46',
            datasets_set=1, 
            run_type=run_type.rt
        )
        
        self.description = 'DAv2-Base-MAEBB-v2'

        '''Learning rate'''
        self.lr = 1e-5
        self.end_lr = 1e-11
        self.lr_scale = 100 
        
        '''Pretrained Backbone'''
        self.pretrained_backbone = 'mae'
        
        self.batch_size = 30          # <---------------
        self.val_batch_size = 300         # <---------------
        self.nepochs = 100
        
        
        self.decoder_main_tasks = [['rgb', 'depth']]

        self.data_augmentation_version = 2
        self.save_checkpoints_after_each_n_epochs = 1
        self.train_function_version = 2 
        
class cfgv4_0_47(base_cfg):
    def __init__(self, epoch: Optional[int] = None):
        super().__init__(
            epoch,
            experiment_name = 'exp_v4.0.47',
            datasets_set=1, 
            run_type=run_type.rt
        )
        
        self.description = 'DAv2-Base-ViTBB-v2'

        '''Learning rate'''
        self.lr = 1e-5
        self.end_lr = 1e-11
        self.lr_scale = 100 
        
        '''Pretrained Backbone'''
        self.pretrained_backbone = 'vit'
        
        self.batch_size = 30          # <---------------
        self.val_batch_size = 300         # <---------------
        self.nepochs = 100
        
        
        self.decoder_main_tasks = [['rgb', 'depth']]

        self.data_augmentation_version = 2
        self.save_checkpoints_after_each_n_epochs = 1
        self.train_function_version = 2 

class cfgv4_0_48(base_cfg):
    def __init__(self, epoch: Optional[int] = None):
        super().__init__(
            epoch,
            experiment_name = 'exp_v4.0.48',
            datasets_set=2, 
            run_type=run_type.rt
        )
        
        self.description = 'DAv2-Base-Set2'

        '''Learning rate'''
        self.lr = 1e-5
        self.end_lr = 1e-11
        self.lr_scale = 100 
        
        self.batch_size = 28 # 25 # 32             # <---------------
        self.val_batch_size = 300 # 150 # 300         # <---------------
        self.nepochs = 100
        
        
        self.decoder_main_tasks = [['rgb', 'depth']]

        self.data_augmentation_version = 2
        self.save_checkpoints_after_each_n_epochs = 1
        self.train_function_version = 2
        
class cfgv4_0_49(base_cfg):
    def __init__(self, epoch: Optional[int] = None):
        super().__init__(
            epoch,
            experiment_name = 'exp_v4.0.49',
            datasets_set=4, 
            run_type=run_type.rt
        )
        
        self.description = 'DAv2-Base-Set4'

        '''Learning rate'''
        self.lr = 1e-5
        self.end_lr = 1e-11
        self.lr_scale = 100 
        
        self.batch_size = 28 # 25 # 32             # <---------------
        self.val_batch_size = 300 # 150 # 300         # <---------------
        self.nepochs = 100
        
        
        self.decoder_main_tasks = [['rgb', 'depth']]

        self.data_augmentation_version = 2
        self.save_checkpoints_after_each_n_epochs = 1
        self.train_function_version = 2

class cfgv4_0_50(base_cfg):
    def __init__(self, epoch: Optional[int] = None):
        super().__init__(
            epoch,
            experiment_name = 'exp_v4.0.50',
            datasets_set=1, 
            run_type=run_type.rt
        )
        
        self.description = 'DAv2-Base-EarlyStopping'

        '''Learning rate'''
        self.lr = 1e-5
        self.end_lr = 1e-11
        self.lr_scale = 100 
        
        self.batch_size = 30 # 25 # 32             # <---------------
        self.val_batch_size = 300 # 150 # 300         # <---------------
        self.nepochs = 25
        
        
        self.decoder_main_tasks = [['rgb', 'depth']]

        self.data_augmentation_version = 2
        self.save_checkpoints_after_each_n_epochs = 1
        self.train_function_version = 2

class cfgv4_0_51(base_cfg):
    def __init__(self, epoch: Optional[int] = None):
        super().__init__(
            epoch,
            experiment_name = 'exp_v4.0.51',
            datasets_set=1, 
            run_type=run_type.rt
        )
        
        self.description = 'DAv5-Base'

        '''Learning rate'''
        self.lr = 1e-5
        self.end_lr = 1e-11
        self.lr_scale = 100
        
        self.batch_size = 28 # 25 # 32             # <---------------
        self.val_batch_size = 300 # 150 # 300         # <---------------
        self.nepochs = 100
        
        self.data_augmentation_version = 5
        self.save_checkpoints_after_each_n_epochs = 1
        self.train_function_version = 2

class cfgv4_0_52(base_cfg):
    def __init__(self, epoch: Optional[int] = None):
        super().__init__(
            epoch,
            experiment_name = 'exp_v4.0.52',
            datasets_set=5, 
            run_type=run_type.rt
        )
        
        self.description = 'DAv5-Base-Set5'

        '''Learning rate'''
        self.lr = 1e-5
        self.end_lr = 1e-11
        self.lr_scale = 100
        
        self.batch_size = 28 # 25 # 32             # <---------------
        self.val_batch_size = 300 # 150 # 300         # <---------------
        self.nepochs = 100
        
        self.data_augmentation_version = 5
        self.save_checkpoints_after_each_n_epochs = 1
        self.train_function_version = 2
        
class cfgv4_0_53(base_cfg):
    def __init__(self, epoch: Optional[int] = None):
        super().__init__(
            epoch,
            experiment_name = 'exp_v4.0.53',
            datasets_set=1, 
            run_type=run_type.rt
        )
        
        self.description = 'DAv5-Base-DecoderDepth6'

        '''Learning rate'''
        self.lr = 1e-5
        self.end_lr = 1e-11
        self.lr_scale = 100
        
        self.batch_size = 28 # 25 # 32             # <---------------
        self.val_batch_size = 300 # 150 # 300         # <---------------
        self.nepochs = 50
        self.decoder_depth = 6
        
        self.data_augmentation_version = 5
        self.save_checkpoints_after_each_n_epochs = 1
        self.train_function_version = 2
        
class cfgv4_0_54(base_cfg):
    def __init__(self, epoch: Optional[int] = None):
        super().__init__(
            epoch,
            experiment_name = 'exp_v4.0.54',
            datasets_set=1, 
            run_type=run_type.rt
        )
        
        self.description = 'DAv5-Base'

        '''Learning rate'''
        self.lr = 1e-5
        self.end_lr = 1e-11
        self.lr_scale = 100
        
        self.batch_size = 28 # 25 # 32             # <---------------
        self.val_batch_size = 300 # 150 # 300         # <---------------
        self.nepochs = 100
        self.decoder_main_tasks = [['rgb', 'depth']]
        
        self.data_augmentation_version = 5
        self.save_checkpoints_after_each_n_epochs = 1
        self.train_function_version = 2
        
class cfgv4_0_55(base_cfg):
    def __init__(self, epoch: Optional[int] = None):
        super().__init__(
            epoch,
            experiment_name = 'exp_v4.0.55',
            datasets_set=1, 
            run_type=run_type.rt
        )
        
        self.description = 'DAv5-Base-MAE'

        '''Learning rate'''
        self.lr = 1e-5
        self.end_lr = 1e-11
        self.lr_scale = 100
        
        self.batch_size = 30 # 25 # 32             # <---------------
        self.val_batch_size = 300 # 150 # 300         # <---------------
        self.nepochs = 50
        self.pretrained_backbone = 'mae'
        self.decoder_main_tasks = [['rgb', 'depth']]
        
        self.data_augmentation_version = 5
        self.save_checkpoints_after_each_n_epochs = 1
        self.train_function_version = 2

class cfgv4_0_56(base_cfg):
    def __init__(self, epoch: Optional[int] = None):
        super().__init__(
            epoch,
            experiment_name = 'exp_v4.0.56',
            datasets_set=6,
            run_type=run_type.rt
        )
        
        self.description = 'DAv5-Base-Set6'

        '''Learning rate'''
        self.lr = 1e-5
        self.end_lr = 1e-11
        self.lr_scale = 100
        
        self.batch_size = 28 # 25 # 32             # <---------------
        self.val_batch_size = 300 # 150 # 300         # <---------------
        self.nepochs = 200
        
        self.data_augmentation_version = 5
        self.save_checkpoints_after_each_n_epochs = 1
        self.train_function_version = 2

class cfgv4_0_57(base_cfg):
    def __init__(self, epoch: Optional[int] = None):
        super().__init__(
            epoch,
            experiment_name = 'exp_v4.0.57',
            datasets_set=1,
            run_type=run_type.rt
        )
        
        self.description = 'DAv5-Large-Set1'
        
        '''MultiMAE'''
        self.dim_tokens = 1024
        self.encoder_depth = 24
        self.num_heads = 16
        self.pretrained_backbone = 'large-mae'

        '''Learning rate'''
        self.lr = 1e-5
        self.end_lr = 1e-11
        self.lr_scale = 100
        
        self.batch_size = 8         # <---------------
        self.val_batch_size = 80    # <---------------
        self.nepochs = 50
        
        self.data_augmentation_version = 5
        self.save_checkpoints_after_each_n_epochs = 1
        self.train_function_version = 2
        
class cfgv4_0_58(base_cfg):
    def __init__(self, epoch: Optional[int] = None):
        super().__init__(
            epoch,
            experiment_name = 'exp_v4.0.58',
            datasets_set=1,
            run_type=run_type.rt
        )
        
        self.description = 'DAv5-Set1-DecDepth8'
        
        '''MultiMAE'''
        self.decoder_depth = 8

        '''Learning rate'''
        self.lr = 1e-5
        self.end_lr = 1e-11
        self.lr_scale = 100
        
        self.batch_size = 22            # <---------------
        self.val_batch_size = 220       # <---------------
        self.nepochs = 50
        
        self.data_augmentation_version = 5
        self.save_checkpoints_after_each_n_epochs = 1
        self.train_function_version = 2

class cfgv4_0_59(base_cfg):
    def __init__(self, epoch: Optional[int] = None):
        super().__init__(
            epoch,
            experiment_name = 'exp_v4.0.59',
            datasets_set=1,
            run_type=run_type.rt
        )
        
        self.description = 'DAv2-Large-Set1'
        
        '''MultiMAE'''
        self.dim_tokens = 1024
        self.encoder_depth = 24
        self.num_heads = 16
        self.pretrained_backbone = 'large-mae'

        '''Learning rate'''
        self.lr = 1e-5
        self.end_lr = 1e-11
        self.lr_scale = 100
        
        self.batch_size = 8         # <---------------
        self.val_batch_size = 80    # <---------------
        self.nepochs = 50
        
        self.data_augmentation_version = 2
        self.save_checkpoints_after_each_n_epochs = 1
        self.train_function_version = 2
        
class cfgv4_0_61(base_cfg):
    def __init__(self, epoch: Optional[int] = None):
        super().__init__(
            epoch,
            experiment_name = 'exp_v4.0.61',
            datasets_set=1,
            run_type=run_type.rt
        )
        
        self.description = 'DAv2-Huge-Set1'
        
        '''MultiMAE'''
        self.dim_tokens = 1280
        self.encoder_depth = 32
        self.num_heads = 16
        self.pretrained_backbone = 'huge-mae'
        self.input_patch_size = 14
        self.output_patch_size = 16
        self.embed_dim = 6144
        self.freeze_encoder = True
        
        '''Learning rate'''
        self.lr = 1e-5
        self.end_lr = 1e-11
        self.lr_scale = 100
        
        self.batch_size = 7         # <---------------
        self.val_batch_size = 15    # <---------------
        self.nepochs = 50
        
        self.data_augmentation_version = 2
        self.save_checkpoints_after_each_n_epochs = 1
        self.train_function_version = 2
        
class cfgv4_0_62(base_cfg):
    def __init__(self, epoch: Optional[int] = None):
        super().__init__(
            epoch,
            experiment_name = 'exp_v4.0.62',
            datasets_set=1,
            run_type=run_type.rt
        )
        
        self.description = 'DAv2-Huge-Set1-GradClip'
        
        '''MultiMAE'''
        self.dim_tokens = 1280
        self.encoder_depth = 32
        self.num_heads = 16
        self.pretrained_backbone = 'huge-mae'
        self.input_patch_size = 14
        self.output_patch_size = 16
        self.embed_dim = 6144
        self.freeze_encoder = True
        
        self.clip_grad = 1.0

        '''Learning rate'''
        self.lr = 1e-5
        self.end_lr = 1e-11
        self.lr_scale = 100
        
        self.batch_size = 7         # <---------------
        self.val_batch_size = 15    # <---------------
        self.nepochs = 50
        
        self.data_augmentation_version = 2
        self.save_checkpoints_after_each_n_epochs = 1
        self.train_function_version = 2

class cfgv4_0_64(base_cfg):
    def __init__(self, epoch: Optional[int] = None):
        super().__init__(
            epoch,
            experiment_name = 'exp_v4.0.64',
            datasets_set=1,
            run_type=run_type.rt
        )
        
        self.description = 'DAv2-Large-Set1-RGB->SOD'
        
        '''MultiMAE'''
        self.dim_tokens = 1024
        self.encoder_depth = 24
        self.num_heads = 16
        self.pretrained_backbone = 'large-mae'

        '''Learning rate'''
        self.lr = 1e-5
        self.end_lr = 1e-11
        self.lr_scale = 100
        
        self.batch_size = 15        # <---------------
        self.val_batch_size = 45    # <---------------
        self.nepochs = 50
        
        self.inputs = ['rgb']
        self.outputs = ['semseg']
        self.decoder_main_tasks: List[List[str]] = [['rgb']]
        
        self.data_augmentation_version = 2
        self.save_checkpoints_after_each_n_epochs = 1
        self.train_function_version = 2

class cfgv4_0_65(base_cfg):
    def __init__(self, epoch: Optional[int] = None):
        super().__init__(
            epoch,
            experiment_name = 'exp_v4.0.65',
            datasets_set=1,
            run_type=run_type.rt
        )
        
        self.description = 'DAv2-Large-Set1-GradClip'
        
        '''MultiMAE'''
        self.dim_tokens = 1024
        self.encoder_depth = 24
        self.num_heads = 16
        self.pretrained_backbone = 'large-mae'
        
        self.clip_grad = 1.0

        '''Learning rate'''
        self.lr = 1e-5
        self.end_lr = 1e-11
        self.lr_scale = 100
        
        self.batch_size = 8         # <---------------
        self.val_batch_size = 80    # <---------------
        self.nepochs = 50
        
        self.data_augmentation_version = 2
        self.save_checkpoints_after_each_n_epochs = 1
        self.train_function_version = 2

class cfgv4_0_66(base_cfg):
    def __init__(self, epoch: Optional[int] = None):
        super().__init__(
            epoch,
            experiment_name = 'exp_v4.0.66',
            datasets_set=1,
            run_type=run_type.rt
        )
        
        self.description = 'DAv2-Large-Set6-GradClip'
        
        '''MultiMAE'''
        self.dim_tokens = 1024
        self.encoder_depth = 24
        self.num_heads = 16
        self.pretrained_backbone = 'large-mae'
        
        self.clip_grad = 1.0

        '''Learning rate'''
        self.lr = 1e-5
        self.end_lr = 1e-11
        self.lr_scale = 100
        
        self.batch_size = 8         # <---------------
        self.val_batch_size = 80    # <---------------
        self.nepochs = 50
        
        self.data_augmentation_version = 2
        self.save_checkpoints_after_each_n_epochs = 1
        self.train_function_version = 2

class cfgv4_0_67(base_cfg):
    def __init__(self, epoch: Optional[int] = None):
        super().__init__(
            epoch,
            experiment_name = 'exp_v4.0.67',
            datasets_set=7,
            run_type=run_type.rt
        )
        
        self.description = 'DAv2-Large-Set7-GradClip'
        
        '''MultiMAE'''
        self.dim_tokens = 1024
        self.encoder_depth = 24
        self.num_heads = 16
        self.pretrained_backbone = 'large-mae'
        
        self.clip_grad = 1.0

        '''Learning rate'''
        self.lr = 1e-5
        self.end_lr = 1e-11
        self.lr_scale = 100
        
        self.batch_size = 8         # <---------------
        self.val_batch_size = 80    # <---------------
        self.nepochs = 50
        
        self.data_augmentation_version = 2
        self.save_checkpoints_after_each_n_epochs = 1
        self.train_function_version = 2

class cfgv4_0_68(base_cfg):
    def __init__(self, epoch: Optional[int] = None):
        super().__init__(
            epoch,
            experiment_name = 'exp_v4.0.68',
            datasets_set=1, # 7
            run_type=run_type.rt
        )
        
        self.description = 'DAv5-Large-Set7-GradClip'
        
        '''MultiMAE'''
        self.dim_tokens = 1024
        self.encoder_depth = 24
        self.num_heads = 16
        self.pretrained_backbone = 'large-mae'
        
        self.clip_grad = 1.0

        '''Learning rate'''
        self.lr = 1e-5
        self.end_lr = 1e-11
        self.lr_scale = 100
        
        self.batch_size = 8         # <---------------
        self.val_batch_size = 80    # <---------------
        self.nepochs = 50
        
        self.data_augmentation_version = 5
        self.save_checkpoints_after_each_n_epochs = 1
        self.train_function_version = 2


arg_cfg: Dict[str, base_cfg] = dict(
    cfgv4_0_18=cfgv4_0_18,
    cfgv4_0_19=cfgv4_0_19,
    cfgv4_0_21=cfgv4_0_21,
    cfgv4_0_22=cfgv4_0_22,
    cfgv4_0_23=cfgv4_0_23,
    cfgv4_0_25=cfgv4_0_25,
    cfgv4_0_26=cfgv4_0_26,
    cfgv4_0_28=cfgv4_0_28,
    cfgv4_0_29=cfgv4_0_29,
    cfgv4_0_30=cfgv4_0_30,
    cfgv4_0_31=cfgv4_0_31,
    cfgv4_0_32=cfgv4_0_32,
    cfgv4_0_33=cfgv4_0_33,
    cfgv4_0_34=cfgv4_0_34,
    cfgv4_0_35=cfgv4_0_35,
    cfgv4_0_36=cfgv4_0_36,
    cfgv4_0_37=cfgv4_0_37,
    cfgv4_0_38=cfgv4_0_38,
    cfgv4_0_39=cfgv4_0_39,
    # cfgv4_0_40=cfgv4_0_40,
    # cfgv4_0_41=cfgv4_0_41,
    cfgv4_0_42=cfgv4_0_42,
    cfgv4_0_43=cfgv4_0_43,
    cfgv4_0_44=cfgv4_0_44,
    cfgv4_0_45=cfgv4_0_45,
    cfgv4_0_46=cfgv4_0_46,
    cfgv4_0_47=cfgv4_0_47,
    cfgv4_0_48=cfgv4_0_48,
    cfgv4_0_49=cfgv4_0_49,
    cfgv4_0_50=cfgv4_0_50,
    
    cfgv4_0_51=cfgv4_0_51, # suspended
    cfgv4_0_52=cfgv4_0_52, # suspended 
    cfgv4_0_53=cfgv4_0_53, # suspended
    
    cfgv4_0_54=cfgv4_0_54,
    cfgv4_0_55=cfgv4_0_55,
    cfgv4_0_56=cfgv4_0_56,
    cfgv4_0_57=cfgv4_0_57,
    cfgv4_0_58=cfgv4_0_58,
    cfgv4_0_59=cfgv4_0_59,
    cfgv4_0_61=cfgv4_0_61,
    cfgv4_0_62=cfgv4_0_62,
    cfgv4_0_64=cfgv4_0_64,
    cfgv4_0_65=cfgv4_0_65,
    cfgv4_0_66=cfgv4_0_66,
    cfgv4_0_67=cfgv4_0_67,
    cfgv4_0_68=cfgv4_0_68,
    
    cfg_set_1=cfg_set_1,
    cfg_set_2=cfg_set_2,
    cfg_set_3=cfg_set_3,
    cfg_set_4=cfg_set_4,
    
    cfg_debug_ubuntu=cfg_debug_ubuntu,
    cfg_debug_colab=cfg_debug_colab,
    cfg_debug_kaggle=cfg_debug_kaggle,
)

configs_dict = dict(
    cfgv4_0_35_epoch136=partial(cfgv4_0_35, epoch=136),
    cfgv4_0_19_epoch175=partial(cfgv4_0_19, epoch=175),
    cfgv4_0_19_epoch285=partial(cfgv4_0_19, epoch=285),
    cfgv4_0_18_epoch180=partial(cfgv4_0_18, epoch=180),
    cfgv4_0_59_epoch49=partial(cfgv4_0_59, epoch=46),
    cfgv4_0_64_epoch36=partial(cfgv4_0_64, epoch=36),
    cfgv4_0_68_epoch10=partial(cfgv4_0_68, epoch=10),
)

def get_config_by_set_version(set_version: int) -> base_cfg:
    if set_version not in [1,2,3,4]:
        raise Exception(f'Unsupported set version {set_version}')
    return arg_cfg[f'cfg_set_{set_version}']()

def get_config(cfg_name: str, epoch: Optional[int] = None) -> base_cfg:
    return arg_cfg[cfg_name](epoch)
