from functools import partial
import os
from typing import Dict, Optional, Tuple, List
import torch
from torch import nn

from ..experiment_manager import ExperimentManager
from ..run_type import RUN_TYPE, run_type
from ..utils import get_production_ckpt_path

from .data_augmentation_config import DataAugmentationConfig

class base_cfg:
    def __init__(
        self,
        epoch: int,
        experiment_name: str, 
        datasets_set: int,
        run_type: str,
    ):
        self.experiment_name = experiment_name
        self.datasets_set = datasets_set
        self.run_type = run_type
        if run_type == RUN_TYPE.KAGGLE:
            self.base_working_dir = '/kaggle/working'
            self.base_datasets_working_dir_path = '/kaggle/input'

            self.datasets_working_dir_path = os.path.join(
                self.base_datasets_working_dir_path, f'rgbdsod-set{datasets_set}'
            )

            '''Source code'''
            self.source_code_dir = os.path.join('/', 'kaggle', 'working', 'sources')

            '''Benchmark'''
            self.sotas_root_working_dir = '/home/sotas'
        elif run_type == RUN_TYPE.UBUNTU:
            self.base_working_dir = '.'
            self.base_datasets_working_dir_path = './datasets'

            self.datasets_working_dir_path = os.path.join(
                self.base_datasets_working_dir_path, f'v{datasets_set}'
            )

            '''Source code'''
            self.source_code_dir = './sources'

            '''Benchmark'''
            self.sotas_root_working_dir = './sotas'
        elif run_type in [RUN_TYPE.COLAB, RUN_TYPE.HUGGINGFACE]:
            self.base_working_dir = '/content'
            self.mount_path = '/content/drive' # GoogleDrive
            self.datasets_dir_path = os.path.join(
                self.mount_path, 'MyDrive', 'RGBD_SOD', 'datasets'
            ) # GoogleDrive
            self.base_datasets_working_dir_path = '/content/datasets' # Colab
            self.datasets_working_dir_path = os.path.join(
                self.base_datasets_working_dir_path, f'v{datasets_set}'
            ) # Colab

            '''Source code'''
            self.source_code_dir = os.path.join(
                self.mount_path, 'MyDrive', 'RGBD_SOD', 'sources'
            )

            '''Benchmark'''
            self.sotas_root_working_dir = '/content/sotas'
        else:
            raise Exception(f'Unsupported run type {run_type}')
            
        self.sotas_working_dir = os.path.join(self.sotas_root_working_dir, f'v{datasets_set}')

        if self.datasets_set == 1:
            '''Set 1: COME15K
            Train dataset contains 8,025 image pairs of RGB-D
            We split our testing     dataset to a moderate-level testing set ("Normal") and a
            hard testing set ("Difficult") with 4,600 image pairs and
            3,000 pairs respectively'''
            self.test_dataset_names = [
                'COME-E', 'COME-H',
                'DES', 'DUT-RGBD', 'LFSD', 'NJU2K', 'NLPR',
                'ReDWeb-S', 'SIP', 'STERE'
            ]
            self.test_dataset_quantities = [
                4600, 3000, 
                135, 400, 100, 500, 300,
                1000, 929, 1000
            ]
            self.sota_model_names = [
                'A2Dele',
                'ATSA',
                'BBS-Net', 
                # 'BTSNet',
                'CDNet',
                
                'CMINet',
                'CoNet',
                'DANet',
                'DCF',
                'DSA2F',

                # 'EFNet',
                'FRDT',
                'HAINet',
                'JLDCF',
                'PGAR',

                'RD3D',
                'S2MA',
                'SSF',
                'UCNet'
            ]
            self.sotas_datasets = [self.test_dataset_names for _ in range(len(self.sota_model_names))]
            self.mapping_test_dataset_names: List[Tuple[str, str]] = [
                ('COME-E', 'cascaded_rgbd_sod'),
                ('COME-H', 'cascaded_rgbd_sod'),
                ('DES', 'cheng2014depth'),
                ('DUT-RGBD', 'piao2019depth'),
                ('LFSD', 'li2014saliency'),
                ('NJU2K', 'ju2014depth'),
                ('NLPR', 'peng2014rgbd'),
                ('ReDWeb-S', 'liu2020learning'),
                ('SIP', 'fan2020rethinking'),
                ('STERE', 'niu2012leveraging'),
            ]
            self.mapping_sota_model_names: List[Tuple[str, str]] = [
                ('A2Dele', 'piao2020a2dele'),
                ('ATSA', 'zhang2020asymmetric'),
                ('BBS-Net', 'fan2020bbs'),
                # ('BTSNet', ''),
                ('CDNet', 'jin2021cdnet'),
                
                ('CMINet', 'cascaded_rgbd_sod'),
                ('CoNet', 'ji2020accurate'),
                ('DANet', 'zhao2020single'),
                ('DCF', 'Ji_2021_DCF'),
                ('DSA2F', 'Sun2021DeepRS'),

                # ('EFNet', ''),
                ('FRDT', 'zhang2020feature'),
                ('HAINet', 'li2021hierarchical'),
                ('JLDCF', 'fu2020jl'),
                ('PGAR', 'chen2020progressively'),

                ('RD3D', 'chen2021rgb'),
                ('S2MA', 'liu2020learning'),
                ('SSF', 'zhang2020select'),
                ('UCNet', 'zhang2020uc'),
            ]

            # GoogleDrive
            if run_type == RUN_TYPE.COLAB:
                self.datasets_dir = os.path.join(self.datasets_dir_path, f'DatasetsV{self.datasets_set}')
                self.train_dataset_zip_path = os.path.join(self.datasets_dir, 'train.zip')
                self.test_datasets_dir_path = os.path.join(self.datasets_dir, 'test')
                self.dev_dataset_zip_path: str = None
                self.benchmark_dir_path = os.path.join(self.datasets_dir, 'benchmark')

            # Colab + Kaggle
            self.train_dataset_working_dir_path = os.path.join(self.datasets_working_dir_path, 'train')
            self.test_datasets_working_dir_path = os.path.join(self.datasets_working_dir_path, 'test')
            self.dev_dataset_working_dir_path = os.path.join(self.datasets_working_dir_path, 'test', 'COME-E')
        elif self.datasets_set == 2:
            '''Set 2: NJU2K and NLPR'''
            self.test_dataset_names = [
                'DES', 'LFSD', 'NJU2K', 'NLPR', 'SIP', 'SSD', 'STERE'
            ]
            self.test_dataset_quantities = [
                135, 100, 500, 300, 929, 80, 1000
            ]
            self.sotas_datasets = [
                ["STERE", "DES", "NLPR", "SIP", "SSD", "NJU2K"],
                ["DES", "NJU2K", "NLPR", "SIP", "STERE"],
                ["STERE", "DES", "NLPR", "SIP", "SSD", "NJU2K", "LFSD"],
                ["STERE", "NLPR", "SIP", "SSD", "NJU2K", "LFSD"],
                ["STERE", "NLPR", "NJU2K", "LFSD"],
                ["STERE", "DES", "NLPR", "SIP", "SSD", "NJU2K", "LFSD"],
                ["STERE", "DES", "NLPR", "SIP", "SSD", "LFSD"],
                ["STERE", "DES", "NLPR", "SIP", "SSD", "NJU2K", "LFSD"],
                ["STERE", "DES", "NLPR", "SIP", "NJU2K", "LFSD"],
            ]
            self.sota_model_names = [
                'SPNet',
                'SPSN',
                'C2DFNet',
                'DCF',
                'MVSalNet',
                'BBS-Net',
                'MobileSal_singletrain',
                'TriTransNet',
                'UCNet',
            ]
            self.mapping_test_dataset_names: List[Tuple[str, str]] = [
                ('DES', 'cheng2014depth'),
                ('LFSD', 'li2014saliency'),
                ('NJU2K', 'ju2014depth'),
                ('NLPR', 'peng2014rgbd'),
                ('SIP', 'fan2020rethinking'),
                ('SSD', 'zhu2017three'),
                ('STERE', 'niu2012leveraging'),
            ]
            self.mapping_sota_model_names: List[Tuple[str, str]] = [
                ('SPNet', 'zhou2021specificity'),
                ('SPSN', 'lee2022spsn'),
                ('C2DF-Net', 'zhang2022c'),
                ('DCF', 'Ji_2021_DCF'),
                ('MVSal-Net', 'zhou2022mvsalnet'),
                ('BBS-Net', 'fan2020bbs'),
                ('Mobile-Sal', 'wu2021mobilesal'),
                ('TriTrans-Net', 'liu2021tritransnet'),
                ('UCNet', 'zhang2020uc'),
            ]

            # GoogleDrive
            if run_type == RUN_TYPE.COLAB:
                self.datasets_dir = os.path.join(self.datasets_dir_path, f'DatasetsV{self.datasets_set}')
                self.train_dataset_zip_path = os.path.join(self.datasets_dir, 'train.zip')
                self.test_datasets_dir_path = os.path.join(self.datasets_dir, 'test')
                self.dev_dataset_zip_path: str = os.path.join(self.datasets_dir, 'dev.zip')
                self.benchmark_dir_path = os.path.join(self.datasets_dir, 'benchmark')

            # Colab + Kaggle
            self.train_dataset_working_dir_path = os.path.join(self.datasets_working_dir_path, 'train')
            self.test_datasets_working_dir_path = os.path.join(self.datasets_working_dir_path, 'test')
            self.dev_dataset_working_dir_path = os.path.join(self.datasets_working_dir_path, 'dev')
        elif self.datasets_set == 3:
            '''Set 3: GO BIG! All posible RGBD-SOD datasets combined'''
            self.test_dataset_names = ['dev_test']
            self.sota_model_names = []
            self.sotas_datasets = [self.test_dataset_names for _ in range(len(self.sota_model_names))]
            self.test_dataset_quantities = [203]

            # GoogleDrive
            if run_type == RUN_TYPE.COLAB:
                self.datasets_dir = os.path.join(self.datasets_dir_path, f'DatasetsV{self.datasets_set}')
                self.train_dataset_zip_path = os.path.join(self.datasets_dir, 'train.zip')
                self.test_datasets_dir_path = os.path.join(self.datasets_dir, 'test')
                self.dev_dataset_zip_path: str = None
                self.benchmark_dir_path = os.path.join(self.datasets_dir, 'benchmark')

            # Colab + Kaggle
            self.train_dataset_working_dir_path = os.path.join(self.datasets_working_dir_path, 'train')
            self.test_datasets_working_dir_path = os.path.join(self.datasets_working_dir_path, 'test')
            self.dev_dataset_working_dir_path = os.path.join(self.datasets_working_dir_path, 'test', 'dev_test')
        elif self.datasets_set == 4:
            '''Set 2: NJU2K, NLPR and DUT-RGBD'''
            self.test_dataset_names = [
                'DES', 'LFSD', 'NJU2K', 'NLPR', 'SIP', 'SSD', 'STERE', 'DUT-RGBD'
            ]
            self.test_dataset_quantities = [
                135, 100, 500, 300, 929, 80, 1000, 400
            ]
            self.sotas_datasets = [
                ['LFSD', 'NJU2K', 'NLPR', 'SSD', 'STERE', 'DUT-RGBD'],
                ['DES', 'LFSD', 'NLPR', 'SSD', 'DUT-RGBD'],
                ['DES', 'LFSD', 'NJU2K', 'NLPR', 'SIP', 'STERE', 'DUT-RGBD'],
                ['DES', 'NJU2K', 'NLPR', 'SIP', 'STERE', 'DUT-RGBD'],                
            ]
            self.sota_model_names = [
                'DCMF',
                'DSA2F',
                'JL-DCF',
                'SSLSOD',
            ]
            self.mapping_test_dataset_names = [
                ('DES', 'cheng2014depth'),
                ('LFSD', 'li2014saliency'),
                ('NJU2K', 'ju2014depth'),
                ('NLPR', 'peng2014rgbd'),
                ('SIP', 'fan2020rethinking'),
                ('SSD', 'zhu2017three'),
                ('STERE', 'niu2012leveraging'),
                ('DUT-RGBD', 'piao2019depth'),
            ]
            self.mapping_sota_model_names = [
                ('DCMF', 'wang2022learning'),
                ('DSA2F', 'Sun2021DeepRS'),
                ('JL-DCF', 'fu2020jl'),
                ('SSLSOD', 'zhao2022self'),
            ]

            # GoogleDrive
            if run_type == RUN_TYPE.COLAB:
                self.datasets_dir = os.path.join(self.datasets_dir_path, f'DatasetsV{self.datasets_set}')
                self.train_dataset_zip_path = os.path.join(self.datasets_dir, 'train.zip')
                self.test_datasets_dir_path = os.path.join(self.datasets_dir, 'test')
                self.dev_dataset_zip_path: str = os.path.join(self.datasets_dir, 'dev.zip')
                self.benchmark_dir_path = os.path.join(self.datasets_dir, 'benchmark')

            # Colab + Kaggle
            self.train_dataset_working_dir_path = os.path.join(self.datasets_working_dir_path, 'train')
            self.test_datasets_working_dir_path = os.path.join(self.datasets_working_dir_path, 'test')
            self.dev_dataset_working_dir_path = os.path.join(self.datasets_working_dir_path, 'dev')
        elif self.datasets_set == 5:
            '''Set 1: COME15K - Use ranking_show instead of gt_right
            Train dataset contains 8,025 image pairs of RGB-D
            '''
            self.test_dataset_names = [
                'COME-E', 'COME-H',
                'DES', 'DUT-RGBD', 'LFSD', 'NJU2K', 'NLPR',
                'ReDWeb-S', 'SIP', 'STERE'
            ]
            self.test_dataset_quantities = [
                4600, 3000, 
                135, 400, 100, 500, 300,
                1000, 929, 1000
            ]
            self.sota_model_names = []
            self.sotas_datasets = [self.test_dataset_names for _ in range(len(self.sota_model_names))]
            self.mapping_sota_model_names = []
            self.mapping_test_dataset_names: List[Tuple[str, str]] = [
                ('COME-E', 'cascaded_rgbd_sod'),
                ('COME-H', 'cascaded_rgbd_sod'),
                ('DES', 'cheng2014depth'),
                ('DUT-RGBD', 'piao2019depth'),
                ('LFSD', 'li2014saliency'),
                ('NJU2K', 'ju2014depth'),
                ('NLPR', 'peng2014rgbd'),
                ('ReDWeb-S', 'liu2020learning'),
                ('SIP', 'fan2020rethinking'),
                ('STERE', 'niu2012leveraging'),
            ]

            # GoogleDrive
            if run_type == RUN_TYPE.COLAB:
                self.datasets_dir = os.path.join(self.datasets_dir_path, f'DatasetsV{self.datasets_set}')
                self.train_dataset_zip_path = os.path.join(self.datasets_dir, 'train.zip')
                self.test_datasets_dir_path = os.path.join(self.datasets_dir, 'test')
                self.dev_dataset_zip_path: str = os.path.join(self.datasets_dir, 'dev.zip')
                self.benchmark_dir_path = os.path.join(self.datasets_dir, 'benchmark')

            # Colab + Kaggle
            self.train_dataset_working_dir_path = os.path.join(self.datasets_working_dir_path, 'train')
            self.test_datasets_working_dir_path = os.path.join(self.datasets_working_dir_path, 'test')
            self.dev_dataset_working_dir_path = os.path.join(self.datasets_working_dir_path, 'dev')
        elif self.datasets_set == 6:
            '''Set 1: COME8K + NLPR + NJU2K + DUT-RGBD + ReDWeb-S
            Train dataset contains 13189 image pairs of RGB-D
            '''
            self.test_dataset_names = [
                'COME-E', 'COME-H',
                'DES', 'DUT-RGBD', 'LFSD', 'NJU2K', 'NLPR',
                'ReDWeb-S', 'SIP', 'STERE'
            ]
            self.test_dataset_quantities = [
                4600, 3000, 
                135, 400, 100, 500, 300,
                1000, 929, 1000
            ]
            self.sota_model_names = []
            self.sotas_datasets = [self.test_dataset_names for _ in range(len(self.sota_model_names))]
            self.mapping_sota_model_names = []
            self.mapping_test_dataset_names: List[Tuple[str, str]] = [
                ('COME-E', 'cascaded_rgbd_sod'),
                ('COME-H', 'cascaded_rgbd_sod'),
                ('DES', 'cheng2014depth'),
                ('DUT-RGBD', 'piao2019depth'),
                ('LFSD', 'li2014saliency'),
                ('NJU2K', 'ju2014depth'),
                ('NLPR', 'peng2014rgbd'),
                ('ReDWeb-S', 'liu2020learning'),
                ('SIP', 'fan2020rethinking'),
                ('STERE', 'niu2012leveraging'),
            ]

            # GoogleDrive
            if run_type == RUN_TYPE.COLAB:
                self.datasets_dir = os.path.join(self.datasets_dir_path, f'DatasetsV{self.datasets_set}')
                self.train_dataset_zip_path = os.path.join(self.datasets_dir, 'train.zip')
                self.test_datasets_dir_path = os.path.join(self.datasets_dir, 'test')
                self.dev_dataset_zip_path: str = os.path.join(self.datasets_dir, 'dev.zip')
                self.benchmark_dir_path = os.path.join(self.datasets_dir, 'benchmark')

            # Colab + Kaggle
            self.train_dataset_working_dir_path = os.path.join(self.datasets_working_dir_path, 'train')
            self.test_datasets_working_dir_path = os.path.join(self.datasets_working_dir_path, 'test')
            self.dev_dataset_working_dir_path = os.path.join(self.datasets_working_dir_path, 'dev')
        else:
            raise NotImplementedError()
        
        assert len(self.test_dataset_names) == len(self.test_dataset_quantities) == len(self.mapping_test_dataset_names), \
            'Number of test_dataset_names must equal to the number of test_dataset_quantities'
        assert len(self.sota_model_names) == len(self.sotas_datasets) == len(self.mapping_sota_model_names), \
            'Number of sota_model_names must equal to the number of sotas_datasets'

        '''Upload to Kaggle dataset to continue training'''
        self.continue_training_dir_path = os.path.join(
            self.base_working_dir, 'continue_training'
        )

        '''Evaluation benchmark'''
        self.benchmark_csv_dir_path = os.path.join(self.source_code_dir, 'csv')

        '''Deployment'''
        self.deployment_dir_path = os.path.join(self.source_code_dir, 'deployment')

        '''Results of salient maps from SOTAs and my experiments'''
        self.results_dir_path = os.path.join(self.source_code_dir, 'results')
        
        self.latex_dir_path = os.path.join(self.source_code_dir, 'latex')
        self.qualitative_evaluation_latex_dir_path = os.path.join(
            self.latex_dir_path, f'qualitative_evaluation_set{datasets_set}'
        )
        self.quantitative_evaluation_latex_dir_path = os.path.join(
            self.latex_dir_path, f'quantitative_evaluation_set{datasets_set}'
        )

        # -------------------------------------------------------------------------

        '''Loggers'''
        self.logs_dir = os.path.join(self.source_code_dir, 'logs')

        '''Experiment'''
        self.experiment_dir_path = os.path.join(self.source_code_dir, 'experiment')

        '''Pickle'''
        self.pickle_dir_path = os.path.join(self.source_code_dir, 'pickle') # Deprecated
        
        '''JSON'''
        self.json_dir_path = os.path.join(self.source_code_dir, 'json')

        self.num_train_imgs: int
        self.niters_per_epoch: int
        self.test_image_size: int = 224
        self.image_size: int = 224
        
        '''Train function version'''
        self.train_function_version: int = 1 # choices=[1, 2]

        '''Gradient Accumulation'''
        self.accum_iter = 1
        
        '''Label smoothing'''
        self.label_smoothing = 0.

        '''Wandb tracking'''
        self.wandb_api_key = 'c3fc6b778d58b02a1519dec88b08f0dae1fd5b3b' # thinh.huynh.re@gmail.com

        '''Whether using fp16 instead of fp32 (default)'''
        self.is_fp16: bool = True

        self.is_padding: bool = False # deprecated due to randomly switch between padding and non-padding
        
        '''Whether using padding for test'''
        self.is_padding_for_test: bool = False 
        
        '''Seed'''
        self.seed: int = 2022

        ''' MultiMAE '''
        self.decoder_depth: int = 4
        self.encoder_depth: int = 12
        self.is_inference_with_no_depth: bool = False
        self.inputs = ['rgb', 'depth']
        self.outputs = ['semseg']
        self.decoder_main_tasks: List[List[str]] = [['rgb']]
        self.learnable_pos_emb: bool = False
        self.decoder_interpolate_mode: str = 'bilinear'  # ['bilinear', 'nearest']
        self.dim_tokens: int = 768
        self.act_fn = partial(nn.ReLU, inplace=True)
        self.num_heads: int = 12

        '''Data Augmentation'''
        self.data_augmentation_version: int = 2
        self.data_augmentation_config = DataAugmentationConfig()

        self.ckpt_path: Optional[str] = None
        self.description: str = '' # Override this
        self.embed_dim: int = 6144
        
        '''Pretrained Backbone'''
        self.pretrained_backbone: Optional[str] = 'multi-vit' # choices=['multi-vit', 'mae', 'vit', 'large-mae', 'huge-mae' None]

        '''Learning rate'''
        self.lr: float
        self.end_lr: float = 1e-11
        self.lr_scale: int 
        self.lr_power: float = 0.9

        self.save_checkpoints_after_each_n_epochs: int = 10

        self.weight_decay = 0.05
        self.num_workers = 2
        self.warmup_epoch = 100
        
        self.betas: Tuple[float, float] = (0.9, 0.999)
        
        self.input_patch_size: int = 16
        self.output_patch_size: int = 16

        '''Warmup batchsize'''
        self.warmup_min_batch_size: Optional[int] = None
        self.warmup_epoch_batch_size: Optional[int] = None
        
        self.batch_size: int
        self.val_batch_size: int
        self.test_batch_size: int = 100
        self.nepochs: int

        if run_type in [RUN_TYPE.COLAB, RUN_TYPE.KAGGLE, RUN_TYPE.UBUNTU]:
            self.em = ExperimentManager(
                self.experiment_name,
                self.json_dir_path,
                self.experiment_dir_path,
            )
            # self.em.clean()
            if self.em.latest_epoch is not None:
                self.ckpt_path = os.path.join(
                    self.experiment_dir_path, 
                    self.experiment_name,
                    f'checkpoint_{self.em.latest_epoch}.pt'
                )
            
            self.cpu_device = torch.device('cpu')
        elif run_type == RUN_TYPE.HUGGINGFACE:
            # when using this in production, please specify "epoch"
            self.ckpt_path = get_production_ckpt_path(self.experiment_name, epoch)

def get_train_cfg(
    cfg: base_cfg,
    no_params: Optional[int] = None,
    gflops: Optional[float] = None,
) -> Dict:
    '''Wandb run's configuration'''
    return dict(
        image_size = cfg.image_size,
        test_image_size = cfg.test_image_size,

        accum_iter = cfg.accum_iter,
        is_fp16 = cfg.is_fp16,
        
        lr = cfg.lr,
        end_lr = cfg.end_lr,
        lr_scale = cfg.lr_scale,
        lr_power = cfg.lr_power,

        weight_decay = cfg.weight_decay,
        batch_size = cfg.batch_size,
        val_batch_size = cfg.val_batch_size,
        nepochs = cfg.nepochs,
        num_workers = cfg.num_workers,
        warmup_epoch = cfg.warmup_epoch,
        num_train_imgs = cfg.num_train_imgs if 'num_train_imgs' in cfg.__dict__ else None,
        niters_per_epoch = cfg.niters_per_epoch if 'niters_per_epoch' in cfg.__dict__ else None,
        betas = cfg.betas,
        decoder_depth = cfg.decoder_depth,
        description = cfg.description if 'description' in cfg.__dict__ is not None else None,
        embed_dim = cfg.embed_dim,
        inputs = cfg.inputs,
        outputs = cfg.outputs,
        decoder_main_tasks = cfg.decoder_main_tasks,
        data_augmentation_version = cfg.data_augmentation_version,

        save_checkpoints_after_each_n_epochs = cfg.save_checkpoints_after_each_n_epochs,
        datasets_set = cfg.datasets_set,
        no_params = no_params,
        gflops = gflops,
        
        train_function_version = cfg.train_function_version,
        label_smoothing = cfg.label_smoothing,
        
        warmup_min_batch_size = cfg.warmup_min_batch_size,
        warmup_epoch_batch_size = cfg.warmup_epoch_batch_size,
    )

