from typing import Optional, OrderedDict, Tuple, Union
from torch import nn, Tensor
import torch
import os

from .utils import clean_cache
from .lr import BaseLR
from .configs.base_config import base_cfg
from .logger_fn import Logger

def save_checkpoint(
    cfg: base_cfg,
    epoch: int, 
    global_step: int, 
    model: nn.parallel.DistributedDataParallel, 
    opt: torch.optim.Optimizer, 
    lr_scheduler: BaseLR,
    scaler: torch.cuda.amp.GradScaler = None
) -> None:
    checkpoint = {
        "epoch": epoch,
        "global_step": global_step,
        "state_dict_model": model.state_dict(),
        "state_optimizer": opt.state_dict(),
        "state_lr_scheduler": lr_scheduler.state_dict()
    }
    if scaler is not None:
        checkpoint["scaler"] = scaler.state_dict()
    current_experiment_dir_path = os.path.join(cfg.experiment_dir_path, cfg.experiment_name)
    os.makedirs(current_experiment_dir_path, exist_ok=True)
    checkpoint_path = os.path.join(current_experiment_dir_path, f"checkpoint_{epoch}.pt")
    print(f'Saved checkpoint into {checkpoint_path}')
    torch.save(checkpoint, checkpoint_path)

    cfg.em.update(epoch, None)

def preprocessing_state_dict(
    model: Union[nn.parallel.DistributedDataParallel, nn.Module],
    state_dict_model: OrderedDict[str, Tensor],
) -> OrderedDict[str, Tensor]:
    if isinstance(model, nn.parallel.DistributedDataParallel):
        return OrderedDict(
            (f'module.{k}' if not k.startswith('module.') else k, v) \
                for k, v in state_dict_model.items()
        )
    elif isinstance(model, nn.Module):
        return OrderedDict(
            (k[7:] if k.startswith('module.') else k, v) \
                for k, v in state_dict_model.items()
        )
    else:
        raise Exception(f'Unsupported model type {type(model)}')

def load_checkpoint_for_inference(
    model: nn.Module, ckpt_path: Optional[str] = None,
    strict: Optional[bool] = True
) -> None:
    ''' Throws error if ckpt_path is not found'''
    dict_checkpoint = torch.load(ckpt_path, map_location='cpu')
    model.load_state_dict(dict_checkpoint, strict=strict)
    
def save_checkpoint(
    cfg: base_cfg,
    epoch: int, 
    global_step: int, 
    model: nn.parallel.DistributedDataParallel, 
    opt: torch.optim.Optimizer, 
    lr_scheduler: BaseLR,
    scaler: torch.cuda.amp.GradScaler = None
) -> None:
    checkpoint = {
        "epoch": epoch,
        "global_step": global_step,
        "state_dict_model": model.state_dict(),
        "state_optimizer": opt.state_dict(),
        "state_lr_scheduler": lr_scheduler.state_dict()
    }
    if scaler is not None:
        checkpoint["scaler"] = scaler.state_dict()
    current_experiment_dir_path = os.path.join(cfg.experiment_dir_path, cfg.experiment_name)
    os.makedirs(current_experiment_dir_path, exist_ok=True)
    checkpoint_path = os.path.join(current_experiment_dir_path, f"checkpoint_{epoch}.pt")
    print(f'Saved checkpoint into {checkpoint_path}')
    torch.save(checkpoint, checkpoint_path)

    cfg.em.update(epoch, None)

def load_checkpoint(
    model: Union[nn.parallel.DistributedDataParallel, nn.Module], 
    optimizer: Optional[torch.optim.Optimizer], 
    lr_scheduler: Optional[BaseLR], logger: Optional[Logger], 
    ckpt_path: Optional[str] = None,
    scaler: Optional[torch.cuda.amp.GradScaler] = None,
    strict: Optional[bool] = True) -> Tuple[int, int, bool]:
    """Load checkpoint - Continue training
    - Load the model inplace [important for consistently learning]
    - Load the previous state of optimizer [important for consistently learning]
    - Load previous state of learning rate policy
    - Logging if logger is provided
    
    Note: Throws error if ckpt_path is not found

    Returns:
        - int: start epoch. Default: 1
        - int: global step. Default: 0
    """
    if ckpt_path:
        dict_checkpoint = torch.load(ckpt_path)
        start_epoch = dict_checkpoint["epoch"]
        global_step = dict_checkpoint["global_step"]
        
        model.load_state_dict(
            preprocessing_state_dict(model, dict_checkpoint["state_dict_model"]),
            strict=strict
        )
        
        if optimizer is not None:
            optimizer.load_state_dict(dict_checkpoint["state_optimizer"])

        if lr_scheduler is not None:
            lr_scheduler.load_state_dict(dict_checkpoint["state_lr_scheduler"])

        if scaler is not None:
            scaler.load_state_dict(dict_checkpoint["scaler"])
        
        del dict_checkpoint
        clean_cache()

        if logger is not None:
            logger.info(f'Load checkpoint {ckpt_path}')
        
        return start_epoch+1, global_step
    else:
        return 1, 0